/* A few functions for dealing with time more easily.  This is fairly
   nasty, and assumes a little endian machine. */
#include <sys/types.h>
#include <assert.h>
#include <printf.h>
#include <stdint.h>

#include "demux.h"

/* Timestamps are measured in microseconds since this point. */
static struct timeval
timestamps_base;
static unsigned
have_ts_base;

static unsigned long long
timestamp_to_ll(timestamp_t *ts)
{
	unsigned long *x;
	x = (unsigned long *)ts->data;
	return *x | ((long long)*(unsigned short *)(x + 1) << 32);
}

static long long
timediff_to_ll(timediff_t *td)
{
	unsigned long *x;
	x = (unsigned long *)td->data;
	return *x | ((long long)*(short *)(x + 1) << 32);
}

static timestamp_t
ll_to_timestamp(unsigned long long x)
{
	union {
		timestamp_t work;
		unsigned long long y;
	} u;
	u.y = x;
	return u.work;
}

static timediff_t
ll_to_timediff(long long x)
{
	union {
		timediff_t work;
		long long y;
	} u;
	u.y = x;
	assert(timediff_to_ll(&u.work) == x);
	return u.work;
}

timediff_t
timestamp_sub(timestamp_t a, timestamp_t b)
{
	return ll_to_timediff(timestamp_to_ll(&a) -
			      timestamp_to_ll(&b));
}

int
timediff_lt(timediff_t a, timediff_t b)
{
	if (timediff_to_ll(&a) < timediff_to_ll(&b))
		return 1;
	else
		return 0;
}

struct timeval
timestamp_to_timeval(timestamp_t a)
{
	struct timeval work;
	unsigned long long ts;
	assert(have_ts_base);
	ts = timestamp_to_ll(&a);
	work = timestamps_base;
	work.tv_usec += ts % 1000000;
	work.tv_sec += ts / 1000000;
	while (work.tv_usec >= 1000000) {
		work.tv_usec -= 1000000;
		work.tv_sec++;
	}
	return work;
}

timestamp_t
timeval_to_timestamp(struct timeval tv)
{
	assert(have_ts_base);
	assert(tv.tv_sec > timestamps_base.tv_sec ||
	       (tv.tv_sec == timestamps_base.tv_sec &&
		tv.tv_usec >= timestamps_base.tv_usec));
	tv.tv_sec -= timestamps_base.tv_sec;
	tv.tv_usec -= timestamps_base.tv_usec;
	while (tv.tv_usec < 0) {
		tv.tv_usec += 1000000;
		tv.tv_sec--;
	}
	return ll_to_timestamp(tv.tv_sec * 1000000ull + tv.tv_usec);
}

double
timediff_to_double(timediff_t td)
{
	long long l = timediff_to_ll(&td);
	return l * 1e-6;
}

timediff_t
double_to_timediff(double d)
{
	return ll_to_timediff(d * 1e6 + 0.5);
}

static int
printf_special_w_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	timediff_t *td;
	long long t;
	td = *(timediff_t **)args[0];
	t = timediff_to_ll(td);
	return fprintf(f, "%d.%.06d", (unsigned)(t / 1000000),
		       (unsigned)t % 1000000);
}

static int
printf_special_W_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	timestamp_t *ts;
	struct timeval tv;
	ts = *(timestamp_t **)args[0];
	tv = timestamp_to_timeval(*ts);
	return fprintf(f, "%ld.%.06ld", tv.tv_sec, tv.tv_usec);
}

static int
printf_special_wW_arginfo_func(const struct printf_info *info,
			     size_t n,
			     int *argtypes)
{
	if (n > 0)
		argtypes[0] = PA_POINTER;
	return 1;
}

void
setup_time(const struct timeval *base)
{
	have_ts_base = 1;
	timestamps_base = *base;
	register_printf_function('w', printf_special_w_handler_func,
				 printf_special_wW_arginfo_func);
	register_printf_function('W', printf_special_W_handler_func,
				 printf_special_wW_arginfo_func);
}
