#include "config.h"

#include <sys/queue.h>
#include <stdio.h>

struct pcap_pkthdr;

/* Some simple constructs to make network<->host byte order
   conversions a bit easier and safer. */
typedef struct {
	unsigned long a;
} n32;
typedef struct {
	unsigned short a;
} n16;
#define net_to_host(x) ((__typeof__((x).a))__net_to_host(&(x), sizeof((x).a)))
unsigned long __net_to_host(const void *x, unsigned s);

/* Sequence number wrapper functions */
typedef struct {
	unsigned long s;
} seqnr_t;
int seq_eq(seqnr_t s1, seqnr_t s2);
seqnr_t seq_plus(seqnr_t s, unsigned long x);
seqnr_t seq_sub(seqnr_t s, unsigned long x);
int seq_gt(seqnr_t s1, seqnr_t s2);
int seq_le(seqnr_t s1, seqnr_t s2);

/* Timestamps.  These are opaque to anyone excpet their handling
   functions.  They should be good to one microsecond over traces up
   to a few years in length. */
typedef struct { /* Use %W in printf format strings */
	unsigned char data[6];
} timestamp_t;
/* Use the typesystem to record whether times are supposed to be
   relative or absolute. */
typedef struct { /* Use %w in printf format strings */
	unsigned char data[6];
} timediff_t;

timediff_t timestamp_sub(timestamp_t, timestamp_t);
int timediff_lt(timediff_t, timediff_t);

struct timeval timestamp_to_timeval(timestamp_t);
timestamp_t timeval_to_timestamp(struct timeval);

/* Convert a timediff to a double.  Careful with rounding etc. */
double timediff_to_double(timediff_t);
timediff_t double_to_timediff(double);

#define TIMEDIFF_MAX ((timediff_t){{0xff,0xff,0xff,0xff,0xff,0x7f}})

#define unaligned __attribute__((aligned (1)))
#define packed __attribute__((packed))

/* Stream identifier stuff */
struct streamid {
	n32 saddr;
	n32 daddr;
	n16 sport;
	n16 dport;
	unsigned char protocol;
} packed unaligned;
int cmp_streamid(const struct streamid *sid1, const struct streamid *sid2);
unsigned long hash_streamid(const struct streamid *sid);

void *xcalloc(unsigned s, unsigned n);
void setup_util(void);
void setup_time(const struct timeval *base);
FILE *open_tee(FILE *f1, FILE *f2);


struct tcpflow {
	seqnr_t fin_seq;
	unsigned char has_sent_fin:1,
		      acked_fin:1,
		      c2s:1,
		      has_packets:1;
} packed unaligned;

/* This can actually represent udp or icmp flows as well, but those
   are kind of a degenerate special cases. */
/* Note that we can easily have tens of millions of these live, and so
   saving a few bytes is very much worthwhile.  There are currently
   six bits of padding here, in the flow structures. */
/* Free instances of this structure are threaded through the
 * lru_chain. */
struct tcpconn {
	struct streamid sid;
	struct tcpflow flow[2];
	int comatose:1;
	int aborted:1;
	int has_file:1;
	int buffer_id:21;
	LIST_ENTRY(tcpconn) hash_chain;
	TAILQ_ENTRY(tcpconn) lru_chain;
	unsigned nr_packets;
	timestamp_t start, last_datagram;
} packed unaligned;

/* Packets are threaded on two linked lists, one by time and one by
   offset in the datagram.  Terminology: a packet is a thing we pick
   off of the wire, a datagram is a complete IP datagram (post
   de-fragmentation). */
struct datagram {
	struct datagram *next, **pdg;
	struct streamid sid;
	union {
		const void *ip_payload;
		const struct tcphdr *tcp;
		const struct udphdr *udp;
		const struct icmphdr *icmp;
	};
	unsigned flownr;
	unsigned tcpdata;
	struct packet *first_packet, *last_packet; /* time */
	struct packet *head_packet, *tail_packet; /* offset */
};

struct packet {
	struct packet *next_time, *prev_time, *next_off, *prev_off;
	const struct ethhdr *eth;
	const struct iphdr *ip;
	timestamp_t ts;
	unsigned len;
	const void *payload;
};


void dump_flow_datagram(struct tcpconn *tcp, struct datagram *dg);
void dump_unhandled_packet(struct packet *p);
void dump_unhandled_datagram(struct datagram *dg);
void release_buffer(struct tcpconn *tcp);
void setup_buffers(void);
void teardown_buffers(void);
void process_file(const char *fname, const char *filter,
		  void (*cb)(const struct pcap_pkthdr *hdr,
			     const unsigned char *data));

void collect_fds(void);

void dump_stats(timestamp_t ts);
void sanity_check_dumpfiles(void);

void histo_sample(double val);

/* Various interesting statistics go in here.  There's only ever one
   instace of this structure, so they might as well be global, but
   this helps to keep the namespace a bit cleaner */
struct statblock {
	unsigned long long total_datagrams, total_packets, tcp_datagrams;
	timestamp_t start_time;

	struct timeval rt_start_time;

	/* FD cache stats */
	unsigned fds_used;
	unsigned long long fd_requests;
	unsigned long long fd_hits;
	unsigned long long gced_fds;

	/* Buffer cache stats */
	unsigned long long buffer_requests;
	unsigned long long buffer_hits;
	unsigned long long buffer_full_flushes;
	unsigned long long gced_buffers;

	/* Connection pool stats */
	unsigned live_connections;
	unsigned gced_conns;
	unsigned gced_non_comatose_conns;
	unsigned comatose_connections;
	unsigned aborted_connections;
	unsigned nr_tcp_connections;

	/* Hash table statistics */
	unsigned long long conn_lookups;
	unsigned long long hash_probes;

	/* Reassembly engine stats */
	unsigned datagrams_on_comatose_conns;

	/* Other stats */
	unsigned incomplete_datagrams;
	timediff_t critical_idle_thresh;
};


extern struct statblock stats;
extern unsigned link_type;
extern FILE *logfile;
extern int debug;
extern char *output_prefix;
