/* All of the stuff which is just managing dump files. */
#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <assert.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <pcap.h>

#include "demux.h"

#ifndef PAGE_SIZE
#define PAGE_SIZE 4096
#endif

#define MAP_SIZE (PAGE_SIZE * 1024)

void
process_file(const char *fname, const char *filter,
	     void (*cb)(const struct pcap_pkthdr *hdr,
			const unsigned char *data))
{
	struct bpf_program bpf;
	int fd = -1;
	void *map = MAP_FAILED;
	unsigned long long map_avail;
	unsigned long long map_offset;
	unsigned long long len, processed;
	struct pcap_file_header *pfh;
	unsigned long long mmap_size;
	int have_filter = 0;
	struct pcap_pkthdr *pkt_hdr;

	fd = open(fname, O_RDONLY);
	if (fd < 0) {
		fprintf(logfile, "cannot open %s (%s)\n", fname,
			strerror(errno));
		return;
	}
	len = lseek(fd, 0, SEEK_END);
	if (len < sizeof(struct pcap_file_header)) {
		fprintf(logfile, "%s is too small", fname);
		close(fd);
		return;
	}

	map_avail = len;
	if (map_avail > MAP_SIZE)
		map_avail = MAP_SIZE;
	mmap_size = (map_avail + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
	map = mmap(NULL, mmap_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(logfile, "cannot map %s (%s)\n", fname,
			strerror(errno));
		close(fd);
		return;
	}
	madvise(map, mmap_size, MADV_SEQUENTIAL);

	pfh = map;
	if (pfh->magic != 0xa1b2c3d4) {
		fprintf(logfile, "%s is not a pcap dump file\n", fname);
		goto out;
	}
	if (pfh->version_major != 2 || pfh->version_minor != 4) {
		fprintf(logfile, "Only pcap version 2.4 is supported; %s is version %d.%d.\n",
			fname, pfh->version_major, pfh->version_minor);
	}

	if (filter) {
		if (pcap_compile_nopcap(pfh->snaplen,
					pfh->linktype,
					&bpf,
					(char *)filter,
					1, 0) < 0)
			errx(1, "cannot compile %s", filter);
		have_filter = 1;
	}
	link_type = pfh->linktype;

	map_offset = 0;
	processed = sizeof(*pfh);
	while (processed < len) {
		pkt_hdr = map + (processed - map_offset);
		if (map_avail > PAGE_SIZE &&
		    processed - map_offset >= mmap_size - PAGE_SIZE) {

			munmap(map, mmap_size);

			map_offset += mmap_size - PAGE_SIZE;

			assert(processed >= map_offset);

			map_avail = len - processed;
			if (map_avail > MAP_SIZE)
				map_avail = MAP_SIZE;

			mmap_size=(map_avail+PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
			map = mmap(NULL, mmap_size, PROT_READ, MAP_PRIVATE, fd,
				   map_offset);
			if (map == MAP_FAILED) {
				fprintf(logfile, "cannot map %s (%s)\n", fname,
					strerror(errno));
				exit(1);
			}
			madvise(map, mmap_size, MADV_SEQUENTIAL);
			continue;
		}
		cb(pkt_hdr, (void *)(pkt_hdr + 1));
		processed += sizeof(*pkt_hdr) + pkt_hdr->caplen;
	}

 out:
	if (have_filter)
		pcap_freecode(&bpf);
	if (fd >= 0)
		close(fd);
	if (map != MAP_FAILED)
		munmap(map, mmap_size);
}

