#define _FILE_OFFSET_BITS 64
#define _GNU_SOURCE
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <sys/queue.h>
#include <string.h>
#include <unistd.h>
#include <err.h>
#include <pcap.h>
#include <assert.h>
#include <errno.h>
#include <printf.h>
#include <math.h>

#include "demux.h"

#define ETH_P_IP 0x0800

static FILE *
logfile;
static FILE *
sense_file;
static FILE *
file_log;

static unsigned long long
total_datagrams;
static unsigned long long
total_packets;
static unsigned long long
tcp_datagrams;
static struct timeval
start_time;
static struct timeval
rt_start_time;
static struct timeval
last_packet_time;

static unsigned
gced_buffers;
static unsigned
gced_fds;
static unsigned
gced_conns;
static unsigned
gced_non_comatose_conns;
static unsigned
datagrams_on_comatose_conns;
static unsigned
comatose_connections;
static unsigned
aborted_connections;

static unsigned
incomplete_datagrams;

static double
critical_idle_thresh = INFINITY;

static unsigned long long
conn_lookups;
static unsigned long long
hash_probes;
static unsigned long long
fd_requests;
static unsigned long long
fd_hits;
static unsigned long long
buffer_requests;
static unsigned long long
buffer_hits;
static unsigned long long
buffer_full_flushes;

static unsigned
nr_tcp_connections;

static char
pcap_errbuf[PCAP_ERRBUF_SIZE];

struct ethhdr {
	unsigned char h_dest[6];
	unsigned char h_src[6];
	n16 h_proto;
};

struct udphdr {
	n16   sport;
	n16   dport;
	n16   len;
	n16   check;
};

struct tcphdr {
        n16   sport;
        n16   dport;
        seqnr_t   seq;
        seqnr_t   ack_seq;
        unsigned   res1:4,
                doff:4,
                fin:1,
                syn:1,
                rst:1,
                psh:1,
                ack:1,
                urg:1,
                ece:1,
                cwr:1;
        n16   window;
        n16   check;
        n16   urg_ptr;
};

static LIST_HEAD(, tcpconn)
hash_table[HASH_SIZE];
static TAILQ_HEAD(, tcpconn)
lru_head;
static TAILQ_HEAD(, tcpconn)
fd_lru;
static TAILQ_HEAD(, tcpconn)
buffer_lru;

static unsigned max_buffer_mb = 32;

#define BUFFER_ROUND_SIZE 16
#define FD_ROUND_SIZE 16
#define LIVE_CONN_ROUND_SIZE 64
#define BUF_USE_THRESH ((max_buffer_mb * 1024 * 1024) / PER_CONN_BUFFER)
#define FD_USE_THRESH 780
#define LIVE_CONN_THRESH 10324441

static unsigned
buffers_used;
static unsigned
fds_used;
static unsigned
live_connections;

#define TAILQ_LAST(head, type, field)                               \
((head)->tqh_last == &(head)->tqh_first ? NULL :                    \
 ((type *)((unsigned long)(head)->tqh_last -                        \
	   (unsigned long)&((type *)0)->field.tqe_next)))

#define TAILQ_PREV(head, type, field, elm)                          \
((elm)->field.tqe_prev == &(head)->tqh_first ? NULL :               \
 ((type *)((unsigned long)(elm)->field.tqe_prev -                   \
	   (unsigned long)&((type *)0)->field.tqe_next)))

#define PER_CONN_BUFFER 1024

static unsigned
link_type;

static void
parse_packet(struct packet *p)
{
	switch (link_type) {
	case DLT_EN10MB:
		p->eth = p->payload;
		if (net_to_host(p->eth->h_proto) == ETH_P_IP) {
			p->ip = (struct iphdr *)(p->eth + 1);
		}
		break;
	case DLT_RAW:
		p->eth = NULL;
		p->ip = p->payload;
		break;
	default:
		errx(1, "unknown BPF linktype %d", link_type);
	}
}

static void
parse_datagram(struct datagram *dg)
{
	struct packet *p = dg->head_packet;
	if (p->ip && p->ip->version == 4) {
		dg->sid.protocol = p->ip->protocol;
		dg->ip_payload = (void *)p->ip + p->ip->hlen * 4;
		switch (p->ip->protocol) {
		case IPPROTO_TCP:
			if (net_to_host(dg->tcp->sport) <
			    net_to_host(dg->tcp->dport) ||
			    (dg->tcp->sport.a == dg->tcp->dport.a &&
			     p->ip->saddr.a < p->ip->daddr.a)) {
				dg->sid.saddr = p->ip->saddr;
				dg->sid.daddr = p->ip->daddr;
				dg->sid.sport = dg->tcp->sport;
				dg->sid.dport = dg->tcp->dport;
				dg->flownr = 0;
			} else {
				dg->sid.daddr = p->ip->saddr;
				dg->sid.saddr = p->ip->daddr;
				dg->sid.dport = dg->tcp->sport;
				dg->sid.sport = dg->tcp->dport;
				dg->flownr = 1;
			}
			dg->tcpdata = net_to_host(p->ip->tot_len) -
				p->ip->hlen * 4 -
				dg->tcp->doff * 4;
			break;
		case IPPROTO_UDP:
			if (net_to_host(dg->udp->sport) <
			    net_to_host(dg->udp->dport) ||
			    (dg->udp->sport.a == dg->udp->dport.a &&
			     p->ip->saddr.a < p->ip->daddr.a)) {
				dg->sid.saddr = p->ip->saddr;
				dg->sid.daddr = p->ip->daddr;
				dg->sid.sport = dg->udp->sport;
				dg->sid.dport = dg->udp->dport;
				dg->flownr = 0;
			} else {
				dg->sid.daddr = p->ip->saddr;
				dg->sid.saddr = p->ip->daddr;
				dg->sid.dport = dg->udp->sport;
				dg->sid.sport = dg->udp->dport;
				dg->flownr = 1;
			}
			break;
		default:
			dg->sid.sport.a = dg->sid.dport.a = 0;
			if (p->ip->saddr.a < p->ip->daddr.a) {
				dg->sid.saddr = p->ip->saddr;
				dg->sid.daddr = p->ip->daddr;
				dg->flownr = 0;
			} else {
				dg->sid.daddr = p->ip->saddr;
				dg->sid.saddr = p->ip->daddr;
				dg->flownr = 1;
			}
			break;
		}
	}
}

/* ------------------------------------------------------------------------- */
/* A bit of infrastructure for managing dump files */
static void
replenish_fd(struct tcpconn *tcp)
{
	static char fname[4096];
	int x;
	struct pcap_file_header pfh;

	TAILQ_INSERT_HEAD(&fd_lru, tcp, fd_lru);
	x = sprintf(fname, "out/%P", tcp);
	fds_used++;
	tcp->fd = open(fname, O_APPEND | O_WRONLY);
	if (tcp->fd >= 0) {
		fprintf(file_log, "open1 %s -> %d\n", fname, tcp->fd);
		return;
	}
	fname[x] = '!';
	while (x > 0) {
		while (x > 0 && fname[x] != '/')
			x--;
		if (x == 0)
			break;
		fname[x] = 0;
#if 0
		/* This turns out to not actually be an
		 * optimisation */
		if (mkdir(fname, 0777) >= 0 || errno == EEXIST)
			break;
#endif
	}
	while (fname[x] != '!' && fname[x])
		x++;
	if (fname[x] != '!')
		fname[x] = '/';
	while (fname[x] != '!' && fname[x])
		x++;
	while (fname[x] != '!') {
		if (mkdir(fname, 0777) < 0 && errno != EEXIST)
			err(1, "making directory %s", fname);
		fname[x] = '/';
		while (fname[x] != '!' && fname[x])
			x++;
	}
	fname[x] = 0;
	tcp->fd = open(fname, O_CREAT | O_WRONLY | O_EXCL, 0666);
	if (tcp->fd < 0)
		err(1, "openning %s", fname);
	fprintf(file_log, "open2 %s -> %d\n", fname, tcp->fd);
	pfh.magic = 0xa1b2c3d4;
	pfh.version_major = 2;
	pfh.version_minor = 4;
	pfh.thiszone = 0;
	pfh.sigfigs = 6;
	pfh.snaplen = 65535;
	pfh.linktype = link_type;
	write(tcp->fd, &pfh, sizeof(pfh));

	if (tcp->sid.protocol == IPPROTO_TCP) {
		if (tcp->flow[0].c2s)
			fprintf(sense_file, "%P client/server\n",
				tcp);
		else if (tcp->flow[1].c2s)
			fprintf(sense_file, "%P server/client\n",
				tcp);
		else
			fprintf(sense_file, "%P unknown\n",
				tcp);
	}
}

static void
flush_buffer(struct tcpconn *tcp)
{
	int r;

	if (tcp->buffer_used == 0)
		return;
	fd_requests++;
	if (tcp->fd < 0)
		replenish_fd(tcp);
	else
		fd_hits++;
	r = write(tcp->fd, tcp->buffer, tcp->buffer_used);
	if (r != tcp->buffer_used)
		abort();
	fprintf(file_log, "write %d -> %d\n", tcp->buffer_used, tcp->fd);
	tcp->buffer_used = 0;
}

static void
replenish_buffer(struct tcpconn *tcp)
{
	if (tcp->buffer_used) {
		if (tcp->fd >= 0) {
			TAILQ_REMOVE(&fd_lru, tcp, fd_lru);
			TAILQ_INSERT_HEAD(&fd_lru, tcp, fd_lru);
		}
		flush_buffer(tcp);
	} else {
		tcp->buffer = malloc(PER_CONN_BUFFER);
		tcp->buffer_used = 0;
		TAILQ_INSERT_HEAD(&buffer_lru, tcp, buffer_lru);
		buffers_used++;
	}
}

static void
write_flow_bytes(struct tcpconn *tcp, const void *buf, unsigned size)
{
	unsigned long written_so_far;
	unsigned long write_this_pass;
	written_so_far = 0;
	buffer_requests++;
	if (!tcp->buffer)
		replenish_buffer(tcp);
	else
		buffer_hits++;
	while (written_so_far < size) {
		if (tcp->buffer_used == PER_CONN_BUFFER) {
			buffer_full_flushes++;
			replenish_buffer(tcp);
		}
		write_this_pass = size - written_so_far;
		if (write_this_pass > PER_CONN_BUFFER - tcp->buffer_used)
			write_this_pass = PER_CONN_BUFFER - tcp->buffer_used;
		memcpy(tcp->buffer + tcp->buffer_used,
		       buf + written_so_far,
		       write_this_pass);
		written_so_far += write_this_pass;
		tcp->buffer_used += write_this_pass;
	}
	TAILQ_REMOVE(&buffer_lru, tcp, buffer_lru);
	TAILQ_INSERT_HEAD(&buffer_lru, tcp, buffer_lru);
}

static void
dump_flow_packet(struct tcpconn *tcp, struct packet *p)
{
	struct pcap_pkthdr ph;
	ph.ts = p->ts;
	ph.caplen = p->len;
	ph.len = p->len;
	write_flow_bytes(tcp, &ph, sizeof(ph));
	write_flow_bytes(tcp, p->payload, p->len);
}

static void
dump_flow_datagram(struct tcpconn *tcp, struct datagram *dg)
{
	struct packet *p;
	for (p = dg->head_packet; p; p = p->next_time)
		dump_flow_packet(tcp, p);
}

static void
dump_unhandled_packet(struct packet *p)
{
	static FILE *f;
	struct pcap_pkthdr ph;
	struct pcap_file_header pfh;
	if (!f) {
		f = fopen("out/ign_packets", "wb");
		if (!f)
			err(1, "openning out/ign_packets");
		pfh.magic = 0xa1b2c3d4;
		pfh.version_major = 2;
		pfh.version_minor = 4;
		pfh.thiszone = 0;
		pfh.sigfigs = 6;
		pfh.snaplen = 65535;
		pfh.linktype = link_type;
		fwrite(&pfh, sizeof(pfh), 1, f);

	}
	ph.ts = p->ts;
	ph.caplen = p->len;
	ph.len = p->len;
	fwrite(&ph, sizeof(ph), 1, f);
	fwrite(p->payload, p->len, 1, f);
}

static void
dump_unhandled_datagram(struct datagram *dg)
{
	struct packet *p;
	for (p = dg->head_packet; p; p = p->next_time)
		dump_unhandled_packet(p);
}


/* ------------------------------------------------------------------------- */
/* Cache management stuff */

static void
close_connection_fd(struct tcpconn *tcp)
{
	assert(tcp->fd >= 0);
	fprintf(file_log, "close %d\n", tcp->fd);
	close(tcp->fd);
	tcp->fd = -1;
	fds_used--;
	gced_fds++;
	TAILQ_REMOVE(&fd_lru, tcp, fd_lru);
}

static void
collect_buffers(void)
{
	struct tcpconn *tcp, *n;
	int counter = 0;
	tcp = TAILQ_LAST(&buffer_lru, struct tcpconn, buffer_lru);
	while (tcp && counter < BUFFER_ROUND_SIZE) {
		assert(tcp->buffer);
		if (tcp->buffer_used)
			flush_buffer(tcp);
		free(tcp->buffer);
		buffers_used--;
		gced_buffers++;
		tcp->buffer = NULL;
		n = TAILQ_PREV(&buffer_lru, struct tcpconn, buffer_lru, tcp);
		TAILQ_REMOVE(&buffer_lru, tcp, buffer_lru);
		tcp = n;
		counter++;
	}
}

static void
collect_fds(void)
{
	struct tcpconn *tcp, *n;
	int counter = 0;
	tcp = TAILQ_LAST(&fd_lru, struct tcpconn, fd_lru);
	while (tcp && counter < FD_ROUND_SIZE) {
		assert(tcp->fd >= 0);
		if (tcp->buffer) {
			flush_buffer(tcp);
			free(tcp->buffer);
			tcp->buffer = NULL;
			buffers_used--;
			gced_buffers++;
			TAILQ_REMOVE(&buffer_lru, tcp, buffer_lru);
		}
		n = TAILQ_PREV(&fd_lru, struct tcpconn, fd_lru, tcp);
		close_connection_fd(tcp);
		tcp = n;
		counter++;
	}
}

static void
collect_live_conns(void)
{
	struct tcpconn *tcp, *n;
	int counter = 0;
	tcp = TAILQ_LAST(&lru_head, struct tcpconn, lru_chain);
	while (tcp && counter < LIVE_CONN_ROUND_SIZE) {
		gced_conns++;
		if (tcp->buffer) {
			flush_buffer(tcp);
			free(tcp->buffer);
			tcp->buffer = NULL;
			buffers_used--;
			gced_buffers++;
			TAILQ_REMOVE(&buffer_lru, tcp, buffer_lru);
		}
		if (tcp->fd >= 0) {
			close_connection_fd(tcp);
		}
		if (tcp->sid.protocol == IPPROTO_TCP) {
			nr_tcp_connections--;
			if (!tcp->comatose) {
				if (!tcp->aborted &&
				    tcp->flow[0].nr_datagrams &&
				    tcp->flow[1].nr_datagrams) {
					struct timeval idle_time;
					idle_time.tv_sec =
						last_packet_time.tv_sec -
						tcp->last_datagram.tv_sec;
					idle_time.tv_usec =
						last_packet_time.tv_usec -
						tcp->last_datagram.tv_usec;
					if (idle_time.tv_usec < 0) {
						idle_time.tv_usec += 1000000;
						idle_time.tv_sec--;
					}
					fprintf(logfile,
						"WARNING: garbage collecting non-comatose connection %P (%d,%d datagrams, idle %ld.%.06ld)\n",
						tcp, tcp->flow[0].nr_datagrams,
						tcp->flow[1].nr_datagrams,
						idle_time.tv_sec, idle_time.tv_usec);
					if (idle_time.tv_sec +
					    idle_time.tv_usec * 1e-6 <
					    critical_idle_thresh)
						critical_idle_thresh =
							idle_time.tv_sec +
							idle_time.tv_usec * 1e-6;
					gced_non_comatose_conns++;
				}
			} else {
				comatose_connections--;
			}
			if (tcp->aborted)
				aborted_connections--;
		}
		n = TAILQ_PREV(&lru_head, struct tcpconn, lru_chain, tcp);
		LIST_REMOVE(tcp, hash_chain);
		TAILQ_REMOVE(&lru_head, tcp, lru_chain);
		free(tcp);
		live_connections--;
		tcp = n;
		counter++;
	}
}

static void
do_cleanup(void)
{
	while (live_connections > LIVE_CONN_THRESH) {
		while (fds_used > FD_USE_THRESH)
			collect_fds();
		collect_live_conns();
	}
	while (buffers_used > BUF_USE_THRESH) {
		while (fds_used > FD_USE_THRESH)
			collect_fds();
		collect_buffers();
	}
	while (fds_used > FD_USE_THRESH)
		collect_fds();
}

/* ------------------------------------------------------------------------- */
/* This is the reassembly engine proper */

static struct tcpconn *
find_flow(struct datagram *p)
{
	unsigned hash;
	struct tcpconn *tcp;

	conn_lookups++;
	hash = hash_streamid(&p->sid);
	tcp = hash_table[hash].lh_first;
	while (tcp) {
		hash_probes++;
		if (!cmp_streamid(&p->sid, &tcp->sid)) {
			if (tcp->comatose) {
				/* Comatose connection.  The datagram
				   we're just picking up could be
				   either a new connection, or a
				   retransmit due to last ack getting
				   dropped.  Decide based on sequence
				   number. */
				/* (i.e. we've seen fins in both
				   directions on this connection) */
				if (p->tcp->fin &&
				    seq_eq(seq_plus(p->tcp->seq, p->tcpdata),
					   tcp->flow[p->flownr].fin_seq)) {
					/* Retransmitted fin */
					datagrams_on_comatose_conns++;
					break;
				}
				if (p->tcp->ack &&
				    seq_eq(p->tcp->ack_seq,
					   seq_plus(tcp->flow[1-p->flownr].fin_seq,
						    1))) {
					/* Retransmitted last ack */
					datagrams_on_comatose_conns++;
					break;
				}
				if (p->tcp->rst) {
					/* RST -> probably belongs to
					 * this connection. */
					datagrams_on_comatose_conns++;
					break;
				}
			} else if (tcp->aborted) {
				/* We've seen an RST on this
				   connection.  Start a new connection
				   if this frame is a SYN, otherwise
				   continue to add stuff to this
				   one. */
				assert(p->sid.protocol == IPPROTO_TCP);
				if (!p->tcp->syn)
					break;
			} else {
				/* Connection is neither comatose nor
				   aborted -> continue to add datagrams
				   to it. */
				break;
			}
		}
		tcp = tcp->hash_chain.le_next;
	}
	if (tcp) {
		LIST_REMOVE(tcp, hash_chain);
		LIST_INSERT_HEAD(&hash_table[hash], tcp, hash_chain);
		TAILQ_REMOVE(&lru_head, tcp, lru_chain);
		TAILQ_INSERT_HEAD(&lru_head, tcp, lru_chain);
		return tcp;
	}
	live_connections++;
	tcp = xcalloc(sizeof(*tcp), 1);
	memcpy(&tcp->sid, &p->sid, sizeof(p->sid));
	tcp->fd = -1;
	tcp->start = p->last_packet->ts;
	LIST_INSERT_HEAD(&hash_table[hash], tcp, hash_chain);
	TAILQ_INSERT_HEAD(&lru_head, tcp, lru_chain);
	if (tcp->sid.protocol == IPPROTO_TCP)
		nr_tcp_connections++;
	return tcp;
}

static void
handle_syn(struct tcpconn *tcp, struct datagram *p)
{
	if (!p->tcp->ack)
		tcp->flow[p->flownr].c2s = 1;
	tcp->flow[p->flownr].syn_seq = p->tcp->seq;
	tcp->flow[p->flownr].has_sent_syn = 1;
}

static void
handle_data(struct tcpconn *tcp, struct datagram *p)
{
}

static void
handle_fin(struct tcpconn *tcp, struct datagram *p)
{
	tcp->flow[p->flownr].fin_seq = seq_plus(p->tcp->seq, p->tcpdata + 1);
	tcp->flow[p->flownr].has_sent_fin = 1;
}

static void
handle_ack(struct tcpconn *tcp, struct datagram *p)
{
	if (tcp->flow[1-p->flownr].has_sent_fin &&
	    seq_eq(p->tcp->ack_seq, tcp->flow[1-p->flownr].fin_seq)) {
		tcp->flow[p->flownr].acked_fin = 1;
		if (tcp->flow[1-p->flownr].acked_fin) {
			if (!tcp->comatose)
				comatose_connections++;
			tcp->comatose = 1;
		}
	}
}

static void
handle_rst(struct tcpconn *tcp, struct datagram *p)
{
	if (!tcp->aborted)
		aborted_connections++;
	tcp->aborted = 1;
}

static void
handle_datagram(struct tcpconn *tcp, struct datagram *p)
{
	if (p->sid.protocol == IPPROTO_TCP) {
		/* If it isn't TCP, it doesn't go through the state
		   machine. */
		if (p->tcp->syn)
			handle_syn(tcp, p);
		if (p->tcpdata)
			handle_data(tcp, p);
		if (p->tcp->fin)
			handle_fin(tcp, p);
		if (p->tcp->ack)
			handle_ack(tcp, p);
		if (p->tcp->rst)
			handle_rst(tcp, p);
		tcp->flow[p->flownr].nr_datagrams++;
	}
	tcp->last_datagram = p->last_packet->ts;
	dump_flow_datagram(tcp, p);
}

static void
process_datagram(struct datagram *dg)
{
	struct tcpconn *f;
	total_datagrams++;
	if (!dg->head_packet->ip) {
		dump_unhandled_datagram(dg);
		return;
	}
	f = find_flow(dg);
	if (!f) {
		dump_unhandled_datagram(dg);
	} else {
		handle_datagram(f, dg);
	}
}

/* ------------------------------------------------------------------------- */
/* Fragment handling.  This is annoying. */
static struct packet *
copy_packet(const struct packet *in)
{
	struct packet *out;
	out = xcalloc(sizeof(*out), 1);
	out->payload = malloc(in->len);
	memcpy((void *)out->payload, in->payload, in->len);
	out->eth = out->payload + ((unsigned long)in->eth -
				   (unsigned long)in->payload);
	out->ip = out->payload + ((unsigned long)in->ip -
				  (unsigned long)in->payload);
	out->ts = in->ts;
	out->len = in->len;
	return out;
}

/* Note that most packets are on the stack; only call this function on
   packets obtained from copy_packet. */
static void
release_packet(struct packet *p)
{
	free((void *)p->payload);
	free(p);
}

static void
release_datagram(struct datagram *dg)
{
	struct packet *p, *n;
	for (p = dg->head_packet; p; p = n) {
		n = p->next_off;
		release_packet(p);
	}
	*dg->pdg = dg->next;
	if (dg->next)
		dg->next->pdg = dg->pdg;
	free(dg);
}

static struct datagram *
head_dg;

static struct datagram *
find_datagram(n32 saddr, n32 daddr, n16 fragid)
{
	struct datagram **pdg;
	struct datagram *dg;
	pdg = &head_dg;
	dg = *pdg;
	while (dg && (dg->first_packet->ip->saddr.a != saddr.a ||
		      dg->first_packet->ip->daddr.a != daddr.a ||
		      dg->first_packet->ip->id.a != fragid.a)) {
		pdg = &dg->next;
		dg = *pdg;
		if (dg) {
			assert(dg->first_packet);
			assert(dg->first_packet->ip);
		}
	}
	if (dg)
		return dg;
	incomplete_datagrams++;
	dg = xcalloc(sizeof(*dg), 1);
	*pdg = dg;
	dg->pdg = pdg;
	return dg;
}

static void
handle_packet(struct packet *p)
{
	struct packet *np, *tmp;
	struct datagram *dg;

	if (!p->ip || p->ip->version != 4) {
		dump_unhandled_packet(p);
		return;
	}
	if (!(net_to_host(p->ip->frag_off) & 0x3fff)) {
		struct datagram triv_dg;
		triv_dg.head_packet = triv_dg.tail_packet = p;
		triv_dg.first_packet = triv_dg.last_packet = p;
		parse_datagram(&triv_dg);
		process_datagram(&triv_dg);
		return;
	}

	/* Okay, we have a fragment.  Find the datagram, and shove it
	 * in. */
	np = copy_packet(p);
	dg = find_datagram(p->ip->saddr, p->ip->daddr, p->ip->id);

	np->prev_time = dg->last_packet;
	np->next_time = NULL;
	if (dg->last_packet)
		dg->last_packet->next_time = np;
	else
		dg->last_packet = dg->first_packet = np;
	for (tmp = dg->head_packet;
	     tmp && (net_to_host(tmp->ip->frag_off) & 0x3fff) >
		     (net_to_host(np->ip->frag_off) & 0x3fff);
	     tmp = tmp->next_off)
		;
	if (!tmp) {
		/* Insert at the end */
		np->prev_off = dg->tail_packet;
		np->next_off = NULL;
		if (dg->tail_packet)
			dg->tail_packet->next_off = np;
		else
			dg->head_packet = dg->tail_packet = np;
	} else {
		/* Insert immediately before tmp */
		np->prev_off = tmp->prev_off;
		np->next_off = tmp;
		if (tmp->prev_off)
			tmp->prev_off->next_off = np;
		else
			dg->head_packet = np;
		tmp->prev_off = np;
		if (dg->tail_packet)
			dg->tail_packet = np;
	}

	fprintf(logfile, "Have a fragment for datagram %D.\n", dg);
	if (net_to_host(p->ip->frag_off) & 0x4000) {
		fprintf(logfile,
			"Huh? Fragment had DONT_FRAGMENT set (datagram %D)\n",
			dg);
	}

	/* Now walk the packet list and see if we've got a complete
	   datagram. */
	if ((net_to_host(dg->head_packet->ip->frag_off) & 0x1fff) != 0) {
		/* Frag off of first packet != 0 -> not complete. */
		return;
	}
	if (net_to_host(dg->tail_packet->ip->frag_off) & 0x2000) {
		/* Last fragment has MORE_FRAGMENTS set -> not
		 * complete */
		return;
	}
	for (tmp = dg->head_packet;
	     tmp != dg->last_packet;
	     tmp = tmp->next_off) {
		unsigned tmp_ends;
		assert((net_to_host(tmp->ip->frag_off) & 0x1fff) <=
		       (net_to_host(tmp->next_off->ip->frag_off) & 0x1fff));
		tmp_ends = net_to_host(tmp->ip->frag_off) & 0x1fff;
		tmp_ends += net_to_host(tmp->ip->tot_len);
		tmp_ends -= sizeof(struct iphdr);
		if (tmp_ends <
		    (net_to_host(tmp->next_off->ip->frag_off) & 0x1fff))
			return;
		if (tmp_ends >
		    (net_to_host(tmp->next_off->ip->frag_off) & 0x1fff)) {
			fprintf(logfile,
				"WARNING: overlapping fragments on datagram %D\n",
				dg);
			abort();
		}
	}

	fprintf(logfile, "Completing datagram %D.\n", dg);

	/* If we get here, we now have a complete datagram.  Pass it
	   down for further processing. */
	parse_datagram(dg);
	process_datagram(dg);
	release_datagram(dg);

	incomplete_datagrams--;
}

/* ------------------------------------------------------------------------- */
/* Stuff for talking to libpcap */

static void
process_packet(unsigned char *ignore,
               const struct pcap_pkthdr *hdr,
               const unsigned char *data)
{
	struct packet p;

	if (total_packets == 0)
		start_time = hdr->ts;
	last_packet_time = hdr->ts;
	memset(&p, 0, sizeof(p));
	p.payload = data;
	p.len = hdr->caplen;
	p.ts = hdr->ts;
	total_packets++;
	parse_packet(&p);
	handle_packet(&p);

	do_cleanup();
	if (total_packets % 100000 == 0) {
		struct timeval time_passed, now, rt_passed, pass_time;
		static struct timeval last_pass_finish;

		gettimeofday(&now, NULL);

		rt_passed.tv_sec = now.tv_sec - rt_start_time.tv_sec;
		rt_passed.tv_usec = now.tv_usec - rt_start_time.tv_usec;
		if (rt_passed.tv_usec < 0) {
			rt_passed.tv_usec += 1000000;
			rt_passed.tv_sec--;
		}

		time_passed.tv_sec = hdr->ts.tv_sec - start_time.tv_sec;
		time_passed.tv_usec = hdr->ts.tv_usec - start_time.tv_usec;
		if (time_passed.tv_usec < 0) {
			time_passed.tv_usec += 1000000;
			time_passed.tv_sec--;
		}

		pass_time.tv_sec = now.tv_sec - last_pass_finish.tv_sec;
		pass_time.tv_usec = now.tv_usec - last_pass_finish.tv_usec;
		if (pass_time.tv_usec < 0) {
			pass_time.tv_usec += 1000000;
			pass_time.tv_sec--;
		}
		last_pass_finish = now;

		fprintf(logfile,
			"Processed %ld.%.06ld seconds.  %lld packets (%lld datagrams, %d incomplete), of which %f tcp.\n",
			time_passed.tv_sec, time_passed.tv_usec,
			total_packets, total_datagrams,
			incomplete_datagrams,
			(double)tcp_datagrams / total_datagrams);
		fprintf(logfile,
			"Real time %ld.%.06ld / trace time = %f; pass time %ld.%.06ld\n",
			rt_passed.tv_sec, rt_passed.tv_usec,
			(rt_passed.tv_sec + rt_passed.tv_usec * 1e-6) /
			(time_passed.tv_sec + time_passed.tv_usec * 1e-6),
			pass_time.tv_sec, pass_time.tv_usec);
		fprintf(logfile,
			"Buffers: %d/%d, %d collected, %d fullness flushes\n",
			buffers_used, BUF_USE_THRESH, gced_buffers,
			buffer_full_flushes);
		fprintf(logfile,
			"File descriptors: %d/%d, %d collected\n",
			fds_used, FD_USE_THRESH, gced_fds);
		fprintf(logfile,
			"Connections: %d/%d (%d TCP, %d comatose, %d aborted), %d collected (%d non-comatose)\n",
			live_connections, LIVE_CONN_THRESH,
			nr_tcp_connections, comatose_connections,
			aborted_connections, gced_conns,
			gced_non_comatose_conns);
		fprintf(logfile,
			"%d packets on comatose connections.\n",
			datagrams_on_comatose_conns);
		fprintf(logfile,
			"Hash badness %f, fd hit rate %f, buffer hit rate %f.\n",
			(double)hash_probes / conn_lookups,
			(double)fd_hits / fd_requests,
			(double)buffer_hits / buffer_requests);
	}
}

int
main(int argc, char * argv[])
{
	pcap_t *p;
	struct tcpconn *tcp;
	int x;
	char fname[4097];
	FILE *f;
	struct bpf_program bpf;
	const char *bpf_program;

	if (argc == 2) {
		bpf_program = argv[1];
	} else if (argc == 1) {
		bpf_program = NULL;
	} else {
		errx(1, "Need at most one argument (bpf filter to use) (filenames go on stdin)");
	}

	for (x = 0; x < HASH_SIZE; x++)
		LIST_INIT(&hash_table[x]);
	TAILQ_INIT(&lru_head);
	TAILQ_INIT(&fd_lru);
	TAILQ_INIT(&buffer_lru);

	register_printf_specials();

	f = fopen("logfile", "w");
	if (!f)
		err(1, "openning logfile");
	setvbuf(f, NULL, _IONBF, 0);
	logfile = open_tee(stdout, f);

	sense_file = fopen("senses", "w");
	if (!f)
		err(1, "openning sense file");

	file_log = fopen("file_log", "w");
	if (!f)
		err(1, "openning file_log");

	gettimeofday(&rt_start_time, NULL);
	while (!feof(stdin)) {
		int r;
		fname[0] = 0;
		for (x = 0; x < 4096; x++) {
			r = fgetc(stdin);
			if (r == EOF || r == '\n')
				break;
			fname[x] = r;
		}
		fname[x] = 0;
		if (r == EOF)
			break;
		fprintf(logfile, "New file: %s\n", fname);
		p = pcap_open_offline(fname, pcap_errbuf);
		if (!p) {
			fprintf(logfile,
				"cannot open %s (%s)\n", fname, pcap_errbuf);
			break;
		}
		if (bpf_program) {
			if (pcap_compile(p, &bpf, (char *)bpf_program, 1, 0) < 0)
				errx(1, "cannot compile %s: %s",
				     bpf_program, pcap_geterr(p));
			pcap_setfilter(p, &bpf);
		}
		link_type = pcap_datalink(p);
		pcap_loop(p, -1, process_packet, NULL);
		pcap_close(p);
		if (bpf_program)
			pcap_freecode(&bpf);
	}

	fprintf(logfile, "pcap_loop exitted; going to final cleanup.\n");

	if (head_dg) {
		fprintf(logfile, "WARNING: still have some uncomplete datagrams.\n");
	}
	while (head_dg) {
		fprintf(logfile, "Uncomplete datagrams: %D\n", head_dg);
		dump_unhandled_datagram(head_dg);
		head_dg = head_dg->next;
	}


	tcp = TAILQ_LAST(&buffer_lru, struct tcpconn, buffer_lru);
	while (tcp) {
		assert(tcp->buffer);
		if (tcp->buffer_used) {
			flush_buffer(tcp);
			tcp->buffer_used = 0;
		}
		do_cleanup();
		tcp = TAILQ_PREV(&buffer_lru, struct tcpconn, buffer_lru, tcp);
	}

	tcp = TAILQ_LAST(&lru_head, struct tcpconn, lru_chain);
	while (tcp) {
		assert(tcp->buffer_used == 0);
		tcp = TAILQ_PREV(&lru_head, struct tcpconn, lru_chain, tcp);
	}

	fprintf(logfile, "Critical idle threshold %f.\n",
		critical_idle_thresh);

	fclose(logfile);
	fclose(f);
	return 0;
}
