#define _FILE_OFFSET_BITS 64
#define _GNU_SOURCE
#include <sys/mman.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <unistd.h>
#include <sched.h>
#include <stdio.h>
#include <err.h>
#include <stdlib.h>
#include <printf.h>
#include <string.h>
#include <pcap.h>

#include "sk98_timers.h"
#include "../uspace_if.h"

#define mb() asm volatile("lock; addl $0, (%%esp)":::"memory")

unsigned client_serials[256];

struct packet_data_area {
	unsigned char payload[SK98_EXPECTED_PACKET_SIZE];
};

static struct sk98_map_area_header *
m;
static struct packet_data_area *
packet_pool;
static FILE *
output_file;

static unsigned long long
bytes_written;
static unsigned
inferred_drops;

static void
process_message(unsigned token, unsigned len, struct timeval *tv)
{
	struct pcap_pkthdr *p;
	int r;

#if 1
	p = (struct pcap_pkthdr *)packet_pool[token].payload;
	p->ts = *tv;
	p->caplen = p->len = len;
	r = fwrite_unlocked(packet_pool[token].payload,
			    len + sizeof(struct pcap_pkthdr),
			    1,
			    output_file);
	if (r < 0)
		err(1, "writing packet header");
#else
	unsigned x = 0;
	unsigned long client, serial;
	/* Touch all of the bytes in the packet. */
	for (r = 0; r < len; r++)
		x += packet_pool[token].payload[r];
	/* Make sure that we're not dropping any packets.  Note that
	   this only works on the nprobe test rig */
	if (((unsigned long *)packet_pool[token].payload)[0] ==
	    0x03020100) {
		client = ((unsigned long *)packet_pool[token].payload)[4];
		serial = ((unsigned long *)packet_pool[token].payload)[5];
		if (client_serials[client] != serial) {
			inferred_drops++;
		}
		if (client_serials[client] == 0)
			printf("New client %ld.\n", client);
		client_serials[client] = serial + 1;
	}
#endif
	bytes_written += len + sizeof(p);
#ifndef NDEBUG
	memset(&packet_pool[token], 0xfb, SK98_EXPECTED_PACKET_SIZE);
#endif
	m->u2k_tokens[m->u2k_prod % SK98_RING_SIZE] = token;
	mb();
	m->u2k_prod++;
}

static void
discard_packets(void)
{
	/* Return packets to the OS without examining them */
	while (m->k2u_prod > m->k2u_cons) {
		m->u2k_tokens[m->u2k_prod % SK98_RING_SIZE] =
			m->k2u_pipe[m->k2u_cons % SK98_RING_SIZE].token;
		mb();
		m->u2k_prod++;
		m->k2u_cons++;
	}
	mb();
}

static int
printf_special_B_handler_func(FILE *f, const struct printf_info *info,
			      const void *const *args)
{
	int x;
	int count = 0;
	int r;
	const unsigned char *buf = *(const unsigned char **)args[0];

	for (x = 0; x < info->prec; x++) {
		r = fprintf(f, "%.2x%c", buf[x], (char)info->pad);
		if (r < 0)
			return r; /* uh oh... */
		count += r;
	}
	return count;
}

static int
printf_special_B_arginfo_func(const struct printf_info *info,
			      size_t n,
			      int *argtypes)
{
	if (n > 0)
		argtypes[0] = PA_POINTER;
	return 1;
}

static void
register_printf_specials(void)
{
	register_printf_function('B', printf_special_B_handler_func,
				 printf_special_B_arginfo_func);
}

int
main(int argc, char *argv[])
{
	int fd;
	struct sk98_ioctl_map ioctl_args;
	unsigned ind;
	unsigned token;
	clock_retimer_t **clock_retimers = NULL;
	struct pcap_file_header fhdr;
	int r;
	unsigned long packet_counter, total_packets;
	int nr_retimers = 0;
	int retimer_array_size = 0;
	int calibrated_retimers = 0;
	int interface;
	struct timeval last_status;
	unsigned pending_packets;

	register_printf_specials();

	initialise_timestamps(1, 78110207, NULL);
	output_file = fopen(argv[1], "w");
	if (output_file == NULL)
		err(1, "openning %s", argv[1]);
	fhdr.magic = 0xa1b2c3d4;
	fhdr.version_major = 2;
	fhdr.version_minor = 4;
	fhdr.thiszone = 0;
	fhdr.sigfigs = 6;
	fhdr.snaplen = -1;
	fhdr.linktype = DLT_EN10MB;
	if (fwrite(&fhdr, sizeof(fhdr), 1, output_file) < 0)
		err(1, "writing header to %s", argv[1]);
	fd = open("/dev/sk98", O_RDWR);
	if (fd < 0)
		err(1, "opening device");
	m = mmap(NULL, sizeof(*m), PROT_READ | PROT_WRITE,
		 MAP_SHARED, fd, 0);
	if (m == MAP_FAILED)
		err(1, "mapping device");
	memset(m->u2k_tokens, 0xfc, sizeof(m->u2k_tokens));
	ioctl_args.len = SK98_EXPECTED_PACKET_SIZE * (SK98_RING_SIZE - 1);
	ioctl_args.version = SK98_CURRENT_VERSION;
	packet_pool = malloc(ioctl_args.len);
	memset(packet_pool, 0xfa, ioctl_args.len);
	ioctl_args.start_addr = packet_pool;
	ioctl_args.offset = 0;
	if (ioctl(fd, SK98_IOCTL_MAP, &ioctl_args) < 0)
		err(1, "doing ioctl");
	printf("All looks good, starting calibration.\n");
	do {
		mb();
		while (m->k2u_cons >= m->k2u_prod)
			mb();
		ind = m->k2u_cons % SK98_RING_SIZE;
		interface = m->k2u_pipe[ind].interface;
		if (interface >= retimer_array_size) {
			printf("Extend retimer array.\n");
			clock_retimers = realloc(clock_retimers,
						 (interface + 1) *
						 sizeof(clock_retimers[0]));
			memset(clock_retimers + retimer_array_size,
			       0,
			       sizeof(clock_retimers[0]) *
			       (interface - retimer_array_size));
			retimer_array_size = interface + 1;
			printf("Extended retimer array.\n");
		}
		if (clock_retimers[interface] == NULL) {
			printf("New retimer.\n");
			clock_retimers[interface] = new_clock_retimer("", 0);
			nr_retimers++;
		}
		r = doTimer(clock_retimers[interface],
			    m->k2u_pipe[ind].tstamp,
			    0);
		if (r != 0) {
			calibrated_retimers++;
			printf("Calibrated %d of %d.\n", calibrated_retimers,
			       nr_retimers);
		}
		discard_packets();
	} while (nr_retimers != 2);

	gettimeofday(&last_status, NULL);
	printf("Calibration achieved.\n");
	packet_counter = 0;
	total_packets = 0;
	while (1) {
		struct timeval tv;
		mb();
		while (m->k2u_cons >= m->k2u_prod)
			mb();
		pending_packets = m->k2u_prod - m->k2u_cons;
		while (m->k2u_cons < m->k2u_prod) {
			ind = m->k2u_cons % SK98_RING_SIZE;
			interface = m->k2u_pipe[ind].interface;
			token = m->k2u_pipe[ind].token;
			if (interface == (unsigned short)-1) {
				/* The kernel decided not to use this
				   token for some reason.  Put it
				   straight back on the u2k pipe. */
				m->u2k_tokens[m->u2k_prod % SK98_RING_SIZE] = token;
				mb();
				m->u2k_prod++;
			} else if (interface >= retimer_array_size ||
				   clock_retimers[interface] == NULL) {
				printf("WARNING, didn't calibrate all interfaces.\n");
			} else {
				getTime(clock_retimers[interface],
					m->k2u_pipe[ind].tstamp,
					&tv, NULL);
				process_message(token,
						m->k2u_pipe[ind].len,
						&tv);
			}
			m->k2u_cons++;
			packet_counter++;
			total_packets++;
		}
		if (packet_counter > 1000) {
			struct timeval now;
			double delta_t;
			double load;
			gettimeofday(&now, NULL);
			delta_t = now.tv_sec - last_status.tv_sec +
				(now.tv_usec - last_status.tv_usec) * 1e-6;
			load = (double)pending_packets / SK98_RING_SIZE;
			if (delta_t > 1) {
				printf("Bandwidth: %e bits/second, load %e\n",
				       bytes_written * 8.0 / delta_t,
				       load);
				printf("%e packets/second (%ld total).\n",
				       packet_counter / delta_t,
				       total_packets);
				printf("%d drops, %d inferred.\n",
				       m->drop_counter, inferred_drops);
				bytes_written = 0;
				last_status = now;
				packet_counter = 0;
			}
		}
	}
}
