###############################################################################
#                                                                             #
#   Copyright 2005 University of Cambridge Computer Laboratory.               #
#                                                                             #
#   This file is part of Nprobe.                                              #
#                                                                             #
#   Nprobe is free software; you can redistribute it and/or modify            #
#   it under the terms of the GNU General Public License as published by      #
#   the Free Software Foundation; either version 2 of the License, or         #
#   (at your option) any later version.                                       #
#                                                                             #
#   Nprobe is distributed in the hope that it will be useful,                 #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#   GNU General Public License for more details.                              #
#                                                                             #
#   You should have received a copy of the GNU General Public License         #
#   along with Nprobe; if not, write to the Free Software                     #
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA #
#                                                                             #
###############################################################################


#
# Stuff for dealing with `ulonglongs' 
# (C: unsigned long long, Python class ulonglong)
#


##############################################################################

import string
from nprobe import ulong_int
import time


##############################################################################



#
# Convert a ulonglong (C: unsigned long long, Python class ulonglong) to a 
#  Python Long integer 
#

def ull2l_old(ull):

    j = ulong_int(ull.low)
    k = ulong_int(ull.high)

    l = long(j.half)*2 + j.odd + ((long(k.half)*2 + k.odd) << 32)

    return l


##############################################################################



#
# Convert a ulonglong (C: unsigned long long, Python class ulonglong) to a 
#  Python Long integer 
#

def ull2l(ull):

    j = ul2l(ull.low)
    k = ul2l(ull.high)

    l = j + (k << 32)

    return l


##############################################################################



#
# Convert a ulonglong (C: unsigned long long, Python class ulonglong) to a 
#  Python Long integer 
#

def ull2l_cmp(ull):

    l = ull2l_old(ull)
    l1 = ull2l_new(ull)

    if l != l1:
        print 'unequal ull2l() old=%d new=%d orig = %x' % (l, l1, ull)
        raw_input('...')

    return l


##############################################################################



#
# Convert a 32 bit unsigned int to a Python Long integer 
#

def ul2l_old(ul):

    j = ulong_int(ul)

    l = long(j.half)*2 + j.odd

    return l


##############################################################################



#
# Convert a 32 bit unsigned int to a Python Long integer 
#

def ul2l(ul):
    
    top = ul & 0x80000000
    l = long(ul & 0x7FFFFFFF)
    if top:
        l += long(0x40000000)*2
    return l


##############################################################################



#
# Convert a 32 bit unsigned int to a Python Long integer 
#

def ul2l_cmp(ul):

    l = ul2l_old(ul)
    l1 = ul2l(ul)

    if l != l1:
        print 'unequal ul2l() old=%d new=%d orig = %x' % (l, l1, ul)
        raw_input('...')

    return l


##############################################################################

#
# Return a Python Long integer as a string justified in a field width
#

def Longstring(l, width = 0, just = 'r'):
    
    if just == 'r':
	return string.rjust(repr(l)[:-1], width)
    elif just == 'l':
	return string.ljust(repr(l)[:-1], width)
    else:
	return string.center(repr(l)[:-1], width)


##############################################################################

#
# Return a Python Long integer i us as a h:m:s string 
# justified in a field width
#

def tsLongstring(l, width = 0, just = 'r'):

    sec = l/1000000
    usec = l % 1000000
    ts = "%02d:%02d:%02d.%06lu" % (sec/3600, (sec%3600)/60, sec % 60, usec)
    
    if just == 'r':
	return string.rjust(ts, width)
    elif just == 'l':
	return string.ljust(ts, width)
    else:
	return string.center(ts, width)


##############################################################################

#
# Return a Python Long integer i us as a date string 
# justified in a field width
#

def tsDatestring(l, width = 0, just = 'r'):

    sec = l/1000000
    usec = l % 1000000
    ttup = time.gmtime(sec)
    ts = "%02d:%02d:%02d.%06lu %d/%d/%d" % (ttup[3], ttup[4], ttup[5], usec, 
					    ttup[2], ttup[1], ttup[0])
    if just == 'r':
	return string.rjust(ts, width)
    elif just == 'l':
	return string.ljust(ts, width)
    else:
	return string.center(ts, width)


##############################################################################

#
# Return a ulonglong as a string justified in a field width
#

def ullstring(ull, width = 0, just = 'r'):

    return Longstring(ull2l(ull), width, just)


##############################################################################

#
# Return an unsigned int as a string justified in a field width
#

def ulstring(ul, width = 0, just = 'r'):

    return Longstring(ul2l(ul), width, just)


##############################################################################

#
# Convert a Class tval (analogue of struct timeval) into a Long in us
#

def tv2l(tv):

    us = long(tv.tv_usec)
    us += 1000000*long(tv.tv_sec)

    return us
