/*  -*- Mode: C;  -*- */

/******************************************************************************
*                                                                             *
*   Copyright 2005 University of Cambridge Computer Laboratory.               *
*                                                                             *
*   This file is part of Nprobe.                                              *
*                                                                             *
*   Nprobe is free software; you can redistribute it and/or modify            *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   Nprobe is distributed in the hope that it will be useful,                 *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with Nprobe; if not, write to the Free Software                     *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
*                                                                             *
******************************************************************************/


%module nprobe



%{
#include <stdio.h>
#include <linux/limits.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "http_errdefs.h"
#include "basic_defs.h"
#include "list.h"
#include "counters.h"
#include "np_file.h"
#include "flows.h"
#include "http.h"
#include "tcp.h"
#include "udp.h"
#include "udp_ns.h"
#include "icmp.h"
#include "except.h"
#include "sundry_records.h"
#include "if_stats.h"
#include "procstat.h"
#include "wread_util.h"
#include "print_util.h"
#include "flow_util.h"
#include "report.h"
#include "http_util.h"
#include "content_t.h"
#include "writer.h"
//#include "malloc_trace.h"
#include "malloc_rec.h"
#include "service.h"

%}

%include "http_errdefs.h"
%include "basic_defs.h"
%include "counters.h"
%include "np_file.h"
%include "fingerprint.h"
%include "flows.h"
%include "tcp.h"
%include "http.h"
%include "parse_object.h"
%include "udp.h"
%include "sundry_records.h"
%include "procstat.h"
%include "udp_ns.h"
%include "icmp.h"
%include "http_util.h"
%include "content_t.h"
%include "writer.h"
%include "service.h"

/*****************************************************************************/

/*
 * Some non-method functions 
 */

char *host_string(unsigned int addr);
char *intoa_string(unsigned int addr);
int _inet_aton(char *dotaddr);
void print_hostname(unsigned int addr);
void set_print_full_hostname(int flag);
char *port_string(unsigned short port, int proto);
char *rectype_str(int type);
char *atmdata_string(unsigned int atmdata);
unsigned char parse_type(char *arg);
char *http_server_returncode_string(unsigned short code);
/* char *http_server_objtype_string(unsigned short objtype); */
char *http_server_objtype_string(int objtype);
char *http_client_method_string(unsigned char method);
//char *linkrec_type_string(int type);
char *http_errstring(int err);
unsigned int n2hl(unsigned int addr);
int accept_conn(struct flow_inner *conn, int fspec);
void filter_help(void);
void _free(long addr);
void malloc_report(void);

%{
char *host_string(unsigned int addr)
{
  unsigned int a = addr;
  return get_hname((char *)&a);
}

char *intoa_string(unsigned int addr)
{
  //return inet_ntoa((struct in_addr)addr);
  return intoa(addr);

}

int _inet_aton(char *dotaddr)
{
  struct in_addr a;
  if (inet_aton(dotaddr, &a) == 0)
    //fprintf(stderr, "addr %s\n", dotaddr);
    //printf("addr %s\n", dotaddr);
    //wr_error("_inet_aton");
    a.s_addr = 0U;

  return (int)a.s_addr;
}

void print_hostname(unsigned int addr)
{
  unsigned int a = addr;
  printf("%s", get_hname((char *)&a));
}

void set_print_full_hostname(int flag)
{
  print_hostnames = flag;
}

char *port_string(unsigned short port, int prot)
{
  return tcpudp_port_string(port, (unsigned char)prot);
}

char *atmdata_string(unsigned int atmdata)
{
  return get_atmaddr(atmdata);
}

unsigned char parse_type(char *arg)
{
  return _parse_type(arg);
}

char *http_server_returncode_string(unsigned short code)
{
  return status_string(code);
}

/* char *http_server_objtype_string(unsigned short objtype)*/
char *http_server_objtype_string(int objtype)
{
  return content_type_string(objtype);
}


char *http_client_method_string(unsigned char method)
{
  return method_string(method);
}

/*  char *linkrec_type_string(int type) */
/*  { */
/*    return link_rec_type_string((unsigned char) type); */
/*  } */


unsigned int n2hl(unsigned int addr)
{
  return ntohl(addr);
}

char *http_errstring(int err)
{
  return http_err_string(err);
}

int accept_conn(struct flow_inner *conn, int fspec)
{
  return _pass(conn, fspec);
}

void filter_help(void)
{
  _filter_help();
}

char *rectype_str(int type)
{
  return np_rectype_str((unsigned char)type);
}


/* Free some allocated chunk of memory */
void _free(long addr)
{
  free((void *)addr);
}

void malloc_report(void)
{
  report_rec_malloc_full();
}


%};



/*
 * Want methods that return python objects to do so without further wrapping
 * 
 */
%typemap(python, out) PyObject *
{
  $target=(PyObject *)_result;
}

/*****************************************************************************/

/*%except(python) {




  char *err;
  clear_except();
  $tell_all
  if ((err = check_except()))
  fprintf(stderr, "%s\n"; err);
  }*/

/*%except(python);*/

/*****************************************************************************/

/*
 * Class  ulong_int - generated from basic-defs.h 
 * - used to convert unsigned ints to Python longs
 */

%addmethods ulong_int {

/* Constructor */
ulong_int(int ul)
{
  struct ulong_int *up;
  if ((up = (struct ulong_int *)malloc(sizeof(struct ulong_int))) == NULL)
    wr_error("ulong_int: malloc");
  up->half = ((unsigned int)ul)/2;
  up->odd = ((unsigned int)ul) & 0x1;

  return up;
}

/*Destructor */
~ulong_int()
{
  free(self);
}

}; /* End addmethods ulong_int */

/*****************************************************************************/

/* 
 * Class counters - generated from counters.h
 * Represents global run-time counters.
 */

%addmethods counters {

/* Constructor */
counters(){
  struct counters *cp;
  tmval tv = {0UL, 0UL};	
  if ((cp = (struct counters *)malloc(sizeof(struct counters))) == NULL)
    wr_error("counters: malloc");
  counters_init(cp, &tv, 0);
  
  return cp;
}

/*Destructor */
~counters()
{
  free(self);
}

/* Add in another counter */

void addin(counters_t *addnd)
{
  add_ctrs(self, addnd);
  return;
}

void printself(char *leader)
{
  report_counters(self, stdout, leader);
}

void printself_tofile(char *path, char *leader)
{
  FILE *file;
  if ((file = fopen(path, "w")) == NULL)
    wr_error("counters_printself_tofile(): open");
  report_counters(self, file, leader);
  if (fclose(file) != 0)
    wr_error("counters_printself_tofile(): close");
}

void printself_tofile_with_preamble(char *path, char *leader, char *preamble)
{
  FILE *file;
  if ((file = fopen(path, "w")) == NULL)
    wr_error("counters_printself_tofile(): open");
  fprintf(file, "%s", preamble);
  report_counters(self, file, leader);
  if (fclose(file) != 0)
    wr_error("counters_printself_tofile(): close");
}

/* Return run start time */
struct tval *run_start()
{
  return (struct tval *)&self->start;
}

/* Return run end time */
struct tval *run_end()
{
  return (struct tval *)&self->stop;
}



}; /* End addmethods counters */


/*****************************************************************************/

/* 
 * Class np_file - generated from np_file.h
 * Represents an Nprobe dump file and state for reading it together with 
 *  header info and associated counters.
 * Contains class fhdr_t
 */

%addmethods np_file { 

/* Constructor */
np_file(char *file, int build_indx)
{
  struct np_file *npf;	
  if ((npf = (struct np_file *)malloc(sizeof(struct np_file))) == NULL)
    wr_error("Np_file: malloc");
  
  strcpy(npf->fnm, file);
  _file_open(npf);
  
  if (build_indx)
    _indx(npf);
  else
    npf->offsets = NULL;		
  
  return npf;
}

/*Destructor */
~np_file() 
{
  if (fclose(self->file) != 0)
    wr_error("np_file_delete: fclose error");
  if (self->offsets)
    free(self->offsets);
  free(self);
}

/* Reset file */
void reset()
{
  _file_reset(self);
}


/* Print file global counters */
void printcounters(char *leader)
{
  report_counters(&self->counters, stdout, leader);
}

/* Print file global counters */
void printcounters_tofile(FILE *file, char *leader)
{
  report_counters(&self->counters, file, leader);
}

/* Print file name */
void printname()
{
  printf("%s\n", self->fnm);
}

/* Print file header */
void printhdr()
{
  printf("File %s (%s)\n", self->fnm, 
	 self->counters.fh.compressed != NOT_COMPRESSED ? "compressed"
	 : "not compressed");
  if (self->counters.fh.how_collected == TCPDUMP_COLL)
    printf("\tResults from dump file %s:\n", self->counters.fh.data);
  else
    printf("\tResults from online run:\n");
  printf("\t%lu records\n", 
	 self->counters.nrecords);
}

/* Get file open time */
struct tval *open()
{
  return (struct tval *)&self->counters.start;
}

/* Get file close time */
struct tval *close()
{
  return (struct tval *)&self->counters.stop;
}


/* Print entire contents - optionally of type X*/
int printfile(int type, int keytype, char *keystr, int quietflag)
{
  unsigned int key;
  struct in_addr addr;

  if(keytype == KEY_CONN_ID || keytype == KEY_REC)
    {
      key = strtoul(keystr, NULL, 0);
    }
  else if (keytype == KEY_SERV_ADDR || keytype == KEY_CLI_ADDR)
    { 
      if ((key = inet_aton(keystr, &addr)) == 0)
	{
	  fprintf(stderr, "Cannot translate address %s\n", keystr);
	  exit(1);
	}
      key = addr.s_addr;
    }
  else if (keytype)
    {
      wr_error("printfile: keytype error");
    }

  return _printfile(self, 1, (unsigned char)type, keytype, key, quietflag);
}

/* Filter for desired record type */
int filter(int type, char *ofile)
{
  return _filter_file(self, type, ofile);
}


/* Get # cpus of collecting machine */
int get_ncpus()
{
  return self->counters.fh.ncpus;
}

/* Get tick rate */
int get_hz()
{
  return self->counters.fh.hz;
}

/* Get page size */
int get_pgsz()
{
  return self->counters.fh.pgsz;
}

void seekrec(unsigned int rec)
{
  _seek_rec(self, rec);
}

/* Position to read next record - return record type */
int 
next_rec()
{
  return _next_rec(self, REC_ALL);
}

/* Position to read next record of specified type - return record type */
int 
next_type_rec(int type)
{
  return _next_rec(self, (unsigned char)type);
}

/* Advance over record to preeamble of next */
void advance()
{
  _advance(self);
}

/* Return index of current record */
unsigned int curr_indx()
{
  return self->indx -1;
}

/* Return 1 if record is of specified type */
int is_rectype(int type, int wanted)
{
  return _is_rectype(type, wanted);
}

/* Return current file offset */
long int offset()
{
  return _offset(self);
}

/* Return file offset of current record (after rec header) */
long int curr_offset()
{
  return self->curr_offset;
}

/* Seek to given offset */
int seek(long offset)
{
  return _seek(self, offset);
}

/* 
 * Position to read next record of type TCP/HTTP or UDP/NS 
 * - return record type or -1 if EOF 
 */
int find_next_http_or_ns_record()
{
  int type;

  while (1)
    {
      type = _next_rec(self, REC_IP);
      if (type == REC_TCP_HTTP || type == REC_UDP_DNS || type == -1)
	return type;
      else 
	_advance(self);
    }
}

/* 
 * Position to read next record of type TCP, TCP_OPEN or TCP_HDRS
 * - return record type or -1 if EOF 
 */
int find_next_tcp_and_hdrs()
{
  int type;
#if 0
  while (1)
    {
      type = _next_rec(self, REC_IP);
      if ((type > REC_TCP_MIN 
	  && type < REC_TCP_MAX)
	  || type == -1)
	return type;
      else 
	_advance(self);
    }
#endif
  return _next_rec(self, REC_TCP_ALL);
}

/* 
 * Position to read next record of type TCP/HTTP, TCP_OPEN, TCP_HDRS or UDP/NS 
 * - return record type or -1 if EOF 
 */
int find_next_http_and_hdrs_or_ns_record()
{
  int type;

  while (1)
    {
      type = _next_rec(self, REC_IP);
      if (type == REC_TCP_HTTP 
	  || type == REC_TCP_HTTP_OPEN 
	  || type == REC_TCP_HTTP_HDRS 
	  || type == REC_UDP_DNS 
	  || type == -1)
	return type;
      else 
	_advance(self);
    }
}

/* 
 * Position to read next record of type period report 
 * - return record type or -1 if EOF 
 */
int find_next_period_record()
{
  int type;

  while (1)
    {
      type = _next_rec(self, REC_OTHER_ALL);
      if (type == REC_PERIOD_REPORT
	  || type ==  REC_WIRE_PERIOD_REPORT 
	  || type == -1)
	return type;
      else 
	_advance(self);
    }
}
      
/*
 * Read a tcp connection open record - return the conn_id 
 */
int read_tcp_conn_open()
{
  struct tcp_open conn;

  _read_tcp_open(self, &conn);

  return conn.conn_id;
}


/*
 * Copy record at offset off to file fd renumbering as n - return new offset
 */
int copy_rec(long int off, int fd, int n)
{
  return _copy_rec(self, off, fd, n);
}

/*
 * Close new rep file by appending pseudo counters based on this one's
 */

void write_pseudocounters(int fd, int reci)
{
  _write_pseudocounters(fd, reci, &self->counters);
}
  
	
}; /* End addmethods np_file */

/****************************************************************************/

/*
 * Class tcp_open - record representing TCP connection open
 *
 * - no specific constructor/destructor required
 */

%addmethods tcp_open { 

/* Constructor */
tcp_open()
{
	struct tcp_open *tof;	
	if ((tof = (struct tcp_open *)malloc(sizeof(struct tcp_open))) == NULL)
	 	wr_error("tcp_open: malloc");
	return tof;
}
	
/*Destructor */
~tcp_open() 
{
  free(self);
}

/* populate the record */
void get_open_rec(struct np_file *file)
{
  _read_tcp_open(file, self);
  return;
}

/* quick accessor for conn_id */
int get_conn_id()
{
  return self->conn_id;
}

/* return the client address for the connection */
unsigned int shost()
{
  return self->flow.srcaddr;
}

/* return the server address for the connection */
unsigned int dhost()
{
  return self->flow.dstaddr;
}

/* Return source port */
unsigned short sport()
{
  return ntohs(self->flow.srcport);
}

/* Return dest port */
unsigned short dport()
{
  return ntohs(self->flow.dstport);
}


/* return the connection open time (first pkt seen) */

struct ulonglong *open()
{
  return (ulonglong *)&self->flow.first_arr_tm;
}


}; /* End addmethods tcp_open */

/****************************************************************************/


/*
 * Class tcp_conn - represents a TCP connection carrying HTTP traffic
 * - corresponds to an Nprobe dump file tcp connection record.
 * Comprises classes tcp (data about TCP connection) and http_conn (data about 
 *  HTTP transactions carried and chain of those transactions).
 */

%addmethods tcp_conn { 

/* Constructor */
tcp_conn()
{
	struct tcp_conn *tpf;	
	if ((tpf = (struct tcp_conn *)malloc(sizeof(struct tcp_conn))) == NULL)
	 	wr_error("tcp_conn: malloc");
	tpf->flow_inner.serv_type = TCP_SERV_OTHER;
	tpf->hdrs.hdrs = NULL;
	return tpf;
}
	
/*Destructor */
~tcp_conn() 
{
  //printf("Freeing tcp_conn at %#x\n", self);
  
  _dealloc_tcp_conn(self);
}

/*
 * Pre-allocate a transaction chain 
 */
void http_alloc_trans()
{
  _http_alloc_trans(self, MAX_NTRANS, PROVIDE_IMGBUFS);
  return;
}

/* Free the trans chain */
void free_http_trans_chain()
{
	assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
	if (self->su.http.trans != NULL)
		_http_dealloc_trans(self);
}

/*
 * Pre-allocate a hdr buff 
 */
void tcp_alloc_hdrbuffs()
{
  _tcp_alloc_hdrbuffs(&self->hdrs, MAX_TCP_DUMPHDRS_HELD);
  return;
}

/*
 * Get current TCP connection record from file including transaction chain
 * - _next_rec already called and record identified as TCP  
 */
void get_conn_and_trans(struct np_file *file)
{
  _read_tcp_conn(file, self, TRANS_PREALLOC, GET_TRANS);
  return;
}

/*
 * Get current TCP connection record from file excluding transaction chain
 * - _next_rec already called and record identified as TCP  
 */
void get_conn(struct np_file *file)
{
  _read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, NO_TRANS);
  self->indx = file->indx -1;
  return;
}

void get_conn_p(struct np_file *file)
{
  _read_tcp_conn(file, self, TRANS_PREALLOC, NO_TRANS);
  self->indx = file->indx -1;
  return;
}

/*
 * Get only the connection data, advance over any service data
 */
void get_conn_and_advance(struct np_file *file)
{
  _read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, NO_TRANS);
  self->indx = file->indx -1;
  _advance(file);
  return;
}

/*
 * Get only the connection data, advance over any service data
 */
void get_conn_and_advance_p(struct np_file *file)
{
  _read_tcp_conn(file, self, TRANS_PREALLOC, NO_TRANS);
  self->indx = file->indx -1;
  _advance(file);
  return;
}

/* Get next HTTP data from file into connection object (including transaction chain) */
int get_http_conn_and_trans(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_TCP_HTTP);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_TCP_HTTP);
      _read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, GET_TRANS);
      assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
      self->indx = file->indx -1;
      return rec_type;
    }
}

/* Get indexed HTTP data from file into connection object (including transaction chain) */
void get_http_conn_and_trans_indxd(struct np_file *file, unsigned int recno)
{
	_seek_rec(file, recno);
	_read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, GET_TRANS);
	assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
}

/* Get next HTTP data from file - excluding transaction chain */
int get_http_conn(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_TCP_HTTP);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_TCP_HTTP);
      _read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, NO_TRANS);
      assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
      self->indx = file->indx -1;
      return rec_type;
    }
}

/* Get indexed HTTP data from file - excluding transaction chain */
void getconn_indxd(struct np_file *file, unsigned int recno)
{
	_seek_rec(file, recno);
	_read_tcp_conn(file, self, TRANS_ALLOC_ON_FLY, NO_TRANS);
	assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
}

/* Given an HTTP connection record build a connected transaction chain */
void get_trans_chain(struct np_file *file)
{
	if (self->flow_inner.serv_type != TCP_SERV_HTTP)
		wr_error("get_trans_chain: not HTTP connection");
	_http_get_transchain(file, self);
}

/* ascii printout of tcp-conn */
void printself(int print_hdrs)
{
	report_tcp_conn(stdout, self, self->indx, print_hdrs);
}

/* ascii printout of tcp-conn */
void printself_tofile(char *path, int print_hdrs)
{
  FILE *file;
  if ((file = fopen(path, "w")) == NULL)
    wr_error("tconn_printself_tofile(): open");
  report_tcp_conn(file, self, self->indx, print_hdrs);
  if (fclose(file) != 0)
    wr_error("tconn_printself_tofile(): close");
}

/* quick accessor for conn_id */
int get_conn_id()
{
  return self->hdrs.conn_id;
}

/* quick accessor for hdrs.nheld */
int get_nhdrs_held()
{
  return self->hdrs.nheld;
}

/* quick accessor for hdrs base time */
struct ulonglong * get_hdrs_abstm()
{
  return (struct ulonglong *)&self->hdrs.atm;
}

/* populate a tcp_dumphdr type given an index into hdrs buffer*/
void get_hdr(tcp_dumphdr_t *hp, int indx)
{
  tcp_dumphdr_t *hdr = &self->hdrs.hdrs[indx];

  hp->rtm = hdr->rtm;
  hp->seq_u32 = hdr->seq_u32;
  hp->ack_u32 = hdr->ack_u32;
  hp->window = hdr->window;
  hp->len = hdr->len;
  hp->flags = hdr->flags;
  hp->way = hdr->way;
}

/*
 * Get tcp_dumphdr fields from the hdrs buffer for given index
 */

int get_rtm(int indx)
{
  return (int)self->hdrs.hdrs[indx].rtm;
}

int get_seq(int indx)
{
  return (int)self->hdrs.hdrs[indx].seq_u32;
}

int get_ack(int indx)
{
  return (int)self->hdrs.hdrs[indx].ack_u32;
}

int get_win_len(int indx)
{
  struct tcp_dumphdr *hdr = &self->hdrs.hdrs[indx];
  return (((int)hdr->window) << 16) + (int)hdr->len;
}
  

int get_flags_way(int indx)
{
  struct tcp_dumphdr *hdr = &self->hdrs.hdrs[indx];
  return (((int)hdr->flags) << 8) + (int)hdr->way;
}
  

/* quick accessor for No. transactions */
unsigned short http_ntrans()
{
	assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
	return self->su.http.meta.ntrans;
}

/* quick accessors for various connection times */

struct ulonglong *open()
{
  return (ulonglong *)&self->flow_inner.first_arr_tm;
}

struct ulonglong *close()
{
  return (ulonglong *)&self->flow_inner.last_arr_tm;
}

unsigned int clisyn()
{
  return self->tcp.client.syn_us;
}

/* NB time the client's SYN is ACKed */
unsigned int cliacksyn()
{
  return self->tcp.client.acksyn_us;
}

unsigned int servsyn()
{
  return self->tcp.server.syn_us;
}

/* NB time the server's SYN is ACKed */
unsigned int servacksyn()
{
  return self->tcp.server.acksyn_us;
}

unsigned int clifin()
{
  return self->tcp.client.fin_us;
}

unsigned int servfin()
{
  return self->tcp.server.fin_us;
}

unsigned int clirst()
{
  return self->tcp.client.rst_us;
}

unsigned int servrst()
{
  return self->tcp.server.rst_us;
}

unsigned int clifirstdata()
{
  return self->tcp.client.firstdata_us;
}

unsigned int servfirstdata()
{
  return self->tcp.server.firstdata_us;
}

unsigned int clilastdata()
{
  return self->tcp.client.lastdata_us;
}

unsigned int servlastdata()
{
  return self->tcp.server.lastdata_us;
}

/* quick accessor for src host NBO */
unsigned int shost()
{
  return self->flow_inner.srcaddr;
}

/* quick accessor for dst host NBO */
unsigned int dhost()
{
  return self->flow_inner.dstaddr;
}

/* Return source port */
unsigned short sport()
{
  return ntohs(self->flow_inner.srcport);
}

/* Return dest port */
unsigned short dport()
{
  return ntohs(self->flow_inner.dstport);
}

/* Print out src host */
void print_srchost()
{
	printf("%s", get_hname((char *)self->flow_inner.srcaddr));
}

/* Print out dst host */
void print_dsthost()
{
	printf("%s", get_hname((char *)self->flow_inner.dstaddr));
}

/* Return connection flags */
int get_flags()
{
  return self->flow_inner.state;
}

/* Return client side flags */
int get_cflags()
{
  return self->tcp.client.state;
}

/* Return server side flags */
int get_sflags()
{
  return self->tcp.server.state;
}

/* Return client side mss - this  */
int get_cmss()
{
  return self->tcp.client.mss;
}

/* Return server side mss */
int get_smss()
{
  return self->tcp.server.mss;
}

/* Return true if connection is HTTP */
int is_http()
{
	return (self->flow_inner.serv_type == TCP_SERV_HTTP);
}

/* Return true if connection is known service */
int serv_is_known()
{
	return (self->flow_inner.serv_type != TCP_SERV_OTHER);
}

/* Return 1 if traffic in both directions seen */
int both_seen()
{
	return ((self->flow_inner.state & TCP_CLIENT_SEEN)
		&& (self->flow_inner.state & TCP_SERVER_SEEN));
}

/* Return 1 if traffic from server seen */
int server_seen()
{
	return (self->flow_inner.state & TCP_SERVER_SEEN);
}

/* Return 1 if traffic from client seen */
int client_seen()
{
	return (self->flow_inner.state & TCP_CLIENT_SEEN);
}

/* Return 1 if connection persistent */
int http_persistent()
{
  assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
  return (self->su.http.meta.status & HTTP_WAS_PERSISTENT);
}

/* Return HTTP connection status */
int http_status()
{
  assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
  return self->su.http.meta.status;
}

/* Return HTTP client/server versions */
int http_vers()
{
  assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
  return self->su.http.meta.versions;
}

/* Return HTTP client/server versions as string */
char  *http_vers_str()
{
  assert(self->flow_inner.serv_type == TCP_SERV_HTTP);
  return http_versions_string(self->su.http.meta.versions);
}

/* Return source atm data */
unsigned int src_atmdata()
{
	return self->flow_inner.src_atmdata;
}

/* Return dest atm data */
unsigned int dst_atmdata()
{
	return self->flow_inner.dst_atmdata;
}
	

/* Return total octets transferred from the client on a connection */
unsigned int tot_client_octs()
{
  return _tcp_tot_client_octs(self);
}
	

/* Return total octets transferred  from the server on a connection */
unsigned int tot_server_octs()
{
  return _tcp_tot_server_octs(self);
}
	

/* Return total octets transferred on a connection (duplex) */
unsigned int tot_octs()
{  
  return _tcp_tot_octs(self);
}
	

/* Return total packets transferred from the client on a connection */
unsigned int tot_client_pkts()
{
  /* excludes retmts */
  return _tcp_tot_client_pkts(self);
}
	

/* Return total data packets transferred from the client on a connection */
unsigned int tot_client_dpkts()
{
  /* excludes retmts */
  return _tcp_tot_client_pkts(self) - self->tcp.client.tot_e_pkts;
}
	

/* Return total packets transferred from the server on a connection */
unsigned int tot_server_pkts()
{
  /* excludes retmts */
  return _tcp_tot_server_pkts(self);
}
	

/* Return total packets transferred from the server on a connection */
unsigned int tot_server_dpkts()
{
  /* excludes retmts */
  return _tcp_tot_server_pkts(self) - self->tcp.server.tot_e_pkts;
}


/* Return total pkts transferred on a connection (duplex) */
unsigned int tot_pkts()
{
  return _tcp_tot_pkts(self);
}

/* Return client/server low/high seq/ack */

unsigned int slowack()
{
  return self->tcp.server.firstack;
}

unsigned int shighack()
{
  return self->tcp.server.lastack;
}
unsigned int slowseq()
{
  return self->tcp.server.firstseq;
}

unsigned int shighseq()
{
  return self->tcp.server.solidseq;
}

unsigned int clowack()
{
  return self->tcp.client.firstack;
}

unsigned int chighack()
{
  return self->tcp.client.lastack;
}
unsigned int clowseq()
{
  return self->tcp.client.firstseq;
}

unsigned int chighseq()
{
  return self->tcp.client.solidseq;
}

int no_rtmts_or_gaps()
{
  struct tcp_simplex_flow *c = &self->tcp.client;
  struct tcp_simplex_flow *s = &self->tcp.server;

  return ((c->duplicate_pkts == 0L) 
	  && (c->ooo_pkts == 0L) 
	  && (c->gap_pkts == 0L) 
	  && (s->duplicate_pkts == 0L) 
	  && (s->ooo_pkts == 0L) 
	  && (s->gap_pkts == 0L));
}

int c_rtmts()
{
  struct tcp_simplex_flow *c = &self->tcp.client;

  return (int)(c->ooo_bytes | c->duplicate_bytes);
  //return (int)c->duplicate_pkts;
}

int s_rtmts()
{
  struct tcp_simplex_flow *s = &self->tcp.server;

  //return (int)(s->ooo_pkts | s->duplicate_pkts);
  return (int)(s->ooo_bytes | s->duplicate_bytes);
}

int n_c_ooo_pkts()
{
  return self->tcp.client.ooo_pkts;
}

int n_s_ooo_pkts()
{
  return self->tcp.server.ooo_pkts;
}

int n_c_dup_pkts()
{
  return self->tcp.client.duplicate_pkts;
}

int n_s_dup_pkts()
{
  return self->tcp.server.duplicate_pkts;
}

int n_c_gap_pkts()
{
  return self->tcp.client.gap_pkts;
}

int n_s_gap_pkts()
{
  return self->tcp.server.gap_pkts;
}

int has_delay()
{
  int hd = 0;
  
  if (self->tcp.client.state & TSP_DUP_SYN) hd = 1; //dup SYN
  if (self->tcp.client.duplicate_pkts) hd = 1; //rtmts
  if (!(self->tcp.server.state & TSP_SYN)) hd = 1; //not connected
  if (_tcp_tot_client_pkts(self) - self->tcp.client.tot_e_pkts == 0 
      || _tcp_tot_server_pkts(self) - self->tcp.server.tot_e_pkts == 0) hd = 1; //no req or resp

  return hd;
}

  


}; /* End addmethods tcp_conn */

/*****************************************************************************/

/*
 * Class tcp_hdrs - carries a block of dumped pkt headers 
 */

%addmethods tcp_hdrs {

/* Constructor */
tcp_hdrs(int prealloc_bufs)
{
  struct tcp_hdrs *hp;	
  if ((hp = (struct tcp_hdrs *)malloc(sizeof(struct tcp_hdrs))) 
      == NULL)
    wr_error("tcp_hdrs: malloc");

  if (prealloc_bufs)
    _tcp_alloc_hdrbuffs(hp, MAX_TCP_DUMPHDRS_HELD);

  return hp;
}

/* Destructor */
~tcp_hdrs()
{
  if (self->hdrs != NULL)
    {
      //printf("Freeing tcp_hdrs At %#x\n", self->hdrs);
      free(self->hdrs);
    }

  free(self);
}

/* Populate from dump file */
void get_hdrs(struct np_file *file)
{
  _read_tcp_hdrs(file, self, TRANS_ALLOC_ON_FLY);
}

/* Populate from dump file = buf pre_allocated */
void get_hdrs_p(struct np_file *file)
{
  _read_tcp_hdrs(file, self, TRANS_PREALLOC);
}

/* Return absolute time tag */
struct ulonglong *get_hdrs_abstm()
{
  return (struct ulonglong *)&self->atm;
}

/* Return number of headers held */
int get_nhdrs_held()
{
  return self->nheld;
}


/*
 * Get tcp_dumphdr fields from the hdrs buffer for given index
 */

int get_rtm(int indx)
{
  return (int)self->hdrs[indx].rtm;
}

int get_seq(int indx)
{
  return (int)self->hdrs[indx].seq_u32;
}

int get_ack(int indx)
{
  return (int)self->hdrs[indx].ack_u32;
}

int get_win_len(int indx)
{
  struct tcp_dumphdr *hdr = &self->hdrs[indx];
  return (((int)hdr->window) << 16) + (int)hdr->len;
}
  

int get_flags_way(int indx)
{
  struct tcp_dumphdr *hdr = &self->hdrs[indx];
  return (((int)hdr->flags) << 8) + (int)hdr->way;
}



}; /* End addmethods tcp_hdrs */

/*****************************************************************************/

/*
 * Class tcp_dumphdr - a dumped tcp pkt headers 
 */

%addmethods tcp_dumphdr {

/* Constructor */
tcp_dumphdr()
{
  struct tcp_dumphdr *dp;	
  if ((dp = (struct tcp_dumphdr *)malloc(sizeof(struct tcp_dumphdr))) 
      == NULL)
    wr_error("tcp_dumphdr: malloc");
  //_rec_malloc(dp);

  return dp;
}

/* Destructor */
~tcp_dumphdr()
{
  //fprintf(stderr, "Freeing tcp_dumphdr at %#x\n", self);
  //if (_rec_free(self, "tcp_dumphdr"))
  free(self);
}

/* Populate from tcp_hdrs type */
void get_hdr(struct tcp_hdrs *hdrs, int indx)
{
  struct tcp_dumphdr *hdr = &hdrs->hdrs[indx];

  self->rtm = hdr->rtm;
  self->seq_u32 = hdr->seq_u32;
  self->ack_u32 = hdr->ack_u32;
  self->window = hdr->window;
  self->len = hdr->len;
  self->flags = hdr->flags;
  self->way = hdr->way;
}

/* Populate from connrec type */
void get_hdr_clo(struct tcp_conn *conn, int indx)
{
  struct tcp_dumphdr *hdr = &conn->hdrs.hdrs[indx];

  self->rtm = hdr->rtm;
  self->seq_u32 = hdr->seq_u32;
  self->ack_u32 = hdr->ack_u32;
  self->window = hdr->window;
  self->len = hdr->len;
  self->flags = hdr->flags;
  self->way = hdr->way;
}

void hdr_printself(struct ulonglong *base_tm)
{
  print_hdr_rec(stdout, self, (us_clock_t *)base_tm);
}

void hdr_printself_rel()
{
  print_hdr_rec(stdout, self, NULL);
}

}; /* End addmethods tcp_dumphdr */

/*****************************************************************************/

/*
 * Class http_trans - represents an HTTP transaction (possibly chained)
 * Contains class http_trans_inner 
 */

%addmethods http_trans {

  /* Constructor */

http_trans()
{
  struct http_trans *htt;
  if ((htt = (struct http_trans *)malloc(sizeof(struct http_trans))) 
      == NULL)
    wr_error("http_trans: malloc");
  htt->links.buf = NULL;
  
  return htt;
}
	
/*Destructor */
~http_trans() 
{
  //printf("Freeing http_trans at %#x\n", self);
  if (self->links.buf != NULL && !self->hold_links)
    {
      //printf("Freeing links buf at %#x\n", self->links.buf);
      free(self->links.buf);
    }
  //  if (self->links.buf != NULL)
  //  free(self->links.buf);
  
  free(self);
}

/* Allocate a set of link buffers */
void http_links_buf_alloc()
{
  self->links.buf 
    = _links_buf_alloc(LINKS_MAX_BUFS*LINKS_BUFSZ);
}


/* Read data from file - MUST follow call to read TCP/HTTP connection record */
void http_gettrans(struct np_file *file, struct tcp_conn *tconnp)
{
	int client_seen = tconnp->flow_inner.state & TCP_CLIENT_SEEN;
  	int server_seen = tconnp->flow_inner.state & TCP_SERVER_SEEN;
	if (tconnp->flow_inner.serv_type == TCP_SERV_HTTP)
		_http_read_trans(self, file, client_seen, server_seen, tconnp->su.http.addit_trans_fields, TRANS_ALLOC_ON_FLY);
	else
		wr_error("http_gettrans: not HTTP connection");
}


/* Read data from file - MUST follow call to read TCP/HTTP connection record */
void http_gettrans_p(struct np_file *file, struct tcp_conn *tconnp)
{
	int client_seen = tconnp->flow_inner.state & TCP_CLIENT_SEEN;
  	int server_seen = tconnp->flow_inner.state & TCP_SERVER_SEEN;
	if (tconnp->flow_inner.serv_type == TCP_SERV_HTTP)
		_http_read_trans(self, file, client_seen, server_seen, tconnp->su.http.addit_trans_fields, TRANS_PREALLOC);
	else
		wr_error("http_gettrans: not HTTP connection");
}

/* Return requested object name */
char *getreq()
{
  if (self->inner.cinf.reqlen)
    return self->req;
  else
    return null_string;
}

/* Return object referrer object name */
char *getref()
{
  if (self->inner.cinf.reflen)
    return self->ref;
  else 
    return NULL;
}

/* Return object host name */
char *gethost()
{
  if (self->inner.cinf.hostlen)
    return self->host;
  else 
    return NULL;
}

/* Return transaction request seen time */
unsigned int http_reqstart_us()
{
  return self->inner.cinf.reqstart_us;
}

/* Return transaction request end time */
unsigned int http_reqend_us()
{
  return self->inner.cinf.reqend_us;
}

/* Return transaction first reply packet time */
unsigned int http_repstart_us()
{
  return self->inner.sinf.repstart_us;
}

/* Return transaction first body packet time */
unsigned int http_bodystart_us()
{
  return self->inner.first_sdata_pkt_us;
}

/* Return transaction last reply packet time */
unsigned int http_repend_us()
{
  return self->inner.sinf.repend_us;
}

/* Return server received body length */
unsigned int http_r_objlen()
{
  return self->inner.hserver.recd_len;
}

/* Return server http header length */
unsigned int http_r_hdrlen()
{
  return self->inner.rep_hdr_len;
}

/* Return server received body type */
unsigned short http_rep_objtype()
{
  return self->inner.hserver.content_type;
}

/* Return any client request body type */
unsigned short http_req_objtype()
{
  return self->inner.hclient.content_type;
}

/* Return number of in_line image URLs */
int http_n_imgurls()
{
  assert (self->inner.hserver.content_type == CT_TEXT_HTML);
  return _http_get_nurls(self->links.buf, self->links.totchars);
}

/* Return number of distinct in_line image URLs */
int http_ndist_imgurls()
{
  assert (self->inner.hserver.content_type == CT_TEXT_HTML);
  return _http_get_ndist_urls(self->links.buf, self->links.totchars);
}

/* Return number of link URLs */
int http_n_linkurls()
{
  assert (self->inner.hserver.content_type == CT_TEXT_HTML);
  return _http_get_nurls(self->links.buf, self->links.totchars);
}

/* Return number of distinct link URLs */
int http_ndist_linkurls()
{
  assert (self->inner.hserver.content_type == CT_TEXT_HTML);
  return _http_get_ndist_urls(self->links.buf, self->links.totchars);
}

/* return user-agent, server, via */
char *get_uagent()
{
  /* assert(self->inner.hclient.status & TRANS_VAL); */
  if (self->inner.cinf.uagentlen)
    return self->uagent;
  else
    return NULL;
}

/* return user-agent, server, via */
char *get_server()
{
  /* assert(self->inner.hserver.status & TRANS_VAL); */
  if (self->inner.sinf.serverlen)
    return self->server;
  else
    return NULL;
}

/* return user-agent, server, via */
char *get_svia()
{
  assert(self->inner.hserver.status & TRANS_VAL);
  if (self->inner.sinf.vialen)
    return self->svia;
  else
    return NULL;
}
char *get_cvia()
{
  assert(self->inner.hclient.status & TRANS_VAL);
  if (self->inner.cinf.vialen)
    return self->cvia;
  else
    return NULL;
}

PyObject *sfinger()
{
  fingerprint_t *f = &self->inner.sinf.finger;
  return Py_BuildValue("iii", f->hi, f->lo, f->tot);
}


/* Print out data */
void printself(struct tcp_conn *tconnp)
{
  print_trans(stdout, self, -1, 
	      tconnp->flow_inner.state & TCP_CLIENT_SEEN, 
	      tconnp->flow_inner.state & TCP_SERVER_SEEN,
	      tconnp->su.http.addit_trans_fields,
	      tconnp->flow_inner.first_arr_tm);
}

void printself_tofile(int indx, struct tcp_conn *tconnp, char *path, 
		      char *mode)
{
  FILE *file;
  if ((file = fopen(path, mode)) == NULL)
    wr_error("trans_printself_tofile(): open");
  print_trans(file, self, indx, 
	      tconnp->flow_inner.state & TCP_CLIENT_SEEN, 
	      tconnp->flow_inner.state & TCP_SERVER_SEEN,
	      tconnp->su.http.addit_trans_fields,
	      tconnp->flow_inner.first_arr_tm);
  if (fclose(file) != 0)
    wr_error("trans_printself_tofile(): close");
}

/* Return client side status */
int http_cli_status()
{
  return self->inner.hclient.status;
}

/* Return server side status */
int http_serv_status()
{
  return self->inner.hserver.status;
}

/* Return 1 if server transaction is valid */
int http_serv_isvalid()
{
  return self->inner.hserver.status & TRANS_VAL;
}

/* Return 1 if client transaction is valid */
int http_cli_isvalid()
{
  return self->inner.hclient.status & TRANS_VAL;
}

/* Return 1 if server transaction is a dummy */
int http_serv_isdummytrans()
{
  return self->inner.hserver.status & (TRANS_DUMMY_UNSYNCH | TRANS_DUMMY_ERR);
}

/* Return 1 if client transaction is a dummy */
int http_cli_isdummytrans()
{
  return self->inner.hclient.status & (TRANS_DUMMY_UNSYNCH | TRANS_DUMMY_ERR);
}

/* Return error number if server transaction resulted in error, else 0 */
int http_serv_iserr()
{
    return self->inner.hserver.error;
}

/* Return error number if client transaction resulted in error, else 0 */
int http_cli_iserr()
{
  return self->inner.hclient.error;
}

/* Return 1 if server transaction not good - no more in chain will be either */
int http_serv_nogood()
{
  return (self->inner.hserver.status & 
	  (TRANS_DUMMY_UNSYNCH | TRANS_DUMMY_ERR | TRANS_ERR))
    || !(self->inner.hserver.status & TRANS_VAL);
}

/* Return 1 if client transaction not good - no more in chain will be either */
int http_cli_nogood()
{
  return (self->inner.hclient.status & 
	  (TRANS_DUMMY_UNSYNCH | TRANS_DUMMY_ERR | TRANS_ERR))
    || !(self->inner.hclient.status & TRANS_VAL);
}
    

/* Return 1 if client side of transaction complete */
int http_cli_comp()
{
	return !(self->inner.hclient.status & TRANS_INCOMPLETE);
}

/* Return 1 if server side of transaction complete */

int http_serv_comp()
{
	return !(self->inner.hserver.status & TRANS_INCOMPLETE);
}

/* Return 1 if server side of transaction finished */

int http_serv_fin()
{
	return (self->inner.hserver.status & TRANS_FINISHED);
}

/* Return > 0 if transaction complete or finished */
int http_obj_comporfin()
{
  int status = self->inner.hserver.status;
  if (status & TRANS_INCOMPLETE)
    {
      if (status & TRANS_FINISHED)
	return 2;
      else
	return 0;
    }
  else
    {
      return 1;
    }
}

/* Return 1 if object size known and all received */
int http_obj_known_all_recd()
{
  return (self->inner.hserver.body_len != HTTP_BODY_LEN_UNKNOWN 
	  && self->inner.hserver.recd_len == self->inner.hserver.body_len);
}

/* Return object size (irrespective of successful completion, etc) */
int http_obj_size()
{
  return self->inner.hserver.body_len != HTTP_BODY_LEN_UNKNOWN ? 
    self->inner.hserver.body_len 
      : self->inner.hserver.recd_len + self->inner.hserver.gaps_bytes;
}

/* Return object claimed size */
int http_obj_len()
{
  return self->inner.hserver.body_len;
}

/* Return object bytes seen (including gaps) */
int http_obj_bytes()
{
  return self->inner.hserver.recd_len + self->inner.hserver.gaps_bytes;
}

/* Return object pkts seen (including gaps) */
int http_obj_pkts()
{
  return self->inner.hserver.recd_pkts + self->inner.hserver.gaps_pkts;
}
  

/* Return HTTP server return code */
int http_server_retcode()
{
  return self->inner.sinf.status_code;
}

/* Return HTTP request method */
int http_meth()
{
  return self->inner.cinf.method;
}

/* Return HTTP object type(s) */
int http_c_obtype()
{
  return self->inner.hclient.content_type;
}

int http_s_obtype()
{
  return (int)self->inner.hserver.content_type;
}

/* Return pointer to links buffer */
/*  struct links_chars *get_links_buf() */
/*  { */
/*    return &self->links; */
/*  } */

/*
 * Return pointer to links buffer
 *  - returned as long otherwise Swig/Python will turn it into a string
 * object
 */
long get_links_buf()
{
  return (long)(self->links.buf);
}

/* Return length of buffer */
int get_links_buflen()
{
  return self->links.totchars;
}

/* Return > 0 if can be implied that whole object has been delivered */
int http_obj_implied_comp(struct tcp_conn *tconnp)
{
	int status_code = self->inner.sinf.status_code;
	int status_ok = (status_code == 200 || status_code == 203 || status_code == 206);

	int body_len_ok = 
		(self->inner.hserver.body_len != HTTP_BODY_LEN_UNKNOWN 
		&& self->inner.hserver.body_len == self->inner.hserver.recd_len);

	if (body_len_ok && status_ok)
		/* got it all or successfully resynched */
		{
			if (self->inner.hserver.status & TRANS_RESYNCHED)
				return 1;
			else
				return 2;
		}

	else if (self->inner.hserver.body_len == HTTP_BODY_LEN_UNKNOWN  
		&& (tconnp->flow_inner.state & TCP_SERV_FIN)
		&& status_ok)
		/* 
		 * completion implied by server close 
		 * - can't be persistent or would have shown as error 
		 */
		{
			if (self->inner.hserver.status & TRANS_RESYNCHED)
				return 3;
			else
				return 4;
		}
		

	else if (tconnp->flow_inner.state & TCP_CLI_RST)
		return 5;

	else if (tconnp->flow_inner.state & TCP_CLI_RST)
		return 6;

	else if (status_code != 200 || status_code != 203)
		return self->inner.sinf.status_code;

	else return 0;
}
	 

/* Print out in squid log form */
void http_print_squidlike(struct tcp_conn *tconn, int code)
{
	_http_print_like_squidlog(self, tconn, code);
}



}; /* End addmethods http_trans */

/*****************************************************************************/


/*
 * Class udp_conn - represents a UDP connection 
 * - corresponds to an Nprobe dump file udp connection record.
 * Comprises class udp_conn (data about UDP connection)
 */

%addmethods udp_conn { 

/* Constructor */
udp_conn()
{
	struct udp_conn *upf;	
	if ((upf = (struct udp_conn *)malloc(sizeof(struct udp_conn))) == NULL)
	 	wr_error("udp_conn: malloc");
	upf->flow_inner.serv_type = UDP_SERV_OTHER;
	upf->service_data = NULL;
	return upf;
}
	
/*Destructor */
~udp_conn() 
{
  	_dealloc_udp_conn(self);
}

/* ascii print out of udp-conn */
void printself()
{
	report_udp_conn(self, self->indx);
}

/*
 * Get current UDP connection record from file - excluding service data
 * - next_rec already called and record identified as UDP  
 */
void get_conn(struct np_file *file)
{
  _read_udp_conn(file, self, DATA_ALLOC_ON_FLY, NO_DATA);
  self->indx = file->indx -1;
  return;
}

/* Get next UDP data from file - excluding service data */
int get_udp_conn(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_UDP_ALL);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type > REC_UDP_MIN && rec_type < REC_UDP_MAX);
      _read_udp_conn(file, self, DATA_ALLOC_ON_FLY, NO_DATA);
      self->indx = file->indx -1;
      return rec_type;
    }
}

/*
 * Get current UDP connection record from file - including service data
 * - next_rec already called and record identified as UDP  
 */
void get_conn_and_data(struct np_file *file)
{
  _read_udp_conn(file, self, DATA_ALLOC_ON_FLY, GET_DATA);
  return;
}

/* Get next UDP data from file - including service data */
int get_udp_conn_and_data(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_UDP_ALL);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type > REC_UDP_MIN && rec_type < REC_UDP_MAX);
      _read_udp_conn(file, self, DATA_ALLOC_ON_FLY, GET_DATA);
      self->indx = file->indx -1;
      return rec_type;
    }
}

/*
 * Get next UDP/NS record from file - excluding ns data
 */

int get_ns_conn(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_UDP_DNS);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_UDP_DNS);
      _read_udp_conn(file, self, TRANS_ALLOC_ON_FLY, NO_TRANS);
      assert(self->flow_inner.serv_type == UDP_SERV_DNS);
      self->indx = file->indx -1;
      return rec_type;
    }
}

/* Quick accessors for times */

struct ulonglong *open()
{
  return (struct ulonglong *)&self->flow_inner.first_arr_tm;
}
struct ulonglong *close()
{
  return (struct ulonglong *)&self->flow_inner.last_arr_tm;
}

/* Return true if connection is known service */
int serv_is_known()
{
  return (self->flow_inner.serv_type != UDP_SERV_OTHER);
}

/* Return total octets transferred by client in a udp flow */
unsigned int tot_client_octs()
{
  return _udp_tot_client_octs(self);
}

/* Return total octets transferred by server in a udp flow */
unsigned int tot_server_octs()
{
  return _udp_tot_server_octs(self);
}
	

/* Return total octets transferred in a udp flow (duplex) */
unsigned int tot_octs()
{
  return _udp_tot_octs(self);
}

/* Return total packets transferred by client in a udp flow */
unsigned int tot_client_pkts()
{
  return _udp_tot_client_pkts(self);
}

/* Return total packets transferred by server in a udp flow */
unsigned int tot_server_pkts()
{
  return _udp_tot_server_pkts(self);
}
	

/* Return total octets transferred in a udp flow (duplex) */
unsigned int tot_pkts()
{
  return _udp_tot_pkts(self);
}

/* quick accessor for src host NBO */
unsigned int shost()
{
  return self->flow_inner.srcaddr;
}

/* quick accessor for dst host NBO */
unsigned int dhost()
{
  return self->flow_inner.dstaddr;
}

/* Return source port */
unsigned short sport()
{
  return ntohs(self->flow_inner.srcport);
}

/* Return dest port */
unsigned short dport()
{
  return ntohs(self->flow_inner.dstport);
}

/* Return 1 if traffic in both directions seen */
int both_seen()
{
	return ((self->flow_inner.state & UDP_CLIENT_SEEN)
		&& (self->flow_inner.state & UDP_SERVER_SEEN));
}

/* Return 1 if traffic from server seen */
int server_seen()
{
	return (self->flow_inner.state & UDP_SERVER_SEEN);
}

/* Return 1 if traffic from client seen */
int client_seen()
{
	return (self->flow_inner.state & UDP_CLIENT_SEEN);
}

}; /* End addmethods udp_conn */

/*****************************************************************************/


/*
 * Class icmp_rec - represents an ICMP message
 */

%addmethods icmp_rec { 

/* Constructor */
icmp_rec()
{
  struct icmp_rec *ip;	
  if ((ip = (struct icmp_rec *)malloc(sizeof(struct icmp_rec))) == NULL)
    wr_error("icmp_rec: malloc");
  return ip;
}

/*Destructor */
~icmp_rec() 
{
  free(self);
}

/* Populate from file */
void get_rec(struct np_file *file)
{
  _read_icmp(file, self);
  return;
}

void printself(int indx)
{
  report_icmp(self, indx);
}

}; /* End addmethods icmp_rec */

/*****************************************************************************/


/*
 * Class ns_fullrec - represents an NS lookup 
 * - corresponds to an Nprobe dump file ns record.
 * Comprises class ns_fullrec
 */

%addmethods ns_fullrec { 

/* Constructor */
ns_fullrec()
{
	struct ns_fullrec *np;	
	if ((np = (struct ns_fullrec *)malloc(sizeof(struct ns_fullrec))) == NULL)
	 	wr_error("ns_fullrec: malloc");
	return np;
}
	
/*Destructor */
~ns_fullrec() 
{
  	_dealloc_ns_rec(self);
}

/* ascii print out of ns record */
void printself()
{
	report_ns(self);
}

/*
 * Get current ns record from file - UDP record has already been read  
 */
void get_ns_rec(struct np_file *file)
{
  _read_ns_rec(file, self, DATA_ALLOC_ON_FLY);
  return;
}

/* Quick accessors for request/response packet times */
unsigned int ns_reqtm()
{
  return self->ns_rec.req_us;
}

unsigned int ns_reptm()
{
  return self->ns_rec.rep_us;
}



}; /* End addmethods ns_fullrec */

/*****************************************************************************/

/*
 * Class rrent - used to parse rr buffer
 */

%addmethods rrent {

rrent()
{
  struct rrent *rp;	
  if ((rp = (struct rrent *)malloc(sizeof(struct rrent))) == NULL)
    wr_error("rrent: malloc");
  return rp;
}

void printself()
{
  printf("code %d\n", self->code);
  switch (self->code)
    {
    case RR_NONE: printf("None\n"); return; break;
    case RR_IPADDR: printf("%s -> %s\n", self->req, intoa(self->addr)); break;
    case RR_CNAME: printf("%s -> %s\n", self->req, self->can); break;
    case RR_NOT_INET:
    case RR_OTHER_TYPE:
    case RR_REQ: printf("Other\n"); break;
    default:  wr_error("rrent: printself"); break;
    }
}

int nextent(int indx, struct ns_fullrec *np)
{
  char errbuf[256];
  char *rr;
  char *cp;

  if (np->rrbuf == NULL)
    return -1;

  rr = np->rrbuf;
  cp = &rr[indx];

  self->code = (int)*cp;
  //printf("XXX %d\n", self->code); fflush(stdout);

  switch (*cp++)
    {
    case RR_NONE:
      return -1;
      break;
    case RR_NOT_INET:
      //printf("\tRR class ");
      //printf("%s", tok2str(class2str, "(Class %d)", *((unsigned short *)cp)));
      cp +=2;
      //printf("\n");
      break;
    case RR_OTHER_TYPE:
      //printf("\tRR type ");
      //printf("%s", tok2str(type2str, "(Type %d)", *((unsigned short *)cp)));
      cp +=2;
      //printf("\n");
      break;
    case RR_CNAME:
      //printf("\t");
      //printf("%s", cp);
      cp += sprintf(self->req, "%s", cp) + 1;
      //printf("  ->  ");
      //printf("%s", cp);
      cp += sprintf(self->can, "%s", cp) + 1;
      //printf ("\n");
      break;
    case RR_REQ:
      //printf("\t(");
      //printf("%s", cp);
      cp += strlen(cp) + 1;
      //printf(")\n");
      break;
    case RR_IPADDR:
      //printf("\t");
      //printf("%s", cp);
      cp += sprintf(self->req, "%s", cp) + 1;
      //printf("  ->  %s\n", intoa(*((unsigned int *)cp)));
      self->addr = *((int *)cp);
      cp += 4;
      break;
    default:
      sprintf(errbuf, "report_ns - unknown rr type %x\n", *(cp - 1));
      wr_error(errbuf);
      break;
    }
  
  return (int)(cp -rr);
}



}; /* End addmethods rrent */

/*****************************************************************************/


/*
 * Class linkrec - used to retrieve next link in il_img or links buffer
 */

%addmethods linkrec { 

/* Constructor */
linkrec()
{
  struct linkrec *lp;	
  if ((lp = (struct linkrec *)malloc(sizeof(struct linkrec))) == NULL)
    wr_error("linkrec: malloc");
  return lp;
}


~linkrec() 
{
  //fprintf(stderr, "Freeing link_rec\n");
  free(self);
}

/* Given index into buffer and ptr to it populate the record */
/*  int next_lrec(int indx, struct links_chars *bufp) */
/*  { */
/*    return _get_next_link_rec(self, indx, bufp); */
/*  } */

/* Given index into buffer and ptr to it populate the record */
int next_lrec(int indx, long bufp)
{
  return _get_next_link_rec(self, indx, bufp);
}



char *type_string()
{
  return link_rec_type_string((unsigned short) self->type);
}

void set_type(int type)
{
  self->type = (uint)type;
}



}; /* End addmethods linkrec */

/*****************************************************************************/


/*
 * Class sk_buff_alloc_fail_rec - records nprobe sk_buf allocation failures
 */

%addmethods sk_buff_alloc_fail_rec { 

/* Constructor */
sk_buff_alloc_fail_rec()
{
  struct sk_buff_alloc_fail_rec *fr;	
  if ((fr = (struct sk_buff_alloc_fail_rec *)malloc(sizeof(struct sk_buff_alloc_fail_rec))) == NULL)
    wr_error("sk_buff_alloc_fail_rec: malloc");
  return fr;
}
	
/*Destructor */
~sk_buff_alloc_fail_rec() 
{
  free(self);
}

/* ascii print out of record */
void printself(unsigned int indx)
{
  report_buf_alloc_fail(self, indx);
}

/*
 * Get current buf_alloc fail record from file
 * - next_rec already called and record identified as fail record  
 */
void read_failrec(struct np_file *file)
{
  _read_buf_alloc_fail(file, self);
  return;
}

/* Get next buf_alloc fail record from file  */
int get_next_failrec(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_BUF_ALLOC_FAIL);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_BUF_ALLOC_FAIL);
      _read_buf_alloc_fail(file, self);
      //self->indx = file->indx -1;
      return rec_type;
    }
}

}; /* End addmethods sk_buff_alloc_fail_rec */

/*****************************************************************************/


/*
 * Class period_report - records nprobe accounting period data
 */

%addmethods period_report { 

/* Constructor */
period_report()
{
  struct period_report *pr;	
  if ((pr = (struct period_report *)malloc(sizeof(struct period_report))) == NULL)
    wr_error("period_report: malloc");
  return pr;
}
	
/*Destructor */
~period_report() 
{
  free(self);
}

/* ascii print out of record */
void printself(unsigned int indx, counters_t *ctrs, int why)
{
  report_period_report(self, indx, ctrs, why);
}

/*
 * Get current period report record from file
 * - next_rec already called and record identified as period report record 
 * - if not wire-driven requires ptr to ru struct to read following ru record 
 */
void read_reprec(struct np_file *file, struct np_rusage *rup, int read_rup)
{
  if (read_rup)
    _read_period_report(file, self, rup);
  else
    _read_period_report(file, self, NULL);
  return;
}

/* Get next report record from file  */
int get_next_reprec(struct np_file *file, struct np_rusage *rup)
{
  int rec_type = _next_rec(file, REC_PERIOD_REPORT);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_PERIOD_REPORT);
      return _read_period_report(file, self, rup);
      //self->indx = file->indx -1;
      return rec_type;
    }
}

/* Return time stamp */
struct tval *get_ts()
{
  return (struct tval *)&self->ts;
}

/* Return HTML octets parsed */
int get_html_parsed()
{
  return self->html_parsed;
}

/* Return max fetch interval */
int get_max_fetch_interval()
{
  return self->max_fetch_interval;
}

/* Return rep bytes written */
int get_rep_bytes_written()
{
  return self->rep_bytes_written;
}

/* Return dump bytes written */
int get_dump_bytes_written()
{
  return self->dump_bytes_written;
}

/* Return rep blks dumped */
int get_rep_blks_dumped()
{
  return self->repblks_dumped;
}

/* Return dump blks dumped */
int get_dump_blks_dumped()
{
  return self->dumpblks_dumped;
}

/* Return wire pkts */
int get_wire_pkts()
{
  return self->wire_pkts;
}

/* Return TCP pkts */
int get_tcp_pkts()
{
  return self->tcp_pkts;
}

/* Return UDP pkts */
int get_udp_pkts()
{
  return self->udp_pkts;
}

/* Return HTTP pkts */
int get_http_pkts()
{
  return self->http_pkts;
}

/* Return tous_in */
int get_tous_in()
{
  return (int)(self->tous_in);
}

/* Return tous_out */
int get_tous_out()
{
  return (int)(self->tous_out);
}

/* Return frus_in */
int get_frus_in()
{
  return (int)(self->frus_in);
}

/* Return frus_out */
int get_frus_out()
{
  return (int)(self->frus_out);
}

/* Return tous_in - tous_out */
int get_tous_inout()
{
  return (int)(self->tous_in - self->tous_out);
}

/* Return frus_in - frus_out */
int get_frus_inout()
{
  return (int)(self->frus_in - self->frus_out);
}

/* Return timeout values */

int get_tcp_to()
{
  return (int)self->tcp_to;
}

int get_udp_to()
{
  return (int)self->udp_to;
}

int get_tcp_seq_to()
{
  return (int)self->tcp_seq_to;
}

int get_tcp_seq_to_forced()
{
  return (int)self->tcp_seq_to_forced;
}

int get_tcp_seq_to_q()
{
  return (int)self->tcp_seq_to_q;
}

int get_tcp_seq_to_ack()
{
  return (int)self->tcp_seq_to_ack;
}

/* Get rejected packet < min IP len */
int get_IP_ulen()
{
  return (int)self->IP_ulen_pkts;
}



}; /* End addmethods period_report */

/*****************************************************************************/


/*
 * Class np_rusage - records nprobe accounting period data
 */

%addmethods np_rusage { 

/* Constructor */
np_rusage()
{
  struct np_rusage *rup;	
  if ((rup = (struct np_rusage *)malloc(sizeof(struct np_rusage))) == NULL)
    wr_error("np_rusage: malloc");
  return rup;
}
	
/*Destructor */
~np_rusage() 
{
  free(self);
}

/* ascii print out of record 
 * - suppress printing rec_no and time if print_stamp == 0
 */
void printself(unsigned int indx, int print_stamp)
{
  report_rusage(self, indx, print_stamp);
}

/* Get next report record from file  */
int get_next_rusage(struct np_file *file)
{
  int rec_type = _next_rec(file, REC_RUSAGE);
  if (rec_type == -1)
    {
      return -1;
    }
  else
    {
      assert(rec_type == REC_RUSAGE);
      _read_rusage(file, self);
      return rec_type;
    }
}

/* Populate class instance */
void read(struct np_file *file)
{
  _read_rusage(file, self);
}

/* Return time stamp */
struct tval *get_ts()
{
  return (struct tval *)&self->ts;
}

/* Return user processor time */
struct tval *get_utime()
{
  return (struct tval *)&self->ru.ru_utime;
}

/* Return system processor time */
struct tval *get_stime()
{
  return (struct tval *)&self->ru.ru_stime;
}

/* Return max RSS */
long max_rss()
{
  return self->ru.ru_maxrss;
}

/* Return max RSS */
long majflt()
{
  return self->ru.ru_majflt;
}

/* Return # swaps */
long nswap()
{
  return self->ru.ru_nswap;
}

/* Return # block io's */
long blkio()
{
  return self->ru.ru_inblock + self->ru.ru_oublock;
}

/* Return # context switches */
long cswitches()
{
  return self->ru.ru_nvcsw + self->ru.ru_nivcsw;
}  

}; /* End addmethods np_rusage */

/*****************************************************************************/


/*
 * Class  wrapper_record
 */

%addmethods wrapper_record {
  

/* Constructor */
wrapper_record()
{
  struct wrapper_record *wp;	
  if ((wp = (struct wrapper_record *)malloc(sizeof(struct wrapper_record))) == NULL)
    wr_error("wrapper_record: malloc");
  return wp;
}
	
/*Destructor */
~wrapper_record() 
{
  free(self);
}

/* ascii print out of record */
void printself(unsigned int indx)
{
  report_wrapper(self, indx);
}

/* Populate */
void read(struct np_file *file)
{
  _read_wrapper(file, self);
}

/* get internal record type */
int type()
{
  return self->type;
}

/*
 * Get internal fields for type call_times
 */

/* Time stamp */
struct tval *get_ts()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (struct tval *)&self->data.call_times.ts;
}

/* IP - usr */
struct ulonglong *get_ip_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.ip.utm;
}

/* IP - sys */
struct ulonglong *get_ip_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.ip.stm;
}

/* UDP - usr */
struct ulonglong *get_udp_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.udp.utm;
}

/* UDP - sys */
struct ulonglong *get_udp_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.udp.stm;
}

/* TCP - usr */
struct ulonglong *get_tcp_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp.utm;
}

/* TCP - sys */
struct ulonglong *get_tcp_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp.stm;
}

/* TCP in sequence  - usr */
struct ulonglong *get_tcp_inseq_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_inseq.utm;
}

/* TCP in sequence  - sys */
struct ulonglong *get_tcp_inseq_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_inseq.stm;
}

/* TCP catch up  - usr */
struct ulonglong *get_tcp_catchup_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_catchup.utm;
}

/* TCP catch up   - sys */
struct ulonglong *get_tcp_catchup_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_catchup.stm;
}

/* TCP service  - usr */
struct ulonglong *get_tcp_serv_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_serv.utm;
}

/* TCP service   - sys */
struct ulonglong *get_tcp_serv_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.tcp_serv.stm;
}

/* HTTP - usr */
struct ulonglong *get_http_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.http.utm;
}

/* HTTP - sys */
struct ulonglong *get_http_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.http.stm;
}

/* HTML - usr */
struct ulonglong *get_html_usrtm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.html.utm;
}

/* HTML - sys */
struct ulonglong *get_html_systm()
{
  assert(self->type == WRAPPER_TIME_CALLS);
  return (ulonglong *)&self->data.call_times.html.stm;
}  

}; /* End addmethods wrapper_record */


/*****************************************************************************/

/*
 * Class  procstat_rec
 */

%addmethods procstat_rec {
  

/* Constructor */
procstat_rec()
{
  struct procstat_rec *prp;	
  if ((prp = (struct procstat_rec *)malloc(sizeof(struct procstat_rec))) == NULL)
    wr_error("procstat_rec: malloc");
  return prp;
}
	
/*Destructor */
~procstat_rec() 
{
  free(self);
}

/* Populate */
void read(struct np_file *file)
{
  _read_procstats(file, self);
}

/* Get time stamp */
struct tval *get_ts()
{
  return (struct tval *)&self->ts;
}

/* Get nprobe process usr time */
int get_p_utm()
{
  return (int)self->proc.utm;
}

/* Get nprobe process sys time */
int get_p_stm()
{
  return (int)self->proc.stm;
}

/* Get nprobe process major page faults */
int get_p_majflt()
{
  return (int)self->proc.majflt;
}

/* Get nprobe process swaps */
int get_p_nswap()
{
  return (int)self->proc.nswap;
}

/* Get system total cpu jiffies */
int get_s_tcpu()
{
  return (int)(self->stat.utm + self->stat.stm + self->stat.itm);
}

/* Get system interrupts */
int get_s_itot()
{
  return (int)self->stat.itot;
}

/* Get system context switches */
int get_s_csw()
{
  return (int)self->stat.csw;
}  

}; /* End addmethods procstat_rec */

/*****************************************************************************/

