###############################################################################
#                                                                             #
#   Copyright 2005 University of Cambridge Computer Laboratory.               #
#                                                                             #
#   This file is part of Nprobe.                                              #
#                                                                             #
#   Nprobe is free software; you can redistribute it and/or modify            #
#   it under the terms of the GNU General Public License as published by      #
#   the Free Software Foundation; either version 2 of the License, or         #
#   (at your option) any later version.                                       #
#                                                                             #
#   Nprobe is distributed in the hope that it will be useful,                 #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#   GNU General Public License for more details.                              #
#                                                                             #
#   You should have received a copy of the GNU General Public License         #
#   along with Nprobe; if not, write to the Free Software                     #
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA #
#                                                                             #
###############################################################################


#
# Some basic utility functions for manipulating TCP/HTTP records in Nprobe logs
#

import string
import glob
import os
import sys
from sys import argv

from np_file_util import EOF_REACHED
from nprobe import tcp_conn, http_trans, MAX_NTRANS


##############################################################################

#
# Given an Nprobe rec file return (the next) TCP/HTTP tconn record, its file 
# index and list of associated transactions
#

def get_next_http_rec(file):

        # get tconn record and its data
	connrec = tcp_conn()
	indx = connrec.get_http_conn(file)

	if indx == -1: #EOF
	    # destructor expects a valid server type value
	    #connrec.tcp.common.serv_type = 255
	    connrec.flow_inner.serv_type = 255
	    raise EOF_REACHED(file)

	# build the transaction list
	translist = []

	ntrans = connrec.su.http.meta.ntrans
	while ntrans > 0:
		trans = http_trans()
		trans.http_gettrans(file, connrec)
		translist.append(trans)
		ntrans = ntrans - 1

	return (indx, connrec, translist)

##############################################################################

#
# Given an Nprobe rec file lined up at the next TCP/HTTP tconn record return 
# the record, its file index and list of associated transactions
#

def get_http_rec(file):

	# get tconn record and its data
	connrec = tcp_conn()
	connrec.get_conn(file)
	indx = connrec.indx

	# build the transaction list
	ntrans = connrec.su.http.meta.ntrans
	translist = []
	while ntrans > 0:
		trans = http_trans()
		trans.http_gettrans(file, connrec)
		translist.append(trans)
		ntrans = ntrans - 1

	return (indx, connrec, translist)


##############################################################################

#
# Given a TCP connection record with chained transactions return a list of 
# contained HTTP transactions
#


def get_trans_list(tconn):
	i = 1
	ntrans = tconn.http.meta.ntrans
	tlist = []
	trans = tconn.http.trans
	tlist.append(trans)
	while i < ntrans:
		trans = trans.next
		tlist.append(trans)
		i = i + 1
	return tlist


##############################################################################

#
# Return a tcp_conn object and maximum list of http_trans objects for repeated
# use
#

def allocate_http_reusable_objects():

	tc = tcp_conn()
	tc.tcp_alloc_hdrbuffs()
	tl = []
	for i in range(MAX_NTRANS):
		tr = http_trans()
		tr.http_links_buf_alloc()
                tr.hold_links = 0
		tl.append(tr)

	return (tc, tl)


##############################################################################

#
# Given an Nprobe rec file lined up at the next TCP/HTTP tconn record
# a pre-allocated tcp_conn object and list of http_trans objects read the data
# from file into the objects
# 

def get_http_rec_and_trans(file, connrec, translist):

	# get tconn record data
	connrec.get_conn_p(file)
	indx = connrec.indx

	# build the transaction list
	ntrans = connrec.su.http.meta.ntrans
	i = 0
	while i < ntrans:
		translist[i].http_gettrans_p(file, connrec)
		i += 1

	return ntrans
