/*
**
** FILE: nexus_util.c - Nexus Utilities.
**
** The nexus is generated by at build time by the 'nembuild' utility.
** For the case where we have M modules, N namespaces and P programs,
** it looks something like this:
**
**   -------------------  (Low end of memory)
**   |      tag        |
**   |   entry point   |    Nexus Primal Info.
**   |    namespace    |
**   |    last addr    |
**   -------------------
**   |      tag        |
**   |     taddr       | 
**   |     daddr       |
**   |     baddr       |    NTSC Info.
**   |     tsize       |
**   |     dsize       | 
**   |     bsize       |
**   -------------------
**   |      tag        |  
**   |      addr       |    Nexus Info.
**   |      size       |
**   -------------------
**   |                 |  
**   |   Module Info   |
**   |    (module 1)   |
**   |                 |
**   -------------------
**   |                 |  
**   |   Module Info   |
**   |    (module 2)   |
**   |                 |
**   -------------------
**   :                 :
**   :                 :
**   -------------------
**   |                 |  
**   |   Module Info   |
**   |    (module M)   |
**   |                 |
**   -------------------
**   |   Blob 1        |
**   -------------------
**   |   Blob n        |
**   -------------------
**   | Blob descriptors|
**   -------------------
**   |                 |  
**   |   Namespace 1   |
**   |                 |
**   -------------------
**   |                 |  
**   |   Namespace 2   |
**   |                 |
**   -------------------
**   :                 :
**   :                 :
**   -------------------
**   |                 |  
**   |   Namespace N   |
**   |                 |
**   -------------------
**   |                 |  
**   |    Program  1   |
**   |                 |
**   -------------------
**   |                 |  
**   |    Program  2   |
**   |                 |
**   -------------------
**   :                 :
**   :                 :
**   -------------------
**   |                 |  
**   |    Program P    |
**   |                 |
**   -------------------
**   |      tag        |    Nexus EOI 
**   |   last addr     |
**   -------------------  (High end of memory)
**
** (lastaddr is the address of the first free page after the nemesis
** image)
**
** This is a bit messy to deal with sometimes, particularly coz of all
** the variable length parts. To deal with this efficiently, we make
** ourselves stateful.  But for modesty's sake, we squirrel away all
** the nasty details inside this file.
** 
*/


#include <nexus.h>
#include <ntsc.h>	/* for syscalls */

/* 
** matches(prefix, suffix, name) returns non-zero if 
** "prefix>suffix" = "name"
*/
int matches(const char *prefix, const char *suffix, const char *name)
{
    if(prefix) {
	while(*prefix && (*prefix++==*name++))
	    ;
    
	if(*prefix || *name !='>')
	    return 0;
	else name++;
    }

    while(*suffix && (*suffix++==*name++))
	  ;
    
    if(*suffix || *name)
	return 0;
    else return 1;
}


/*
** prefix(prefix, name) returns "suffix" if name is of the form
** "prefix/suffix" or "prefix>suffix"; otherwise it returns NULL
*/
char *prefix(const char *prefix, const char *name)
{
    char *res= name;

    if(prefix==NULL)
	return(res);

    while(*prefix && (*prefix++==*res++))
	;
    
    if(*prefix || *res !='>')
	return (NULL);
    else return(++res);
}



static namespace_entry *cur_name;
static char *cur_prefix;

/*
** set_namespace() is really nasty.  What is does is define the
** current name and associated prefix for names to be looked up using
** lookup_name.  A value of NULL for prefix means no prefix
** expected. The current name should be the first in the namespace
** (the one got at using that "naddr" field of a nexus_name record.)
*/
void set_namespace(namespace_entry *namespace, char *prefix)
{
    cur_name= namespace;
    cur_prefix= prefix;
    return;
}


/*
** lookup_name() looks inside the current namespace for the specified name, 
** prefixed with the current prefix. 
** If it fails to find it, it generates NULL
*/
void *lookup_name(char *name)
{
    namespace_entry *nptr= cur_name;

    while (nptr->name)
    {
	if (matches(cur_prefix, name, nptr->name))
	    return nptr->ptr;

	nptr++;
    }
    return NULL;
}


/*
** lookup_next() returns the next suffix and value of the next name within
** the current namespace which matches the current prefix.
** If there are no more matching entries left, it returns NULL.
**
** NB: stateful. Side effect is that it modifies the current name pointer.
** Also note the result is a pointer into the namespace strings.
*/
char *lookup_next(addr_t *val)
{
    char *res;

    while (cur_name->name)
    {
	if ((res=prefix(cur_prefix, cur_name->name))!=NULL) {
	    *val= cur_name->ptr;
	    cur_name++;
	    return res;
	}
	cur_name++;
    }

    return NULL;
}



static void *nexus_start;
static nexus_prog *cur_prog;


/*
** find_next_prog() - returns the next program in the program list, or NULL if
** there are no more present (as indicated by the nexus_EOI). 
** Panics if anything else happens.
**
** NB: init_nexus() must have been called before we call this routine.
*/

addr_t find_next_prog()
{
    nexus_ptr_t nexus_ptr;
    addr_t nexus_end;

    if(cur_prog==(nexus_prog *)NULL) {

	/*
	** First time here.   
	** We need to find the first program, which involves running
	** through the variable length nexus stuff until we get to the
	** start of the program list. Note that the program we find (and
	** we are guaranteed one as long as all is ok) is NOT NECESSARILY
	** the 'primal' program, it is the first one specified in the .nbf
	** file.  Thus we can set cur_prog to this, and find_next_prog()
	** will work fine.
	*/

	nexus_ptr= (nexus_ptr_t)nexus_start;
	nexus_end= nexus_start+FRAME_SIZE;	/* initial estimate */
	while((addr_t)nexus_ptr.generic < nexus_end) {
	    
	    switch(*nexus_ptr.tag) {
		
	      case nexus_primal:
		nexus_ptr.nu_primal++;
		break;
		
	      case nexus_ntsc:
		nexus_ptr.nu_ntsc++;
		break;
		
	      case nexus_nexus:
		nexus_end=nexus_ptr.nu_nexus->size+nexus_start;
		nexus_ptr.nu_nexus++;
		break;
		
	      case nexus_namespace:
		nexus_ptr.generic=(char *)nexus_ptr.generic + 
		    nexus_ptr.nu_name->nmods*sizeof(void *);
		nexus_ptr.nu_name++;
		break;
		
	      case nexus_module:
		nexus_ptr.nu_mod++;
		break;

	    case nexus_blob:
		nexus_ptr.nu_blob++;
		break;		
		
	    case nexus_program:
		cur_prog= nexus_ptr.nu_prog;
		return(cur_prog);
		break;
		
	      case nexus_EOI:
	      default:
		/* Gone too far, or bogus tag */
		ntsc_halt();
		return NULL; 
	    }
	}
    } else { /* been here before; just inc cur_prog */

	cur_prog++;
	
	if(cur_prog->tag == nexus_program)
	    return cur_prog;
	else if(cur_prog->tag == nexus_EOI)
	    return NULL;
	
	/* Here we have found something after the programs, but before
	   the EOI => our nexus is corrupted. This is fatal. */
	eprintf("find_next_prog(fatal): got something  >progs, <EOI.");
	eprintf("Halting system.\n");
	ntsc_halt();
	return NULL; 
	
    }
}



static int streq(const char *s1, const char *s2)
{
    while(*s1 && *s2 && (*s1++ == *s2++))
	;
    return(*s1 == '\0' && *s2 == '\0');
}



/*
** init_nexus() gets some of the stateful stuff sorted out, and hence should
** be called before any other functions in here are used.
** Returns the start & end addresses of the boot image, and the size 
** the initial heap should be (from the "primal" entry in nbf).
*/
void init_nexus(void *nexus, /* OUT */ addr_t *image_start, 
		addr_t *image_end, word_t *init_heapsz) 
{
    nexus_ptr_t n = (nexus_ptr_t)nexus;
    struct nexus_primal *pinfo;
    nexus_prog *prog;

    /* assign some intial values to our 'state' */
    nexus_start = nexus;
    cur_name    = (namespace_entry *)NULL;
    cur_prefix  = (char *)NULL;
    cur_prog    = (nexus_prog *)NULL;

    if (*n.tag != nexus_primal) {
	/* This is not good ;-( */
	*image_start = *image_end = 0;
	return;
    }

    pinfo        = (struct nexus_primal *)n.nu_primal;
    *image_start = pinfo->firstaddr;
    *image_end   = pinfo->lastaddr;

    /* We get the initial heap size from the prog info for "primal" */
    while((prog = find_next_prog())!=NULL) {
	if(streq("Primal", prog->program_name)) {
	    *init_heapsz = prog->params.astr;
	    break;
	}
    }

    /* Reset the prog globals for later on */
    nexus_start = nexus;
    cur_prog    = (nexus_prog *)NULL;

    if(prog == NULL) {
	/* Failed to find "Primal"; not good. Hope for the best?!? */
	*init_heapsz = (256 << 10);
	return;
    }

    return;
}


/* End */

