/******************************************************************************
 * private.h
 * 
 * Shared structures which are private to the protocol library.
 * 
 * Copyright (c) 1999-2000, K A Fraser
 * 
 * $Id: private.h,v 3.2 1999/12/18 16:27:38 kaf24 Exp kaf24 $
 */

#ifndef _PRIVATE_H_
#define _PRIVATE_H_

#include "ipv4/ipv4.h"  /* Linux protocol stack public definitions. */

#include "thread.h"
#include "debug.h"
#define SOCK ((pcb) ? (pcb->uid) : (0))
#include "profile.h"

/******************************************************************************
 * Table where we hash connection state on socket id.
 */
#include "hash.h"
extern hash_table_t *pcbs;

/******************************************************************************
 * Some convenient simple type definitions.
 */
#undef TRUE
#undef FALSE
typedef enum {FALSE, TRUE} bool;

/******************************************************************************
 * TCP and UDP: specification of receive and transmit data queues.
 */
#define BUFFERS_PER_RING   32   // max number of packets in each queue
#define BYTES_PER_BUFFER   0x3000 // max number of bytes in each packet

#define TCP_FIFO_SIZE        (BUFFERS_PER_RING * BYTES_PER_BUFFER)
#define TCP_FIFO_GRANULARITY BYTES_PER_BUFFER

#include "lowlevel.h"

/******************************************************************************
 * Regions in the shared data area.
 */
#define RX_DATA    0 
#define TX_DATA    (RX_DATA + TCP_FIFO_SIZE + TCP_FIFO_GRANULARITY)
#define RX_HEADER  (TX_DATA + TCP_FIFO_SIZE + TCP_FIFO_GRANULARITY)
#define TX_HEADER  (RX_HEADER + BUFFERS_PER_RING * MAX_HEADER_LEN)
#define SHARED_DATA_SIZE (TX_HEADER + BUFFERS_PER_RING * MAX_HEADER_LEN)

extern pthread_mutex_t global_poll_mutex;
extern pthread_cond_t  global_poll_cond;

extern __inline__ int min(int a, int b) { return((a<b)?a:b); }
extern __inline__ int max(int a, int b) { return((a>b)?a:b); }

typedef struct mem_block_st mem_block_t;
struct mem_block_st { mem_block_t *next; };
#define ADD_TO_MEM_LIST(__l, __e) ({ (__e)->next = (__l); (__l) = (__e); }) 
#define REMOVE_FROM_MEM_LIST(__l) \
({ mem_block_t *__e; if ( (__e = (__l)) ) (__l) = __e->next; (void *)__e; })

/* A simple set of connection state bits. */
enum { STATE_LOCAL_BOUND=1, STATE_CONNECTED, STATE_LISTENING, STATE_CLOSING};

/* Our wrapper around the socket structure. */
struct user_pcb
{
    /* Miscellaneous state. */
    int uid;         /* Unique identifier for this socket.                */
    int type;        /* The type of socket that we are.                   */
    u32 state;       /* What state we are in.                             */
    int fcntl_flags; /* State returned/set by the `fcntl' interface call. */
    
    /* Information for our yielding algorithm. */
#define PACKETS_PER_YIELD 5
    int recvs_since_upcall;   /* Packets received by BH since call to TH. */
    int sends_since_downcall; /* Packets queued by TH since call to BH.   */

    /* Our view of the data path to the user-safe device. */
    usd_device_conn_t *usd_conn;
    caddr_t     shared_data_area;

    /* Our view of the Linux stack. */
    struct u_socket sock; /* connection to top    */
    struct u_sock *sk;    /* connection to bottom */

    /* Buffer management. */
    mem_block_t *tx_free_hdr_list;
    mem_block_t *tx_free_data_list; 
    struct u_sk_buff_head tx_queued_skbuffs;
};

/******************************************************************************
 * Upcall thread creation/deletion functions.
 */
int  init_upcalls();
void kill_upcalls(void);

/******************************************************************************
 * skbuff functions -- called from outside the stack itself.
 */
int init_locked_tx_mem(struct user_pcb *pcb,
                       int num_hdr_bufs,
                       int hdr_size,
                       u_char *hdr_space,
                       int data_size,
                       u_char *data_space);
int rx_skb(struct user_pcb *pcb);
int free_tx_skbs(struct user_pcb *pcb);

struct u_socket *create_new_socket(int protocol);
void free_socket(struct u_socket *sock);
int bind_and_connect_new_socket(struct u_socket *sock,
                                u32 saddr,
                                u16 sport,
                                u32 daddr,
                                u16 dport);

#endif _PRIVATE_H_
