/*
 *  acenic_usd.h
 *  ------------
 *
 *  User-level bits of the USD channel control path for the Alteon
 *  gigabit ethernet card.
 *
 *  Copyright (c) 1999-2000, University of Cambridge
 *
 *  Tim Deegan, Cambridge University Computer Laboratory
 *  April-June 1999
 *
 *  Modifications by Ian Pratt and Keir Fraser
 *
 */

#ifndef _ACENIC_USD_H_
#define _ACENIC_USD_H_

#include <asm/types.h>
#include <asm/bitops.h>

typedef u8 uint8; typedef u16 uint16; typedef u32 uint32;
#include "dpf/dpf.h"

#include "acenic.h" 

#undef AT_LEAST_ONE_CACHE_LINE

#ifdef __KERNEL__ 

/*
 *  Callbacks to the client from the driver.
 *
 *  These functions should be *very* lightweight, just enough to wake up
 *  the user application, as they run inside an interrupt handler.  Also,
 *  for those who enjoy the barking side of life, remember that they 
 *  must return!
 */

/*
 *  Type of the function that gets called to signify completion of a 
 *  USD connection setup in the card.  The arguments are:
 *  1: (void *) An arbitrary pointer, supplied at setup request time.
 *  2: (int)    USD id for the new channel, or 0 for an error.
 */

typedef void (usd_setup_callback_t)(void *, int);

/*
 *  Type of the function that gets called when a running channel takes 
 *  an interrupt from the card.  One argument, the samer pointer that was 
 *  passed to the setup callback of this channel.
 */

typedef void (usd_irq_callback_t)(void *);

/*
 *  Type of the function called to indicate that the channel has been
 *  deleted from the system.  Again, one argument, the same pointer as
 *  with the other callbacks.
 */

typedef void (usd_teardown_callback_t)(void *);

/*
 *  Functions for setting up and tearing down connections.
 *  N.B. These are liable to change as functionality is added.
 */

extern int 
acenic_new_connection (struct net_device *dev, 
		       void *buffer,
		       u32 size,
		       usd_setup_callback_t *callback,
		       void *argument,
                       usd_irq_callback_t *rx_irq_call,
                       usd_irq_callback_t *tx_irq_call,
                       usd_teardown_callback_t *teardown_call,
		       u8   *dest_mac_addr,
		       u32   src_ip_addr,
		       u32   dest_ip_addr,
		       u16   src_port,
		       u16   dest_port,
		       u8    ip_proto);
/* dev           = ethernet driver's device struct.
 * buffer        = ptr to 'size' contiguous bytes of nailed memory.
 * size          = size of the buffer in bytes ( >= USD_ENDPOINT_MIN_SIZE ).
 * callback      = function to call on setup completion.
 * argument      = arg 1 of the callback calls.
 * irq_call      = fn to call when this channel takes an interrupt.
 * teardown_call = fn to call when this channel is destroyed.
 * dest_mac_addr = ptr to 6 byte MAC address, (which need not persist).
 * src_ip_addr   = the source IP address of valid frames on this channel.
 * dest_ip_addr  = the destination IP address of valid frames on this channel.
 * src_port      = the source TCP or UDP port number of valid frames.
 * dest_port     = the destination TCP or UDP port number of valid frames.
 * ip_proto      = the protocol in use (IPPROTO_UDP or IPPROTO_TCP only).
 *
 * returns 0 for success or an error code of -EFOO. 
 *
 * The RX path will route incoming packets _from_ the 'destination'
 * address/protocol/port _to_ the 'source' into this channel.
 */

extern int 
acenic_del_connection (struct net_device *dev, int id);
/* dev = ethernet driver's device struct.
 * id  = USD id of channel to delete.
 *
 * returns 0 to indicate that the request was passed to the card, 
 *         non-zero if that ID was not valid or not in use 
 */


extern int
ace_install_connection_filter(struct net_device *dev, int id, 
                              struct dpf_ir *ir);
extern int
ace_del_connection_filter(struct net_device *dev, int id);
/*
 * dev = ethernet driver's device struct.
 * id  = USD id of channel tpo install filter for.
 * ir  = filter code to process, merge and install.
 */


/*
 * Functions for fast trap fun: how to punt indices to the card
 *
 * First, the fn for sending a command to the card
 */


static __inline__ int acenic_usd2_set_tx2_producer_index(
    struct net_device *dev, 
    u16 chan, 
    u32 prod)
{
    struct ace_private *ap = dev->priv;
#ifdef ACENIC_DEBUGGING_IOCTLS    
    /* Make sure the command path hasn't been trampled by the debugging
     * ioctls.  Important to build with the *same* copy of acenic.h as 
     * the acenic driver... */
    writel(USD_CMD_BASE, &ap->regs->WinBase);
#endif

    /* Put the PROD into the relevant slot */
    ap->regs->usd2_ctrl[chan].tx_prod = prod; // U16 write

    /* Tell the card to poll that slot, using channel Id as hint */
    ap->regs->UsdMb[USD_MBOX_BASE + (chan % NUM_USD_MBOXS)].lo = chan;

    /* done */
    return 0;
}



static __inline__ int acenic_usd2_set_rx2_producer_index(
    struct net_device *dev, 
    u16 chan, 
    u32 prod)
{
    struct ace_private *ap = dev->priv;

    mbox_t *mb;

#ifdef ACENIC_DEBUGGING_IOCTLS    
    /* Make sure the command path hasn't been trampled by the debugging
     * ioctls.  Important to build with the *same* copy of acenic.h as 
     * the acenic driver... */
    writel(USD_CMD_BASE, &ap->regs->WinBase);
#endif

    /* Put the PROD into the relevant slot */
    ap->regs->usd2_ctrl[chan].free_prod = prod; // U16 write

    /* Tell the card to poll that slot, using channel Id as hint */

    mb = &ap->regs->UsdMb[USD_MBOX_BASE + (chan % NUM_USD_MBOXS)];
    mb -= 16;

    mb->lo = chan; 

    /* done */
    return 0;
}


/* This function will result in a callback when rx_prod crosses the
value loaded into rx_ref, or potentially earlier if a `safe race'
occurs */

static __inline__ void acenic_usd2_rx2_request_callback( 
    struct net_device *dev, 
    u16 chan, 
    u32 ref)
{
    struct ace_private *ap = dev->priv;
    mbox_t *mb;

    unsigned long flags;
    //XXXX
    spin_lock_irqsave(&ap->lock, flags);


#ifdef ACENIC_DEBUGGING_IOCTLS    
    /* Make sure the command path hasn't been trampled by the debugging
     * ioctls.  Important to build with the *same* copy of acenic.h as 
     * the acenic driver... */
    writel(USD_CMD_BASE, &ap->regs->WinBase);
#endif



    /* Set the (un)mask bit in the driver */
    //ap->rx_int_mask[chan/32] |= 1<<(chan%32);
    set_bit( chan%32, & ap->rx_int_mask[chan/32] );  // SMP safe

    /* If an acenic interrupt occurs in the next few cycles, then we
       may end up getting called back early. This is not dangerous,
       unlike never getting called back... */

    /* Put the PROD into the relevant slot */
    ap->regs->usd2_ctrl[chan].rx_ref = ref; // U16 write

    /* Tell the card to poll that slot, using channel Id as hint */

    mb = &ap->regs->UsdMb[USD_MBOX_BASE + (chan % NUM_USD_MBOXS)];
    mb -= 16;
    mb->lo = chan; 

    spin_unlock_irqrestore(&ap->lock, flags);


    /* done */
    return;
} 

/* This function will result in a callback when tx_cons crosses the
value loaded into tx_ref, or potentially earlier if a `safe race'
occurs */

static __inline__ void acenic_usd2_tx2_request_callback( 
    struct net_device *dev, 
    u16 chan, 
    u32 ref)
{
    struct ace_private *ap = dev->priv;
    mbox_t *mb;

    unsigned long flags;
    //XXXX
    spin_lock_irqsave(&ap->lock, flags);


#ifdef ACENIC_DEBUGGING_IOCTLS    
    /* Make sure the command path hasn't been trampled by the debugging
     * ioctls.  Important to build with the *same* copy of acenic.h as 
     * the acenic driver... */
    writel(USD_CMD_BASE, &ap->regs->WinBase);
#endif



    /* Set the (un)mask bit in the driver */
    //ap->tx_int_mask[chan/32] |= 1<<(chan%32);
    set_bit( chan%32, & ap->tx_int_mask[chan/32] );  // SMP safe

    /* If an acenic interrupt occurs in the next few cycles, then we
       may end up getting called back early. This is not dangerous,
       unlike never getting called back... */

    /* Put the PROD into the relevant slot */
    ap->regs->usd2_ctrl[chan].tx_ref = ref; // U16 write

    /* Tell the card to poll that slot, using channel Id as hint */

    mb = &ap->regs->UsdMb[USD_MBOX_BASE + (chan % NUM_USD_MBOXS)];
    mb->lo = chan; 

    spin_unlock_irqrestore(&ap->lock, flags);


    /* done */
    return;
} 


#endif __KERNEL__ 

#endif /* _ACENIC_USD_H_ */

/*
 *  EOF (acenic_usd.h)
 */
