/*
 *  ctxt.c
 *  ------
 *
 *  Context switch support for USD on acenic.
 *
 *  TJD:      May 1999
 *  KAF, IAP: 1999-2000
 *
 *  Copyright (c) 1999-2000, University of Cambridge
 *
 *  Please read the FIRMWARE-LICENSE file distributed with this source file
 *  for details of the license agreement.
 *
 *  $Id: ctxt.c,v 1.22 2000/03/31 12:19:13 kaf24 Exp iap10 $
 */

#include "alt_def.h"
#include "nic_conf.h"
#include "nic_api.h"
#include "tg.h"

#include "nic.h"
#include "proto.h"
#include "ring.h"
#include "mac.h"
#include "usd_conn.h"
#include "assert.h"
#include "usd2_util.h"
#include "event.h"


/******************************************************************************
 **** DMA ASSIST RING CLEANUP
 */

/*
 * Clears all requests related to <id> from the given assist <ring>.
 * Calls a <specific_cleanup> function for each nuked entry, so global 
 * resources can be freed.
 */
static void clear_assist_ring(tgDmaDescr_t *ring, 
                              tgDmaDescr_t *ref,
                              tgDmaDescr_t *prod,
                              void (*specific_cleanup)(tgDmaDescr_t*),
                              U16 id)
{
    int i;
    tgDmaDescr_t *dd;

    NIC_UTRACE("ClrAsst", 0x99294, ring, ref, prod, 0 );

    for ( i = 0, dd = ref; 
          dd != prod; 
          i++, dd = (dd == (ring + TG_DMA_ASSIST_DESCRS - 1)) ? ring : dd+1 )
    {
        if ( dd->usd_id == id )
        {
            (*specific_cleanup)(dd);
	    dd->len  = 0;
	    dd->type = TIGON_TYPE_NULL;
        }
	if( i > TG_DMA_ASSIST_DESCRS ) PANIC();	
    }
}

static void read_lo_cleanup(tgDmaDescr_t *dd)
{
    if ( dd->type == TIGON_TYPE_SEND_BD )
    {
        nicbfp->bd_dmas_in_q--; 
    }
    else if ( dd->type == TIGON_TYPE_SEND_DATA_LAST ) 
    {
        nicbfp->tx2_free_bufs |= 1 << dd->index;
    }
}

static void write_lo_cleanup(tgDmaDescr_t *dd) {}

static void write_hi_cleanup(tgDmaDescr_t *dd)
{
    if ( dd->type == TIGON_TYPE_RECV_DATA_LAST ) 
    {
        // mark descriptor as BAD -- we'll skip it later...
        tdp->rx_mac_ring[dd->index].w0 = TG_MAC_RX_W1_STATUS_MASK;
    }
}

static void read_hi_cleanup(tgDmaDescr_t *dd)
{
    if ( dd->type == TIGON_TYPE_FREE_BD ) nicfp->free_bd_dmas_in_q--; 
}



/******************************************************************************
 **** INITIALISATION
 */

void init_usd_contexts (void)
{
    /* Set up the command space: 2k, the host gets at it through the 
     * 2k memory window. (which is one reason for insisting on host-side
     * TX ring) */
    bzero((U32 *)usd_cmd_p, 2048);

    /* Set up the user-safe command chain 
     * -- split mailboxes 16-31 off onto CPU B's event register */
    trp->gen_control.misc_config |= TG_MSC_SPLIT_MAILBOX_EVENTS;
}



/******************************************************************************
 * CONNECTION SETUP
 */

/*
 *  Add a new context to the world, using info <hp> from the host.
 *  We assume 'id' is valid already, i.e. it was checked before DMA.
 */
void init_new_usd_context(U16 id, conn_ctxt_host_part_t *hp)
{
    rx2_con_ctxt_t *c = &(nicfp->rx2_ctxt[id]);

    bzero((void *)c, sizeof(rx2_con_ctxt_t));

    c->host_range_base   = hp->host_range_base;
    c->host_range_length = hp->host_range_length;
      
    c->rx_h_mask         = hp->rx2_ring_size - 1;
    c->rx2_ring_ptr      = hp->rx2_ring_ptr;

    c->ipSourceAddr      = hp->ipSourceAddr;
    c->ipDestAddr        = hp->ipDestAddr;
    c->sourcePort        = hp->sourcePort;
    c->destPort          = hp->destPort;

    clear_bit(nicfp->rx2_interrupts_primed, id);
    clear_bit(tsp->gen_com.rx2_interrupts_active, id);
    
    NIC_UTRACE("USDstg2", 0xfe030,
	      (U32) id, c->rx2_ring_ptr, c->host_range_base,
               c->host_range_length);

    send_event_to_B_with_args(EVB_ADD_CONTEXT | id,
                              (U32*)hp, sizeof(*hp)>>2);
}


void h_usd_add_context_B(U16 id, conn_ctxt_host_part_t *hp) // Runs on CPU B
{
    tx2_con_ctxt_t *c  = &(nicbfp->tx2_ctxt[id]);

    bzero((void *)c, sizeof(tx2_con_ctxt_t));
    c->host_range_base   = hp->host_range_base;
    c->host_range_length = hp->host_range_length;
    c->tx_h_mask         = hp->tx2_ring_size - 1;
    c->tx2_consumer_ptr  = hp->tx2_csm_ptr;
    c->tx2_ring_ptr      = hp->tx2_ring_ptr;

    clear_bit( tsp->gen_com.tx2_interrupts_active, id );

    /* Subtle. We only update nicp->max_ctxt_id on CPU B, to avoid races. */
    if ( id > nicbfp->max_ctxt_id_b )
    {
        nicbfp->max_ctxt_id_b = id;
        nicp->max_ctxt_id     = id;
    }

#ifdef TX_SHAPER_SCHED
    c->credit_inc  = 10; //67;   // XXX 64KB (in KB) 
    c->time_inc    = 8; //4;    // XXX (in 256uS increments)

    c->credit_inc = 255;
    c->time_inc   = 0;

    //c->credit_inc  =  5; //138; //67;   // XXX 64KB (in KB) 
    //c->time_inc    = 2; //5; //4;    // XXX (in 256uS increments)

    c->credit      = c->credit_inc<<10;
    c->last_credit = trp->gen_control.timer;

    set_bit( nicbfp->tx2_bds_allowed, id );
#endif

    NIC_UTRACE("USDstg3", 0xfe030,
	      (U32) id, c->tx_h_mask, c->tx2_ring_ptr, c->tx2_consumer_ptr);

    send_event_to_A(EVA_ADD_CONTEXT_FINAL | id);
}

void h_usd_add_context_final(U16 id) // CPU A
{
    struct tg_event event;

    NIC_UTRACE("USDstg4", 0xfe030,
	      (U32) id, 0,0,0);

    /* CPU B may have changed max ctxt. */
    nicfp->max_ctxt_id_a = nicp->max_ctxt_id;

    /* All done - tell the host the good news. */
    event.TG_EVENT_w0 = (TG_EVENT_USD_CTXT_ADDED << TG_EVENT_EVENT_SHIFT) | id;
    must_enq_event(&event);
}



/******************************************************************************
 * CONNECTION TEARDOWN
 */

void delete_usd_context(U16 id) // CPU A
{
    NIC_UTRACE("USDDel1", 0xfe020, (U32)id,0,0,0);

    /*
     * We need the semaphore until we've disabled DMA, or someone else
     * may stooge in on CPU B and reenable it under our feet!
     */
    grab_spin_lock();

    /* One more teardown in progress... */
    nicp->ctxt_teardown_count++;

    trp->host_dma_assist.assist_state |= TG_DMA_ASSIST_PAUSE;
    free_spin_lock();

    /* Remove the IP demux filter. All outstanding packets will go to kernel */
    bzero((void *)&(nicfp->rx2_ctxt[id]), sizeof(rx2_con_ctxt_t));

    /*
     * Clear out any outstanding DMA requests -- in particular, spewing
     * data into host memory we no longer have pinned down would be bad!
     * 
     * Just clear CPU A rings here (hi pri). Low priority ones have events 
     * controlled by CPU B -- we should thus leave it to him to clear the 
     * assist ring. Otherwise the comp handler on B could fill a new entry 
     * for the id being deleted after we have cleared it out!
     */
    clear_assist_ring(tdp->wr_dma_hi_ring,
                      trp->host_dma_assist.write_chan_hi_pri_ref,
                      trp->host_dma_assist.write_chan_hi_pri_producer,
                      write_hi_cleanup,
                      id);
    clear_assist_ring(tdp->rd_dma_hi_ring, 
                      trp->host_dma_assist.read_chan_hi_pri_ref,
                      trp->host_dma_assist.read_chan_hi_pri_producer,
                      read_hi_cleanup,
                      id);

    clear_bit(nicfp->rx2_bds_unseen, id);
    clear_bit(nicfp->rx2_bds_pend, id);

    /* Blow away the receive-specific fields of the control structure. */
    usd2_ctrl_p[id].free_prod = 0;
    usd2_ctrl_p[id].rx_ref    = 0;
    usd2_ctrl_p[id].rx_prod   = 0;
    
    /* Kick CPU B to get it to do its stuff. */
    send_event_to_B(EVB_DEL_CONTEXT | id);
}


void h_usd_delete_context_B(U16 id) // CPU B
{
    /* Clear out the state. */
    bzero((void *)&(nicbfp->tx2_ctxt[id]), sizeof(tx2_con_ctxt_t));

    NIC_UTRACE("USDDel2", 0xfe030,
	      (U32)id,
               !!is_bit_set(nicbfp->tx2_bds_unseen,id),
               !!is_bit_set(nicbfp->tx2_bds_pend,id),
               !!is_bit_set(nicbfp->tx2_bds_rdy,id));

    /* Clear out the assist rings controlled by CPU B (low priority rings) */
    clear_assist_ring(tdp->wr_dma_lo_ring,
                      trp->host_dma_assist.write_chan_lo_pri_ref,
                      trp->host_dma_assist.write_chan_lo_pri_producer,
                      write_lo_cleanup,
                      id);
    clear_assist_ring(tdp->rd_dma_lo_ring,
                      trp->host_dma_assist.read_chan_lo_pri_ref,
                      trp->host_dma_assist.read_chan_lo_pri_producer,
                      read_lo_cleanup,
                      id);

    /* Wait for current DMA to complete -- it could be for id being deleted! */
    while ( trp->host_dma.dma_rd_state & TG_DMA_STATE_ACTIVE );
    while ( trp->host_dma.dma_wr_state & TG_DMA_STATE_ACTIVE );

    grab_spin_lock();
    if ( --nicp->ctxt_teardown_count == 0 )
    {
        /* We are the last teardown currently in progress - things to do: */

        /* 1. Reenable DMA. */
        trp->host_dma_assist.assist_state &= ~TG_DMA_ASSIST_PAUSE;
        free_spin_lock();

        /* 2. Reset the max_ctxt_id. */
        if ( id == nicbfp->max_ctxt_id_b )
        {
            int chan, max=0;
            for ( chan = 1; chan < nicbfp->max_ctxt_id_b; chan++ )
            {
                if ( nicbfp->tx2_ctxt[chan].tx2_ring_ptr ) max = chan;
            }
            nicbfp->max_ctxt_id_b = max;
            nicp->max_ctxt_id     = max;
        }
    }
    else
    {
        free_spin_lock();
    }

    clear_bit(nicbfp->tx2_bds_unseen,id);
    clear_bit(nicbfp->tx2_bds_pend,id);
    clear_bit(nicbfp->tx2_bds_rdy,id);

    /* Clear the shared comms region, to prevent mistaken polls in {r|t}x2.c */
    bzero((U32*)(usd2_ctrl_p + id), sizeof(usd2_ctrl_t));

    /* CPU A has to kick the host, as event state is kept in its scratchpad. */
    send_event_to_A(EVA_DEL_CONTEXT_FINAL | id);
}


void delete_usd_context_final(U16 id) // CPU A
{
    struct tg_event event;

    nicfp->max_ctxt_id_a = nicp->max_ctxt_id;

    /* Host can now reuse this channel. */
    event.TG_EVENT_w0 = 
        (TG_EVENT_USD_CTXT_DELETED << TG_EVENT_EVENT_SHIFT) | id;
    must_enq_event(&event);
}








/*XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX*/
/*XXX KAF -- to be moved host side XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX*/
#if 0
void shaper_calc( int enable, int KBps, int burst_KB, 
		  U8 *credit_inc, U8 *time_inc )
{
  // bandwidth in KB/s
  // max burst size in KB

  if( !enable )
    {
      *credit_inc = 255;
      *time_inc   = 0;    // effectively unlimited bandwidth (could be 4)
    }

  else
    {
      U32 rough_bursts_per_sec = KBps / burst_KB;
      
      U32 t_inc = 3906 / rough_bursts_per_sec;

      U32 bursts_per_sec = t_inc * 3096;      

      U32 c_inc = (KBs + (bursts_per_sec>>1) ) / bursts_per_sec;

      if( (c_inc > 255) || (t_inc > 255) || (c_inc == 0) || (t_inc == 0) )
	{
	  NIC_UTRACE("ShapCalc", 0x2234,
		     KBs, burst_KB, t_inc, c_inc );

	  PANIC();
	}

      *credit_inc = c_inc;
      *time_inc   = t_inc;
    }
}
#endif
