/*
 * COPYRIGHT NOTICE
 * Copyright (c) Alteon Networks, Inc. 1996 
 * All rights reserved
 */
/*
 * HISTORY
 * $Log: mii.c,v $
 * Revision 1.4  1999/06/22 12:43:01  tjd21
 * Before removing extra recv rings
 *
 * Revision 1.1  1999/06/07 10:53:09  tjd21
 * Start of archive
 *
 * Revision 1.1.2.9  1998/05/27  01:19:32  fhwang
 * 	remove 100 and FD when enable autonegotiate mode, and add restart_auto mode
 * 	[1998/05/27  00:55:28  fhwang]
 *
 * Revision 1.1.2.8  1998/01/03  02:16:45  kyung
 * 	fast enet board support - added by J.H.
 * 	[1998/01/03  02:16:30  kyung]
 * 
 * Revision 1.1.2.7  1997/10/21  21:59:37  tibor
 * 	converted macros to subr
 * 	[1997/10/21  21:57:17  tibor]
 * 
 * Revision 1.1.2.6  1997/10/10  20:38:05  hayes
 * 	cleaned up tracing
 * 	[1997/10/09  21:39:35  hayes]
 * 
 * Revision 1.1.2.5  1997/10/06  23:53:34  hayes
 * 	more adjustments to ICS 1890 initialization
 * 	[1997/10/06  23:53:17  hayes]
 * 
 * Revision 1.1.2.4  1997/09/17  03:26:30  hayes
 * 	Reset override register after using it.
 * 	Changed trace strategy.
 * 	[1997/09/15  22:23:20  hayes]
 * 
 * Revision 1.1.2.3  1997/09/09  02:57:33  hayes
 * 	Add support for various MII devices
 * 	Add device support for Level One LXT 970 in
 * 	addition to ICS 1890
 * 	[1997/09/09  02:57:13  hayes]
 * 
 * 	completed link code restructuring
 * 	[1997/08/29  22:10:59  hayes]
 * 
 * Revision 1.1.2.2  1997/08/29  22:11:56  hayes
 * 	completed link code restructuring
 * 
 * Revision 1.1.2.1  1997/08/26  21:08:59  hayes
 * 	initial checkin
 * 	[1997/08/26  21:01:29  hayes]
 * 
 * $EndLog$
 */

#include "link.h"
#include "link_op.h"
#include "mii.h"
#include "zconf.h"
#include "np.h"
#include "mac.h"
#include "timer.h"
#include "trace.h"

#if defined(_SP_CONF_H_)
#include "sp_regs.h"
#endif /* defined(_SP_CONF_H_) */

/*
 * phy reset state variables
 */
U32 mii_reset_state = RESET_PHY_MII_RESET;
U32 mii_reset_timer;

U32 mii_device_attached;

mii_pair_t mii_set[4][4] = {    /* duplex, speed */
    {           /* no duplex - error */
        {0, 0}, /* x, x */
        {0, 0}, /* x, 10 */
        {0, 0}, /* x, 100 */
        {0, 0}  /* x, 10/100 */
    },
    {   /* full duplex */
        {0, 0},         /* FD, x */
        {MII_CONTROL_SPEED_10 | MII_CONTROL_FULL_DUPLEX, 0}, /* FD, 10 */
        {MII_CONTROL_SPEED_100 | MII_CONTROL_FULL_DUPLEX, 0},/* FD, 100 */
        {MII_CONTROL_AUTONEG_ENA | MII_CONTROL_RESTART_AUTONEG,
         MII_AUTONEG_ADV_10_FD | MII_AUTONEG_ADV_100_FD}/* FD, 10/100 */
    },
    {   /* half duplex */
        {0, 0},          /* HD, x */
        {MII_CONTROL_SPEED_10 | MII_CONTROL_HALF_DUPLEX, 0}, /* HD, 10 */
        {MII_CONTROL_SPEED_100 | MII_CONTROL_HALF_DUPLEX, 0},/* HD, 100 */
        {MII_CONTROL_AUTONEG_ENA | MII_CONTROL_RESTART_AUTONEG,
         MII_AUTONEG_ADV_10_HD | MII_AUTONEG_ADV_100_HD} /* HD, 10/100 */
    },
    {   /* full/half duplex */
        {0, 0},         /* FD/HD, x */
        {MII_CONTROL_AUTONEG_ENA | MII_CONTROL_RESTART_AUTONEG,
         MII_AUTONEG_ADV_10_FD | MII_AUTONEG_ADV_10_HD},/* FD/HD, 10 */
        {MII_CONTROL_AUTONEG_ENA | MII_CONTROL_RESTART_AUTONEG,
         MII_AUTONEG_ADV_100_FD | MII_AUTONEG_ADV_100_HD},/* FD/HD, 100 */
        {MII_CONTROL_AUTONEG_ENA | MII_CONTROL_RESTART_AUTONEG,
         MII_AUTONEG_ADV_10_FD | MII_AUTONEG_ADV_100_FD |
         MII_AUTONEG_ADV_10_HD | MII_AUTONEG_ADV_100_HD}/* FD/HD, 10/100 */
    }
};
 

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/*
 * Function: read_mii_mgt
 *
 * Params:
 *         U32 device   -
 *         U32 reg      -
 *         U16 *data    -
 *
 * Return: None
 *
 * Side effects:
 *
 * Description:
 *
 * MII mgt frame format
 *
 * preamble - 32 bits of 1
 * start of frame - 01
 * operation - 10 (read), 01 (write)
 * phyaddr - which mac entity to talk to - 5 bits.
 * register addr - which MII reg to talk to - 5 bits.
 * ta - turn around - 2 bits
 * data - 16 bits
 */
 
void
read_mii_mgt(U32 device, U32 reg, U16 *data)
{
    U32 i, bit;
 
    *data = 0;
 
    /*  preamble */
    for (i=0; i<32; i++) {
        write_mii_bit(1);
    }
 
    /* SOF */
    write_mii_bit(0);
    write_mii_bit(1);
 
    /* operation - read */
    write_mii_bit(1);
    write_mii_bit(0);
 
    /* phyaddr */
    for (i=0x10; i; i>>=1) {
        bit = device & i;
        if (bit) {
            write_mii_bit(1);
        } else {
            write_mii_bit(0);
        }
    }
 
    /* register addr */
    for (i=0x10; i; i>>=1) {
        bit = reg & i;
        if (bit) {
            write_mii_bit(1);
        } else {
            write_mii_bit(0);
        }
    }
 
    /* TA */
    write_mii_z();
    write_mii_z();
 
    /* read reg */
    for (i=0x8000; i; i>>=1) {
        if (read_mii_bit()) {
            *data |= i;
        }
    }
    write_mii_z();
    return;
}
 
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/*
 * Function: write_mii_mgt
 *
 * Params:
 *         U32 device   -
 *         U32 reg      -
 *         U16 *data    -
 *
 * Return: None
 *
 * Side effects:
 *
 * Description:
 *
 */
void
write_mii_mgt(U32 device, U32 reg, U16 *data)
{
U32 i;
U32 bit;
 
    /*  preamble */
    for (i=0; i<32; i++) {
        write_mii_bit(1);
    }
 
    /* SOF */
    write_mii_bit(0);
    write_mii_bit(1);
 
    /* operation - write */
    write_mii_bit(0);
    write_mii_bit(1);
 
    /* phyaddr */
    for (i=0x10; i; i>>=1) {
        bit = device & i;
        if (bit) {
            write_mii_bit(1);
        } else {
            write_mii_bit(0);
        }
    }
 
    /* register addr */
    for (i=0x10; i; i>>=1) {
        bit = reg & i;
        if (bit) {
            write_mii_bit(1);
        } else {
            write_mii_bit(0);
        }
    }
 
    /* TA */
    write_mii_bit(1);
    write_mii_bit(0);
 
    /* data */
    for (i=0x8000; i; i>>=1) {
        bit = *data & i;
        if (bit) {
            write_mii_bit(1);
        } else {
            write_mii_bit(0);
        }
    }
    write_mii_z();
    return;
 
}

/* need to save across invocations... */
U32 mii_reset_state_last = 0;

void
reset_mii_phy(void)
{
    U16 mii_data;

    if (mii_reset_state != mii_reset_state_last) {
	LINK_TRACE(TRACE_TYPE_LNK_PHY, "rst_mii", 0xd0101, 
	    mii_reset_state, 0, 0, 0);
	mii_reset_state_last = mii_reset_state;
    }

    switch (mii_reset_state) {
        case RESET_PHY_MII_RESET:

            write_mii_z();
            mii_data = MII_CONTROL_RESET;
            write_mii_mgt(1, MII_REG_CONTROL, &mii_data);
            mii_reset_state = RESET_PHY_MII_RESET_WAIT;
            break;
 
        case RESET_PHY_MII_RESET_WAIT:
            /* wait for the reset bit to clear */
            read_mii_mgt(1, MII_REG_CONTROL, &mii_data);
            if (!(mii_data & MII_CONTROL_RESET))
                mii_reset_state = RESET_PHY_MII_REG_CONTROL;
            break;
 
        case RESET_PHY_MII_REG_CONTROL:
            /*
             * set control register to enable autonegotiation,
             */

	    /* start auto negotiation */
	    mii_data = MII_CONTROL_RESTART_AUTONEG | MII_CONTROL_AUTONEG_ENA;
	    write_mii_mgt(1, MII_REG_CONTROL, &mii_data);


            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_REG_CONTROL_WAIT;
            break;
 
        case RESET_PHY_MII_REG_CONTROL_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)

		switch (mii_device_attached) {
		    case MII_DEVICE_ICS_1890_2:
			mii_reset_state = RESET_PHY_MII_ICS_REG18;
			break;

		    case MII_DEVICE_LXT_970:
			mii_reset_state = RESET_PHY_MII_LEVL1_REG19;
			break;

		    case MII_DEVICE_UNKNOWN:
			mii_reset_state = RESET_PHY_MII_READY;
			break;

		    case MII_DEVICE_NONE:
		    default:
			/* XXX */
			mii_reset_state = RESET_PHY_MII_READY;
		}
            break;

	/*
	 * XXX The following states are specific to the ICS 1890
	 * device
	 */
        case RESET_PHY_MII_ICS_REG18:
            /*
             * XXX this is ICS 1890 specific
             * set the 10Base-T operations register to
             * inhibit jabber isolation, disable loopback,
             * and disable SQE test
             */
            mii_data = 0x0014;
            write_mii_mgt(1, 18, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG18_WAIT;
            break;
 
        case RESET_PHY_MII_ICS_REG18_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_ICS_REG19;
            break;
 
        case RESET_PHY_MII_ICS_REG19:
            /*
             * XXX this is ICS 1890 specific
             * set the extended control register to
             * autonegotiate with qualified 10Base-T data and
             * protect 10Base-T data from 100Base-TX data.
             */
            mii_data = 0x0006;
            write_mii_mgt(1, 19, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG19_WAIT;
            break;
 
        case RESET_PHY_MII_ICS_REG19_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_ICS_REG24;
            break;
 
        case RESET_PHY_MII_ICS_REG24:
	    /*
	     * There is a bug in the ICS 1890 where a test mux is left enabled.
	     * Disabling this will result in a savings of ~20ma in power.
	     * To disable, pin 20 must be grounded (this will be in switch rev d
	     * and later implementations) and then bit 0 of register 24
	     * (register overwrite bit) must be set, followed by setting
	     * bits 8, 9, and 10 of register 23.  Do not disturb the other
	     * bits in the registers.
	     */
            read_mii_mgt(1, 24, &mii_data);
            mii_data |= 0x0001;        /* set bit 0 */
            write_mii_mgt(1, 24, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG24_WAIT;
            break;
 
        case RESET_PHY_MII_ICS_REG24_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_ICS_REG23;
            break;
 
        case RESET_PHY_MII_ICS_REG23:
            read_mii_mgt(1, 23, &mii_data);
            mii_data |= 0x0700; /* set bits 8, 9 and 10 */
            write_mii_mgt(1, 23, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG23_WAIT;
            break;

        case RESET_PHY_MII_ICS_REG23_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_ICS_REG20;
            break;

	/*
	 * the following sets bit 4 in register 20, which allows
	 * the ICS part to enable the acceptance of undelimited
	 * packets.  See ICS 1890 PR1A and PR1B release note, 
	 * July 8, 1997 for more details
	 */
        case RESET_PHY_MII_ICS_REG20:
            read_mii_mgt(1, 20, &mii_data);
            mii_data |= 0x0010; /* set bit 4 */
            write_mii_mgt(1, 20, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG20_WAIT;
            break;
 
        case RESET_PHY_MII_ICS_REG20_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_ICS_REG24_2;
            break;

        case RESET_PHY_MII_ICS_REG24_2:
	    /*
	     * disable register overwrite bit...
	     */
            read_mii_mgt(1, 24, &mii_data);
            mii_data &= ~0x0001;        /* clear bit 0 */
            write_mii_mgt(1, 24, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_ICS_REG24_2_WAIT;
            break;
 
        case RESET_PHY_MII_ICS_REG24_2_WAIT:
            /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_READY;
            break;
 
	/*
	 * XXX End of ICS 1890 device specific initialization
	 */

	/*
	 * XXX Start of Level 1 LXT 970 device specific initialization
	 */

        case RESET_PHY_MII_LEVL1_REG19:
	    /*
	     * Set the the LXT 970 configuration register for normal operation,
	     * DTE mode, MDIO interrupt disabled, loopback disabled, SQE
	     * disabled, jabber enabled, LEDC off, normal TX clk cycle,
	     * decode, enable 100TP interface
	     */
            mii_data = 0x0840;
            write_mii_mgt(1, 19, &mii_data);
            mii_reset_timer = trp->gen_control.timer;
            mii_reset_state = RESET_PHY_MII_LEVL1_REG19_WAIT;
            break;
 
        case RESET_PHY_MII_LEVL1_REG19_WAIT: 
           /* wait 100uS */
            if (((mii_reset_timer + MII_TIMER) -
                trp->gen_control.timer) >= MII_TIMER)
                mii_reset_state = RESET_PHY_MII_READY;
            break;
	/*
	 * XXX End of Level 1 LXT 970 device specific initialization
	 */

        case RESET_PHY_MII_READY:
	    break;

        default:
    }
}
 

/*********************************************************************************
*
* Function:	read_mii_bit
*
* Purpose:	receive bit from phy
*		The MDC (managment data clock) is defined in section 22.2.2.11 of
*		the 802.3u spec.
*
* Parms:	none
*
* Error:	none
*
* Returns:	bit
*
*********************************************************************************/
U32
read_mii_bit(void)
{
    U32 bit;
    
    /* clock up */ 
    bit = TRP->gen_control.misc_local_control & TG_MLC_MII_DATA_IN;
    TRP->gen_control.misc_local_control |= TG_MLC_MII_CLK_OUT; 
    WAIT_US(1);
    /* clock down */
    TRP->gen_control.misc_local_control &= ~TG_MLC_MII_CLK_OUT;
    WAIT_US(1);

    return bit;
}


/*********************************************************************************
*
* Function:	write_mii_bit
*
* Purpose:	transmit bit from mii to phy
*		The MDC (managment data clock) is defined in section 22.2.2.11 of
*		the 802.3u spec.
*
* Parms:	bit - bit to send
*
* Error:	none
*
* Returns:	nothing
*
*********************************************************************************/
void
write_mii_bit(U32 bit)
{
    /* data */ 
    TRP->gen_control.misc_local_control = 
	(TRP->gen_control.misc_local_control & ~TG_MLC_MII_MASK) | 
	TG_MLC_MII_DATA_OUT_ENA | 
	(bit << 18);
    /* clock up */ 
    TRP->gen_control.misc_local_control |= TG_MLC_MII_CLK_OUT;
    WAIT_US(1);
    /* clock down */ 
    TRP->gen_control.misc_local_control &= ~TG_MLC_MII_CLK_OUT;
    WAIT_US(1);
}


/*********************************************************************************
*
* Function:	write_mii_z
*
* Purpose:	transmit 0 bit from mii to phy
*		The MDC (managment data clock) is defined in section 22.2.2.11 of
*		the 802.3u spec.
*
* Parms:	none
*
* Error:	none
*
* Returns:	nothing
*
*********************************************************************************/
void
write_mii_z(void)
{
    /* drop out enable */ 
    TRP->gen_control.misc_local_control = 
	(TRP->gen_control.misc_local_control & ~TG_MLC_MII_MASK);
    /* clock up */ 
    TRP->gen_control.misc_local_control |= TG_MLC_MII_CLK_OUT; 
    WAIT_US(1); 
    /* clock down */ 
    TRP->gen_control.misc_local_control &= ~TG_MLC_MII_CLK_OUT; 
    WAIT_US(1);
}
