/******************************************************************************
 * tcp_signals.c
 * 
 * Main loop for the bottom half thread.
 * 
 * Copyright (c) 1999-2000, K A Fraser
 * 
 * $Id: tcp_signals.c,v 3.2 1999/12/18 16:27:38 kaf24 Exp kaf24 $
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include "thread.h"
#include <asm/types.h>
#include "hash.h"
#include "private.h"
#include "lowlevel.h"
#include "libusernet.h"
#undef printf
#undef fprintf

pthread_t callback_thread;

/******************************************************************************
 * upcall_worker:
 *   Start-point of the upcall worker thread. Sits in a loop, waiting for
 *   callback events, and doing the necessary processing.
 */
static void *upcall_worker(void *arg)
{
    struct user_pcb *pcb;
    sigset_t         sigs;
    pth_time_t       timeout;
    pth_event_t      ev;
    u32              work_type;
    static int count = 0;

    DB("entered");

    /* Set up our signal callback */
    usd_init_callback();

    /*
     * Allows our parent thread to kill us even if we are blocked in the
     * kernel waiting for work to do.
     */
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
    gettimeofday(&timeout, NULL);
    timeout.tv_usec += COARSE_TIMEOUT_PERIOD*1000;
    ev = pth_event(PTH_EVENT_TIME, timeout);

    /* Signal back to our creator that things are all set up. */
    pthread_mutex_lock(&global_poll_mutex);
    pthread_cond_signal(&global_poll_cond);
    pthread_mutex_unlock(&global_poll_mutex);

    for ( ; ; )
    {
        pth_sigwait_ev(&sigs, NULL, ev);

        if ( pth_event_occurred(ev) )
        {
#if 0
            if ( count++ == 150 )
            {
                int j;
                count = 0;
                usd_dump_stats();
                for ( j = 0; j < 10; j++ )
                {
                    struct user_pcb *p;
                    if ( (p = element_for_key(pcbs, j)) )
                    {
                        tcp_dump_stats(p->sk);
                        printf("phys_tx = %d\n",
                               p->tx_queued_skbuffs.qlen);
                    }
                }
            }
#endif
            process_tcp_timers();
            gettimeofday(&timeout, NULL);
            timeout.tv_usec += COARSE_TIMEOUT_PERIOD*1000;
            pth_event_free(ev, PTH_FREE_THIS);
            ev = pth_event(PTH_EVENT_TIME, timeout);
        }

        while ( (pcb = usd_get_next_connection_with_work(&work_type)) )
        {
            /******************************************************************
             * CODE FOR TRANSMIT EVENTS -- UDP only!!
             */
            if ( work_type & 2 ) 
            {
                usd_clear_tx_event(pcb->usd_conn);
                bh_lock_sock(pcb->sk);
                if ( pcb->usd_conn ) free_tx_skbs(pcb);
                bh_unlock_sock(pcb->sk);
                sched_yield();
            }

            /******************************************************************
             * CODE FOR RECEIVE EVENTS -- TCP & UDP.
             */
            if ( work_type & 1 ) 
            {
                if ( rx_skb(pcb) == 0 )
                {
                    /*
                     * We don't yield on every packet, but attempt to bundle
                     * things a little.
                     */
                    if ( ++pcb->recvs_since_upcall == PACKETS_PER_YIELD )
                    {
                        pcb->recvs_since_upcall = 0;
                        sched_yield();
                    }
                }
                else
                {
                    /*
                     * We will almost certainly block now, so try doing a
                     * reschedule: the extra time this uses may be enough for
                     * some new data to arrive. In that case we avoid blocking!
                     */
                    if ( pcb->recvs_since_upcall )
                    {
                        pcb->recvs_since_upcall = 0;
                        sched_yield();
                    }

                    /* Pessimistically clear the event. */
                    usd_clear_rx_event(pcb->usd_conn);

                    if ( pcb->usd_conn )
                    {
                        if ( rx_skb(pcb) == 0 )
                        {
                            /*
                             * Wahoo! More work has arrived, so we'd better set
                             * the event bit again...
                             */
                            usd_set_rx_event(pcb->usd_conn);
                        }
                        else
                        {
                            /* :-(( No more work: request event. */
                            usd_rx_req_callback(
                                pcb->usd_conn,
                                usd_free_buffers_in_rx_queue(pcb->usd_conn)/4);
                        }
                    }
                } 
            }
        }
    }
}


/******************************************************************************
 * Externally called functions
 */

/******************************************************************************
 * init_upcalls:
 *   Initialises the upcall worker thread which deals with callback events.
 */
int init_upcalls()
{
    /* We create a worker thread which will perform all protocol processing. */
    if ( pthread_create(&callback_thread, NULL, upcall_worker, NULL) )
    {
        DB_ERR("couldn't create worker thread: %s", strerror(errno));
        return(0);
    }

    /* Wait for new thread to finish initialisation. */
    pthread_mutex_lock(&global_poll_mutex);
    pthread_cond_wait(&global_poll_cond, &global_poll_mutex);
    pthread_mutex_unlock(&global_poll_mutex);

    DB("created callback thread (pid=%d)", callback_thread);
    return(1);
}


/******************************************************************************
 * kill_upcalls:
 *   Kills the upcall worker thread on stack termination.
 */
void kill_upcalls(void)
{
    DB("killing the callback thread");
    pthread_cancel(callback_thread);
    pthread_join(callback_thread, NULL);
    DB("thread dead -- leaving");
}

