/*
 * All modifications are copyright (c) 2000, K A Fraser.
 * 
 * Please note the original copyright and license details below.
 */

/*
 * Copyright (C) 1997 Massachusetts Institute of Technology 
 *
 * This software is being provided by the copyright holders under the
 * following license. By obtaining, using and/or copying this software,
 * you agree that you have read, understood, and will comply with the
 * following terms and conditions:
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose and without fee or royalty is
 * hereby granted, provided that the full text of this NOTICE appears on
 * ALL copies of the software and documentation or portions thereof,
 * including modifications, that you make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. COPYRIGHT
 * HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE OR
 * DOCUMENTATION.
 *
 * The name and trademarks of copyright holders may NOT be used in
 * advertising or publicity pertaining to the software without specific,
 * written prior permission. Title to copyright in this software and any
 * associated documentation will at all times remain with copyright
 * holders. See the file AUTHORS which should have accompanied this software
 * for a list of all copyright holders.
 *
 * This file may be derived from previously copyrighted software. This
 * copyright applies only to those changes made by the copyright
 * holders listed in the AUTHORS file. The rest of this file is covered by
 * the copyright notices, if any, listed below.
 */

#ifndef __DPF_INTERNAL_H__
#define __DPF_INTERNAL_H__
/*
 *  DPF internals.   For the moment, we emphasize simplicity by wasting
 *  space. It requires no brains to eliminate this, only attention to detail.
 *
 * SHould have both in-place and regen options.  E.g., after getting
 * to a certain size, keep the code associated with each atom rather
 * than copying.  (Hopefully, this is not too much of a big deal, since
 * will mostly be dealing with the same protocol families.)
 *
 * Make the hash table be uniform.
 * Add deq opcode.
 * Make pptr a ptr to a ptr.
 */
#include "defs.h"
#include "dpf.h"

/*
 * Site-specific macros.  These may be redefined.
 */
#ifndef DPF_MINMSG
#	define DPF_MINMSG     64      /* Minimum message size (in bytes). */
#endif

#ifndef DPF_MSG_ALIGN
#	define DPF_MSG_ALIGN 8        /* Message alignment (4 bytes).  */
#endif

/* If this is larger than 64k, need to increase refcnt and pid in atom. */
#define DPF_MAXFILT		256 /* also in ash_ae_net.h */
#define DPF_MAXELEM		256

/* Note, we make eq's and meq's -> meq's and shifts and mshifts -> shifts. */
enum {
	DPF_OP_EQ = 1,
#	define dpf_iseq(x) ((x) == DPF_OP_EQ)
	DPF_OP_SHIFT,
#	define dpf_isshift(x) ((x) == DPF_OP_SHIFT)
        DPF_OP_SHIFTI,
#	define dpf_isshifti(x) ((x) == DPF_OP_SHIFTI)

#	define dpf_islegalop(x) (((unsigned)(op)) < DPF_OP_END)
	DPF_OP_END			/* mark end */
};

/* 
 * (*DANGER*): we rely on op, offset, nbits and mask being located in 
 * the same place in struct eq and struct shift.
 */

/* size of the load is captured in the opcode. */
struct eq {
    uint8	    op;	    /* opcode */	
    uint16	    offset; /* message offset */
    uint8	    nbits;  /* length (in bits) */
    uint32	    mask;   /* maximum width */
    uint32	    val;    /* maximum size value */
};

/* size of the shift is captured in the opcode */
struct shift {
    uint8	    op;	    /* opcode */
    uint16 	    offset;  
    uint8	    nbits;  /* length (in bits) */
    uint32	    mask;
    uint8	    shift;
    uint8	    align;  /* alignement given by shift */
    uint8           ext;    /* KAF: do we inc hdr and/or pkt len counts? */
};

struct shifti {
    uint8           op;
    uint16          shift;  /* absolute shift amount */
    uint8           align;
    uint8           ext;
};

union ir {
    struct eq eq;
    struct shift shift;
    struct shifti shifti;
};

/* container for code. */
struct dpf_ir {
	uint32  version;	/* ir may change from version to version. */
	uint16	irn;
        uint32  moffset;
	union ir ir[DPF_MAXELEM+1];		/* pointer to code */
};


/* Hash table */
typedef struct ht {
	uint16 htsz;	/* size of the hash table. */
	uint16 ent;	/* number of unique buckets. */
	uint16 coll;	/* number of collisions. */
	uint16 nterms;	/* number of non-terminals. */

#	define DPF_INITIAL_HTSZ	8
	struct atom *head;	/* ptr to list of hash table entries. */
	struct atom *ht[DPF_INITIAL_HTSZ]; 	/* struct hack */
} *Ht;

/* 
 * Could squeeze a lot of space by aggregating straight-line atoms or
 * by using segments for the pointers.  Size of node:
 *	1. with 64 bit ptrs = 5 * 8 + 4 + 2 + 2 + 3 * 4 + 4 = 64 bytes.
 *	2. with 32 bit ptrs = 5 * 4 + 4 + 2 + 2 + 3 * 4 + 4 = 44 bytes.
 * Uch.
 */
typedef struct atom {
	/* The four pointers that situate us in the filter trie. */
	struct atom *parent;	/* parent of this node (if any) */

	/* or and orp are overloaded for hash tables. */
	struct atom **orp;	/* (left) or ptr */
        struct atom *or;        /* (right) non-nil if there is an or-branch */

	struct atom *child;	/* child of this node. */
	
	/* Non-nil if we are a disjunction node. */
	struct ht *ht;		/* pointer to hash table (if any) */

	/* If you want to do more than 64k filters, redefine these. */
	uint16	refcnt;		/* number of filters dependent on this node. */
	uint16	pid;		/* process id of owner. */

	/* 
	 * Maximum offset into message of code that we have to take
	 * if this node succeeds.  Basically, for each or and shift
	 * maxoffset holds the largest message offset of all nodes until
	 * the next or, shift or end-of-filter.  This allows us to aggregate 
	 * message checks rather than having to do them on every atom.
	 *
	 * Computed during insertion.
	 */
	uint16  maxoffset;	

	/* Used by hash table to perform indirect jump to child (if any). */
        void *label;

	union ir ir;		/* the atom at this level. */
} *Atom;

extern Atom dpf_active[DPF_MAXFILT];

extern Atom dpf_base;

/* dump trie. */
#ifndef __KERNEL__
void dpf_output(void);
#else
#define dpf_output() ((void)0)
#endif

Atom ht_lookup(Ht p, uint32 val);

/* Coalesce adjacent atoms. */
void dpf_coalesce(struct dpf_ir *irs);

/* Compute alignement given by ir + alignment. */
int dpf_compute_shift_alignment(union ir *ir, int alignment);
int dpf_compute_shifti_alignment(union ir *ir, int alignment);

void gen_ret_zero(void);

int (*dpf_compile(Atom trie))();

#endif /* __DPF_INTERNAL_H__ */
