/******************************************************************************
 * af_user.c
 * 
 * Implementation of socket family for raw networking to user-space.
 * 
 * Copyright (c) 1999-2000, K A Fraser (kaf24@cl.cam.ac.uk)
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/bigphysarea.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/arp.h>
#include <net/route.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <net/raw.h>
#include <net/icmp.h>
#include <net/ipip.h>
#include <net/inet_common.h>

#include <asm/io.h>
#include "acenic_usd.h"
#include "af_user.h"


/******************************************************************************
 * Debugging macros.
 */

#ifdef DEBUG
#define ASSERT(__x)                                        \
    if(!(__x)) {                                           \
        printk(__FUNCTION__": assertion "#__x" failed\n"); \
        cli(); for ( ;; ); }
#define TRC(__x) __x
#else
#define ASSERT(__x)
#define TRC(__x)
#endif


/******************************************************************************
 * Time to wait for connection setup/teardown callbacks.
 */

#define SETUP_CALLBACK_WAIT_IN_SECS  5


/******************************************************************************
 * Private functions.
 */

static int  get_buffers(void *kern_addr, int len, unsigned long *user_addr);
static void usd_setup_callback(void *argument, int usd_id);
static void usd_rx_callback(void *argument);
static void usd_tx_callback(void *argument);
static void usd_teardown_callback(void *argument);
static void dummy_state_change(struct sock *sk);
static void dummy_write_space(struct sock *sk);
static void dummy_error_report(struct sock *sk);
static void dummy_data_ready(struct sock *sk, int len);
static int  get_route_info(struct eth_ipv4_route_req_rsp *);


/******************************************************************************
 * Functions that can be performed on our socket family.
 */

static int user_create(struct socket *, int);
static int user_release(struct socket *);
static int user_bind(struct socket *, struct sockaddr *, int);
static int user_connect(struct socket *, struct sockaddr *, int, int);
static int user_socketpair(struct socket *, struct socket *);
static int user_getname(struct socket *, struct sockaddr *, int *, int);
static int user_ioctl(struct socket *, unsigned int, unsigned long);
static int user_listen(struct socket *, int);


/******************************************************************************
 * Function tables for the kernel.
 */

/* Operation table for standard user-space connections. */
struct proto_ops user_proto_ops  = 
{
    AF_USER,
    
    user_release,            /* (release) a socket                  */
    user_bind,               /* (bind) a socket to an address       */
    user_connect,            /* (connect) a socket to another       */
    user_socketpair,         /* create a (sockpair)                 */
    sock_no_accept,          /* (accept) a connection on a socket   */
    user_getname,            /* (getname) returns where we're going */ 
    sock_no_poll,            /* (poll) on this sort of socket       */
    user_ioctl,              /* perform an (ioctl) on this socket   */
    user_listen,             /* (listen) for new connections        */
    sock_no_shutdown,        /* (shutdown) the connection           */
    sock_no_setsockopt,      /* (setsockopt) writes options         */
    sock_no_getsockopt,      /* (getsockopt) reads options          */
    sock_no_fcntl,           /* perform (fcntl)s on the socket      */
    sock_no_sendmsg,         /* (sendmsg) down socket               */
    (void *)sock_no_recvmsg, /* (recvmsg) from socket               */
    NULL
};

/* Operation table for control sockets. */
struct proto_ops user_ctrl_proto_ops;


struct net_proto_family user_net_proto_family =
{
    AF_USER, user_create, 0, 0, 0
};


/******************************************************************************
 * Allow processes to map kernel timer structures into their address space.
 */
static void timermap_init(void);
static int timermap_read(char *, char **, off_t, int);
static void timermap_close(void);

/******************************************************************************
 * We can output timestamp information for callbacks.
 */
#undef TEST_PROC
#ifdef TEST_PROC
static u16 *test_buffer;
static u16 *test_curr;
#define TEST_BUFLEN 256000
static struct sock *socklist[8] = { 0 }; static int numsocks = 0;
/*#define STAMP(id)                                                 \
    { unsigned long __t; __asm__("rdtsc" : "=a" (__t) : : "edx"); \
    *timestamp_curr++ = (id); *timestamp_curr++ = __t;            \
    if ( timestamp_curr-timestamp_buffer >= 256*256 ) timestamp_curr -= 2; }*/
static void test_init(void);
static int test_read(char *, char **, off_t, int);
static void test_close(void);
static struct proc_dir_entry test_file = {
    0, 11, "afuser_test", S_IFREG | S_IRUGO, 1, 0, 0, 80, NULL,
    test_read, NULL };
#endif
#define STAMP(id)


/******************************************************************************
 * Pretty-print network addresses
 */
#define PRINT_IP_AND_PORT(ip,port) \
TRC(printk("%d.%d.%d.%d:%d\n", (ip)&0xff, ((ip)>>8)&0xff, \
           ((ip)>>16)&0xff, ((ip)>>24)&0xff, ntohs(port)));
#define PRINT_IP(ip) \
TRC(printk("%d.%d.%d.%d\n", (ip) & 0xff, ((ip)>>8) & 0xff, \
           ((ip)>>16) & 0xff, ((ip)>>24) & 0xff));
#define PRINT_ETHER(eth) \
TRC(printk("%2x:%2x:%2x:%2x:%2x:%2x\n", \
           (eth)[0], (eth)[1], (eth)[2], (eth)[3], (eth)[4], (eth)[5]));


/******************************************************************************
 * --- APPLICATION-ACCESSIBLE FUNCTION DEFINITIONS ---
 */

/******************************************************************************
 * user_proto_init:
 *   Called by socket.c to initialise our socket family.
 */
void __init user_proto_init(struct net_proto *pro)
{
    /*extern __u32 sysctl_wmem_max;*/
    /*extern __u32 sysctl_rmem_max;*/

    printk("Raw networking to user space ("__DATE__" "__TIME__").\n");

    memcpy(&user_ctrl_proto_ops, &inet_dgram_ops, sizeof(struct proto_ops));
    user_ctrl_proto_ops.ioctl = user_ioctl;

#ifdef TEST_PROC
    test_init();
#endif
    timermap_init();
    sock_register(&user_net_proto_family);
}


/******************************************************************************
 * user_create:
 *   Create a new socket of the appropriate type.
 */
static int user_create(struct socket *sock, int protocol)
{
    struct sock    *sk;
    int             err = 0;
    extern struct net_proto_family inet_family_ops;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(sock != NULL);

    if ( sock->type == SOCK_RAW )
    {
        /*
         * XXX KAF: This is a big black security hole, but how else to
         * give raw access to a user-space stack for its control socket?
         * 
         * Hopefully we'll be defining a nice interface betwen kernel and US
         * which will avoid this nastiness...
         */
        int lower = !cap_raised(current->cap_effective, CAP_NET_RAW);
        cap_raise(current->cap_effective, CAP_NET_RAW);
        err = inet_family_ops.create(sock, protocol);
        sock->ops = &user_ctrl_proto_ops;
        if ( lower ) cap_lower(current->cap_effective, CAP_NET_RAW);
        return(err);
    }
    else if ( (sock->type != SOCK_STREAM) && (sock->type != SOCK_DGRAM) )
    {
        return(-EINVAL);
    }

    if ( (sk = sk_alloc(PF_USER, GFP_KERNEL, 1)) == NULL )
    {
        TRC(printk(__FUNCTION__":could not allocate sock struct\n"));
        return(-ENOBUFS);
    }
    TRC(printk(__FUNCTION__":alloced socket: %p\n", sk));

    /* Stolen from af_inet.c, but looks like the right thing to do. */
    sock_init_data(sock, sk);
    sk->reuse        = 1;
    sk->zapped       = 0;
    sk->family       = AF_USER;
    sk->protocol     = protocol;
    sk->state        = TCP_CLOSE;
    sk->destruct     = NULL;
    sk->state_change = dummy_state_change;
    sk->write_space  = dummy_write_space;
    sk->error_report = dummy_error_report;
    sk->data_ready   = dummy_data_ready;

    init_waitqueue_head(&sk->protinfo.af_user.connection_wait_queue);

    /* Find the ethernet device information. */
    if ( (sk->protinfo.af_user.device = dev_get_by_name("eth1")) == NULL )
    {
        TRC(printk(__FUNCTION__":could not find eth%d\n", 
                   ETHERNET_DEVICE_NUM));
        return(-EINVAL);
    }
    TRC(printk(__FUNCTION__":found eth%d\n", ETHERNET_DEVICE_NUM));
    sk->bound_dev_if = sk->protinfo.af_user.device->ifindex;

    /* Set up local address information. */
    memcpy(sk->protinfo.af_user.our_ether, 
           sk->protinfo.af_user.device->dev_addr,
           6);
    TRC(printk(__FUNCTION__":set local ethernet address to "));
    PRINT_ETHER(sk->protinfo.af_user.our_ether);

    sk->protinfo.af_user.our_addr.sin_addr.s_addr =
        ((struct in_device *)sk->protinfo.af_user.device->ip_ptr)->
        ifa_list->ifa_address;
    TRC(printk(__FUNCTION__":set local ip address to "));
    PRINT_IP(sk->protinfo.af_user.our_addr.sin_addr.s_addr);

#ifndef DYN_BUFS
    /* XXXXXXX TO BE REMOVED */
    /* Shared buffer space. */
    sk->protinfo.af_user.physarea = bigphysarea_alloc_pages(
        AF_USER_PHYSAREA_SIZE / PAGE_SIZE, 
        0, 
        GFP_KERNEL);
    if ( sk->protinfo.af_user.physarea == NULL )
    {
        TRC(printk(__FUNCTION__":returning, bigphysarea_alloc failed\n"));
        dev_put(sk->protinfo.af_user.device);
        release_sock(sk);
        sock->sk = NULL;
        return(-ENOMEM);
    }

    /* Map the buffers into user space. */
    sk->protinfo.af_user.user_physarea = 0;
    err = get_buffers(sk->protinfo.af_user.physarea, 
                      AF_USER_PHYSAREA_SIZE, 
                      &(sk->protinfo.af_user.user_physarea));
    if ( err )
    {
        TRC(printk(__FUNCTION__":leaving, could not get_buffers\n"));
        bigphysarea_free_pages(sk->protinfo.af_user.physarea);
        dev_put(sk->protinfo.af_user.device);
        release_sock(sk);
        sock->sk = NULL;
        return(-ENOMEM);
    }
    TRC(printk(__FUNCTION__":got bigphysarea=%p(%p) (%p in user space)\n", 
               sk->protinfo.af_user.physarea,
               (void *)virt_to_phys(sk->protinfo.af_user.physarea),
               (void *)sk->protinfo.af_user.user_physarea));
    memset(sk->protinfo.af_user.physarea, 0, AF_USER_PHYSAREA_SIZE);

#else

    /* Allocate a  NIC control structure. */
    sk->protinfo.af_user.nic_ctrl = kmalloc(sizeof(struct usd_endpoint));
    if ( !sk->protinfo.af_user.nic_ctrl || 
         ((u32)sk->protinfo.af_user.nic_ctrl & ~PAGE_MASK) )
    {
        TRC(printk(__FUNCTION__":returning, kmalloc failed or unaligned"
                   " (ret=%p)\n", sk->protinfo.af_user.nic_ctrl));
        release_sock(sk);
        sock->sk = NULL;
        return(-ENOMEM);
    }
    memset(sk->protinfo.af_user.physarea, 0, AF_USER_PHYSAREA_SIZE);

    /* Map the buffers into user space. */
    sk->protinfo.af_user.user_nic_ctrl = 0;
    err = get_buffers(sk->protinfo.af_user.nic_ctrl, 
                      sizeof(struct usd_endpoint), 
                      &(sk->protinfo.af_user.user_nic_ctrl));
    if ( err )
    {
        TRC(printk(__FUNCTION__":leaving, could not map ctrl region\n"));
        kfree(sk->protinfo.af_user.nic_ctrl);
        release_sock(sk);
        sock->sk = NULL;
        return(-ENOMEM);
    }
    TRC(printk(__FUNCTION__":got nic_ctrl=%p(%p) (%p in user space)\n", 
               sk->protinfo.af_user.nic_ctrl,
               (void *)virt_to_phys(sk->protinfo.af_user.nic_ctrl),
               (void *)sk->protinfo.af_user.user_nic_ctrl));

#endif

    /* Set up the function table pointer. */
    sock->ops = &user_proto_ops;

#ifdef TEST_PROC
    {
        int i;
        for ( i = 0; (i<8)&&(socklist[i]); i++ ) ;
        if ( i < 8 ) { socklist[i] = sk; numsocks++; }
    }
#endif


    TRC(printk(__FUNCTION__":leaving\n"));
    return(0);
}


/******************************************************************************
 * user_release:
 *   Destroy socket <sock>. The peer socket should be NULL.
 */
static int user_release(struct socket *sock)
{
    /* NB. Most of this is taken from af_inet.c: destroy_sock */
    struct sk_buff *skb;
    struct sock    *sk;
    int err;

    ASSERT(sock     != NULL);
    sk = sock->sk;

    TRC(printk(__FUNCTION__":entering (sk=%p)\n", sk));

#ifdef TEST_PROC
    {
        int i;
        for ( i = 0; (i<8)&&(socklist[i]!=sk); i++ ) ;
        if ( i < 8 ) { socklist[i] = NULL; numsocks--; }
    }
#endif

    /* We don't want things changing under us. */
    lock_sock(sk);		   

    /* Kill the connection in the card. */
    if ( sk->protinfo.af_user.usd_id != 0 &&
         sk->protinfo.af_user.usd_id != -1 )
    {
        int retries = 0;
    retry:
        if ( (err = acenic_del_connection(sk->protinfo.af_user.device, 
                                          sk->protinfo.af_user.usd_id)) )
        {
            if ( (err == -EAGAIN) && (retries++ < 30) )
            {    
                TRC(printk(__FUNCTION__":retrying connection deletion...\n"));
                current->state = TASK_INTERRUPTIBLE;
                schedule_timeout(HZ/5); /* wait 200ms */
                current->state = TASK_RUNNING;
                goto retry;
            }
            TRC(printk(__FUNCTION__":couldn't delete connection: %d\n", -err));
            return err;
        }

        /* Wait for the device to confirm the tear-down. */
        TRC(printk(__FUNCTION__":waiting for teardown callback...\n"));
        if ( wait_event_timeout(sk->protinfo.af_user.connection_wait_queue, 
                                sk->protinfo.af_user.usd_id == 0,
                                SETUP_CALLBACK_WAIT_IN_SECS * HZ) )
        {
            TRC(printk(__FUNCTION__":leaving, didn't get callback!\n"));
            return(-1);
        }
        TRC(printk(__FUNCTION__":got the callback!\n"));
    }

#ifndef DYN_BUFS
    /* XXX TO BE REMOVED!!! */
    /* Get rid of buffers, and the user-space mapping of them. */
    if ( sk->protinfo.af_user.user_physarea != 0 )
        do_munmap(sk->protinfo.af_user.user_physarea, AF_USER_PHYSAREA_SIZE);
    if ( sk->protinfo.af_user.physarea != NULL )
        bigphysarea_free_pages(sk->protinfo.af_user.physarea);

#else

    /* Free the NIC control structure,a nd the user mapping of it. */
    if ( sk->protinfo.af_user.user_nic_ctrl )
        do_munmap(sk->protinfo.af_user.user_nic_ctrl);
    if ( sk->protinfo.af_user.nic_ctrl )
        kfree(sk->protinfo.af_user.nic_ctrl);

    /* Free the dynamically-mapped data buffers. */
    while ( buf_mappings )
    {
        buf_mapping_t *m = buf_mappings;
        buf_mappings = m->next;
        udma_unlock(m->uaddr, m->len);
        kfree(m);
    }

#endif

    dev_put(sk->protinfo.af_user.device);

    /*
     * Check we don't need to delete stuff that may have been created by the
     * kernel (if we received data via the kernel).
     */
    while((skb = skb_dequeue(&sk->receive_queue)) != NULL) 
    {
        if (skb->sk != NULL && skb->sk != sk) skb->sk->prot->close(skb->sk, 0);
        kfree_skb(skb);
    }

    if ( sk->dst_cache ) dst_release(sk->dst_cache);
    sk_free(sk);

    /* Ensure the port number is freed. */
    if ( sk->prev )
    {
        switch ( sock->type )
        {
        case SOCK_DGRAM:  udp_prot.unhash(sk); break;
        case SOCK_STREAM: tcp_put_port(sk); break;
        }        
    }

    sk->state = TCP_CLOSE;
    sk->dead  = 1;

    TRC(printk(__FUNCTION__":leaving\n"));
    return(0);
}


/******************************************************************************
 * user_bind:
 *   Bind a packet filter to a socket, and tell the card all about it.
 */
static int user_bind(struct socket *sock,		     
                     struct sockaddr *umyaddr,
		     int sockaddr_len)
{
    struct sockaddr_in *in = (struct sockaddr_in *)umyaddr;
    struct sock        *sk;
    unsigned short      snum;
    struct proto       *prot = NULL;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(sock    != NULL);
    ASSERT(umyaddr != NULL);
    sk = sock->sk;

    if ( sockaddr_len < sizeof(struct sockaddr_in) )
    {
        TRC(printk(__FUNCTION__":leaving, incorrect address length\n"));
        return(-EINVAL);
    }

    /* Check the given IP address, and set to default if not specified. */
    if ( in->sin_addr.s_addr != 0 )
    {
        TRC(printk(__FUNCTION__":IP address specified\n"));
        if ( in->sin_addr.s_addr != 
             sk->protinfo.af_user.our_addr.sin_addr.s_addr )
        {
            /* Address isn't ours. */
            TRC(printk(__FUNCTION__":leaving, address isn't ours\n"));
            return(-EADDRNOTAVAIL);
        }
    }
    else
    {
        in->sin_addr.s_addr = sk->protinfo.af_user.our_addr.sin_addr.s_addr;
    }

    if ( sk->num != 0 )
    {
        TRC(printk(__FUNCTION__":leaving, already bound\n"));
        return(-EINVAL);
    }

    /*
     * We now work out the socket number. We steal it from the inet code if
     * the user has not specified.
     */
    switch ( sock->type )
    {
    case SOCK_DGRAM:  prot = &udp_prot; break;
    case SOCK_STREAM: prot = &tcp_prot; break;
    default: 
        TRC(printk(__FUNCTION__":leaving, invalid socket type\n")); 
        return(-EINVAL);
    }
    if ( prot->get_port(sk, ntohs(in->sin_port)) )
    {
        TRC(printk(__FUNCTION__":leaving, couldn't get port\n")); 
        return(-EINVAL);        
    }
    snum = sk->num;
    TRC(printk(__FUNCTION__":got port %d\n", sk->num));

    /*
     * Set up address/port pairs in the sock structure. Destination pair is
     * not available yet.
     */
    sk->rcv_saddr = in->sin_addr.s_addr;
    sk->saddr     = in->sin_addr.s_addr;
    sk->sport     = htons(snum);
    sk->daddr     = 0;
    sk->dport     = 0;

    /* Set up address/port pairs in family-specific section. */
    sk->protinfo.af_user.laddr.sin_addr.s_addr = in->sin_addr.s_addr;
    sk->protinfo.af_user.laddr.sin_port        = htons(snum);
    sk->protinfo.af_user.our_addr.sin_port     = htons(snum);
    sk->protinfo.af_user.raddr.sin_addr.s_addr = 0;
    sk->protinfo.af_user.raddr.sin_port        = 0;
    TRC(printk(__FUNCTION__":set local address to "));
    PRINT_IP_AND_PORT(sk->protinfo.af_user.laddr.sin_addr.s_addr,
                      sk->protinfo.af_user.laddr.sin_port);
    
    /* Now set up the ethernet addresses. */
    memset(sk->protinfo.af_user.rether, 0, ETH_ALEN);
    memcpy(sk->protinfo.af_user.lether, 
           sk->protinfo.af_user.our_ether, 
           ETH_ALEN);
    TRC(printk(__FUNCTION__":set local ethernet address to "));
    PRINT_ETHER(sk->protinfo.af_user.lether);

    TRC(printk(__FUNCTION__":leaving\n"));
    return(0);
}


/******************************************************************************
 * user_connect:
 *   Connect to a remote server.
 */
static int user_connect(struct socket *sock,
                        struct sockaddr *uservaddr,
			int sockaddr_len, 
			int flags)
{
    struct sockaddr_in *in = (struct sockaddr_in *)uservaddr;
    struct sock        *sk;
    int                 addr_type, err, retries = 0;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(sock      != NULL);
    ASSERT(uservaddr != NULL);
    sk = sock->sk;

    if ( sockaddr_len < sizeof(struct sockaddr_in) )
    {
        TRC(printk(__FUNCTION__":leaving, incorrect address length\n"));
        return(-EINVAL);
    }

    sk->protinfo.af_user.raddr.sin_addr.s_addr = in->sin_addr.s_addr;
    sk->protinfo.af_user.raddr.sin_port        = in->sin_port;

    addr_type = inet_addr_type(sk->protinfo.af_user.raddr.sin_addr.s_addr);
    TRC(printk(__FUNCTION__":getting ethernet address\n"));
    if ( addr_type == RTN_MULTICAST || addr_type == RTN_BROADCAST )
    {
        TRC(printk(__FUNCTION__":don't support multicast/broadcast: "));
        PRINT_IP(sk->protinfo.af_user.raddr.sin_addr.s_addr);
        return(-EINVAL);
    }
    else
    {
        struct eth_ipv4_route_req_rsp req_rsp;

        req_rsp.saddr   = sk->protinfo.af_user.our_addr.sin_addr.s_addr;
        req_rsp.daddr   = sk->protinfo.af_user.raddr.sin_addr.s_addr;
        req_rsp.ifindex = sk->protinfo.af_user.device->ifindex;
        if ( (err = get_route_info(&req_rsp)) ) return(err);

        memcpy(sk->protinfo.af_user.rether, req_rsp.mac_addr, ETH_ALEN);
    }

    /* Set up a new connection in the ethernet device. */
    TRC(printk(__FUNCTION__":SETTING UP NEW FIXED CONNECTION...\n"));
    TRC(printk(__FUNCTION__": -> local address = "));
    PRINT_IP_AND_PORT(sk->protinfo.af_user.laddr.sin_addr.s_addr,
                      sk->protinfo.af_user.laddr.sin_port);
    TRC(printk(__FUNCTION__": -> local ethernet address = "));
    PRINT_ETHER(sk->protinfo.af_user.lether);
    TRC(printk(__FUNCTION__": -> remote address = "));
    PRINT_IP_AND_PORT(sk->protinfo.af_user.raddr.sin_addr.s_addr,
                      sk->protinfo.af_user.raddr.sin_port);
    TRC(printk(__FUNCTION__": -> remote ethernet address = "));
    PRINT_ETHER(sk->protinfo.af_user.rether);

 retry:
    if ( (err = acenic_new_connection(sk->protinfo.af_user.device,
#ifndef DYN_BUFS
                                      sk->protinfo.af_user.physarea,
#else
                                      sk->protinfo.af_user.nic_ctrl,
#endif
                                      AF_USER_PHYSAREA_SIZE,
                                      usd_setup_callback,
                                      sk,
                                      usd_rx_callback,
                                      usd_tx_callback,
                                      usd_teardown_callback,
                                      sk->protinfo.af_user.rether,
                                      sk->protinfo.af_user.laddr.
                                        sin_addr.s_addr,
                                      sk->protinfo.af_user.raddr.
                                        sin_addr.s_addr,
                                      sk->protinfo.af_user.laddr.sin_port,
                                      sk->protinfo.af_user.raddr.sin_port,
                                      (sock->type == SOCK_DGRAM) ? 
                                        IPPROTO_UDP : IPPROTO_TCP)) )
    {            
        if ( (err == -EAGAIN) && (retries++ < 30) )
        {    
            TRC(printk(__FUNCTION__":retrying connection creation...\n"));
            current->state = TASK_INTERRUPTIBLE;
            schedule_timeout(HZ/5); /* wait 200ms */
            current->state = TASK_RUNNING;
            goto retry;
        }
        TRC(printk(__FUNCTION__":couldn't create connection: %d\n", -err));
        return err;
    }

    /* Wait for the device to confirm the connection. */
    TRC(printk(__FUNCTION__":waiting for connection callback...\n"));
    if ( wait_event_timeout(sk->protinfo.af_user.connection_wait_queue, 
                            sk->protinfo.af_user.usd_id != 0,
                            SETUP_CALLBACK_WAIT_IN_SECS * HZ) )
    {
        TRC(printk(__FUNCTION__":leaving, didn't get callback!\n"));
        return(-1);
    }
    TRC(printk(__FUNCTION__":got the connection callback!\n"));

    /* Check that the device did not reject the connection for any reason. */
    if ( sk->protinfo.af_user.usd_id == -1 ||
         sk->protinfo.af_user.usd_id == 0 )
    {
        TRC(printk(__FUNCTION__":leaving, card was unable to connect\n"));
        return(-EINVAL);
    }       
    TRC(printk(__FUNCTION__":got usd_id=%d\n", sk->protinfo.af_user.usd_id));

    TRC(printk(__FUNCTION__":leaving\n"));
    return(0);
}


/******************************************************************************
 * user_socketpair:
 *   Should create a pair of connected sockets.
 */
static int user_socketpair(struct socket *sock1, struct socket *sock2)
{
    return(-EOPNOTSUPP);
}


/******************************************************************************
 * user_getname:
 *   Get the name of where the socket goes to.
 */
static int user_getname(struct socket *sock,
			struct sockaddr *uaddr,
			int *usockaddr_len, 
			int peer)
{
    struct sockaddr_in *sin = (struct sockaddr_in *)uaddr;
    struct sock        *sk;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(sock          != NULL);
    ASSERT(uaddr         != NULL);
    ASSERT(usockaddr_len != NULL);
    sk = sock->sk;

    *usockaddr_len = sizeof(struct sockaddr_in);

    /* Fill in the address structure. */
    sin->sin_family = AF_USER;
    if ( peer )
    {
        TRC(printk(__FUNCTION__":leaving, returning remote address\n"));
        sin->sin_port        = sk->protinfo.af_user.raddr.sin_port;
        sin->sin_addr.s_addr = sk->protinfo.af_user.raddr.sin_addr.s_addr;
    }
    else
    {
        TRC(printk(__FUNCTION__":leaving, returning local address\n"));
        sin->sin_port        = sk->protinfo.af_user.laddr.sin_port;
        sin->sin_addr.s_addr = sk->protinfo.af_user.laddr.sin_addr.s_addr;
    }

    return(0);
}


/******************************************************************************
 * get_route_info:
 *   Get route information for a given connection, defined by the IPv4
 *   addresses of its endpoints, and the input/output network interface.
 */
static int get_route_info(struct eth_ipv4_route_req_rsp *rr)
{
    struct rtable *rt;
    struct neighbour *n;
    int err;

    /* Get IP route table entry for the given connection endpoints. */
    if ( (err = ip_route_output(&rt, rr->daddr, rr->saddr, 0, rr->ifindex)) )
    {
        TRC(printk(__FUNCTION__":leaving, route failed (%d)\n", -err));
        return(err);            
    }

    /* Get the MAC address of the gateway. */
    n = rt->u.dst.neighbour;
    if ( neigh_event_send(n, NULL) )
    {
        /* Okay: neigh struct may be incomplete, so wait until it's updated. */
        TRC(printk(__FUNCTION__":waiting for arp update\n"));
        wait_event_interruptible(n->update_queue,
                                 n->nud_state != NUD_INCOMPLETE);
        TRC(printk(__FUNCTION__":neigh struct updated\n"));

        /* The structure is now complete, but we only proceed if it's valid. */
        if ( !(n->nud_state & NUD_VALID) )
        {
            TRC(printk(__FUNCTION__":leaving, arp failed\n"));
            return(-EHOSTUNREACH);
        }
    }

    /* Get the details we require from the table entries. */
    rr->gateway = rt->rt_gateway;
    rr->pmtu    = rt->u.dst.pmtu;
    memcpy(rr->mac_addr, n->ha, ETH_ALEN);
    
    /* This reduces the dst_entry's reference count. */
    ip_rt_put(rt);

    return(0);
}


/******************************************************************************
 * user_ioctl:
 *   Various operations on our socket type.
 */
#define cfu(k,u,l) if ( copy_from_user(k,u,l) ) return(-EFAULT);
#define ctu(u,k,l) if ( copy_to_user(u,k,l) ) return(-EFAULT);
static int user_ioctl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
    int          err;
    struct sock *sk;

    ASSERT(sock != NULL);
    sk = sock->sk;

    switch ( cmd )
    {
    case IOCTL_GET_ROUTE_INFO:
    {
        struct eth_ipv4_route_req_rsp req_rsp;
        TRC(printk(__FUNCTION__":get route info ioctl\n"));
        cfu(&req_rsp, (void *)arg, sizeof(struct eth_ipv4_route_req_rsp));
        if ( req_rsp.ifindex == -1 ) 
        {
            req_rsp.ifindex = sk->protinfo.af_user.device->ifindex;
        }
        if ( req_rsp.saddr == 0 ) 
        {
            req_rsp.saddr = sk->protinfo.af_user.laddr.sin_addr.s_addr;
        }
        if ( (err = get_route_info(&req_rsp)) ) return(err);
        ctu((void *)arg, &req_rsp, sizeof(struct eth_ipv4_route_req_rsp));
        return(0);        
    }
    break; 

    /* Get local mac address. */
    case IOCTL_GET_LOCAL_MAC_ADDRESS:
    {
        TRC(printk(__FUNCTION__":get local mac ioctl\n"));
        ctu((u8 *)arg, sk->protinfo.af_user.our_ether, ETH_ALEN);
        TRC(printk(__FUNCTION__":succeeded; returning local mac address "));
        PRINT_ETHER(sk->protinfo.af_user.our_ether);
        return(0);
    }
    break;

#ifndef DYN_BUFS
    /* XXXXX TO BE REMOVED!!! */
    case IOCTL_GET_SHARED_DATA_AREA:
    {
        void *pointer_buf[2];
        TRC(printk(__FUNCTION__":get shared data area ioctl\n"));
        pointer_buf[0] = sk->protinfo.af_user.physarea;
        pointer_buf[1] = (void *)sk->protinfo.af_user.user_physarea;
        ctu((void *)arg, pointer_buf, 8); 
    }
    break;

#else
    case IOCTL_GET_DESCRIPTOR_REGION:
    {
        TRC(printk(__FUNCTION__":get control region ioctl\n"));
        return(sk->protinfo.af_user.user_nic_ctrl);
    }
    break;

    case IOCTL_MAP_BUFFERS:
    {
        buf_mapping_t *m;
        void *arg_buf[3]; /* 0=uaddr, 1=len, 2=mapping_array */
        TRC(printk(__FUNCTION__"map buffers ioctl\n"));

        cfu(arg_buf, (void *)arg, 12);

        if ( (err = verify_area(VERIFY_WRITE,arg_buf[0],(size_t)arg_buf[1])) ||
             (err = verify_area(VERIFY_WRITE,
                                arg_buf[2],
                                (size_t)arg_buf[1] / PAGE_SIZE)) )
        {
            TRC(printk(__FUNCTION__":leaving, couldn't verify write\n"));
            return(err);
        }

        err = udma_lock(arg_buf[0], (size_t)arg_buf[1]);
        if ( !err ) 
        {
            udma_build_maplist(arg_buf[0], (size_t)arg_buf[1], arg_buf[2]);
     
            if ( (m = kmalloc(sizeof(buf_mapping_t))) == NULL )
            {
                TRC(printk(__FUNCTION__":leaving, kmalloc failed\n"));
                udma_unlock(arg_buf[0], (size_t)arg_buf[1]);
                return(-ENOMEM);
            }
            m->uaddr = arg_buf[0];
            m->len   = (size_t)arg_buf[1];
            m->next  = sk->protinfo.af_user.buf_mappings;
            sk->protinfo.af_user.buf_mappings = m;
        }
        return(err);
    }
    break;

    case IOCTL_UNMAP_BUFFERS:
    {
        buf_mapping_t **m;
        TRC(printk(__FUNCTION__"unmap buffers ioctl\n"));

        for (  m = &(sk->protinfo.af_user.buf_mappings);
              *m != NULL;
               m = &((*m)->next) ) 
        {
            /* Look for a mapping starting at uaddr <arg> in the map chain. */
            buf_mapping_t *t = *m;
            if ( (*m)->uaddr != arg ) continue;

            /* Found it! Unmap and free the chain member. */
            udma_unlock((*m)->uaddr, (*m)->len);
            *m = (*m)->next;
            kfree(t);
            return(0);
        }

        /* We were given an illegal mapping handle. */
        return(-ENOBUFS);
    }
    break;

#endif

    /* Get the user-accessible direct-control mailbox. */
    case IOCTL_GET_MAILBOX:
    {
        struct ace_private *ap = sk->protinfo.af_user.device->priv;
        return((int)&(ap->regs->UsdMb[USD_MBOX_BASE + 
                      (sk->protinfo.af_user.usd_id % NUM_USD_MBOXS)]));
    }
    break;

    /* Get the user-accessible direct-control structure pointer. */
    case IOCTL_GET_CONTROL_STRUCT:
    {
        struct ace_private *ap = sk->protinfo.af_user.device->priv;
        return((int)&(ap->regs->usd2_ctrl[sk->protinfo.af_user.usd_id]));
    }
    break;

    /* Get the connection identifier. */
    case IOCTL_GET_CONNECTION_ID:
    {
        return(sk->protinfo.af_user.usd_id);
    }

    /* Register the task to be woken when the given socket gets a callback. */
    case IOCTL_REGISTER_SOCKET_FOR_CALLBACKS:
    {
        struct task_struct *p;
        int                 pid = arg;
        TRC(printk(__FUNCTION__":register socket for callbacks ioctl\n"));

        sk->protinfo.af_user.callback_task = NULL;
	read_lock(&tasklist_lock);
	for_each_task( p ) 
        {
            if ( pid == p->pid ) 
            {
                sk->protinfo.af_user.callback_task = p;
                break;
            }
        }
	read_unlock(&tasklist_lock);

        return((sk->protinfo.af_user.callback_task != NULL) ? 0 : -1);
    }
    break;

    /* Request an rx callback. */
    case IOCTL_RX_REQ_CALLBACK:
    {
        acenic_usd2_rx2_request_callback(sk->protinfo.af_user.device,
                                         sk->protinfo.af_user.usd_id,
                                         arg);
        return(0);
    }
    break;

    /* Request a tx callback. */
    case IOCTL_TX_REQ_CALLBACK:
    {
        acenic_usd2_tx2_request_callback(sk->protinfo.af_user.device,
                                         sk->protinfo.af_user.usd_id,
                                         arg);
        return(0);
    }
    break;

    case IOCTL_INSTALL_FILTER:
    {
        struct user_opt *u = &sk->protinfo.af_user;
        struct dpf_ir *f = kmalloc(FILTER_SIZE, GFP_USER);
        int retries = 0;

        TRC(printk(__FUNCTION__":install filter ioctl\n"));

        if ( f == NULL )
        {
            TRC(printk(__FUNCTION__":couldn't kmalloc %d bytes for filter\n",
                       FILTER_SIZE));
            return(-ENOMEM);
        }

        dpf_begin(f);

        /* Check Ethernet frame type == ETH_IP */
        dpf_mkeq(f, 16, 12, 0xffff, 0x0800);

        /* Shift on to IP header */
        dpf_shifti(f, 14, SHIFT|HDR_EXT|PKT_EXT);

        /* Check IP version == IPv4 */
        dpf_mkeq(f, 8, 0, 0xf0, 0x40);

        /* Get the datagram length */
        dpf_mkshift(f, 16, 2, 0xffff, 0, PKT_EXT);

        /* Check IP protocol type == TCP/UDP as appropriate. */
        dpf_mkeq(f, 8, 9, 0xff, 
                 (sock->type == SOCK_DGRAM) ? IPPROTO_UDP : IPPROTO_TCP);

        /* Check IP source & destination addresses */
        dpf_mkeq(f, 32, 16, ~0, ntohl(u->laddr.sin_addr.s_addr));
        if ( u->raddr.sin_addr.s_addr )
        {
            /* Only active sockets check destination address. */
            dpf_mkeq(f, 32, 12, ~0, ntohl(u->raddr.sin_addr.s_addr));
        }

        /* Shift on to TCP/UDP header, and add IP header length to total. */
        dpf_mkshift(f, 8, 0, 0x0f, 2, SHIFT|HDR_EXT);

        /* Add TCP/UDP header length to total header length. */
        if ( sock->type == SOCK_DGRAM )
        {
            dpf_shifti(f, 8, HDR_EXT);
        }
        else
        {
            dpf_mkshift(f, 8, 12, 0xf0, -2, HDR_EXT);
        }
        
        /* Check TCP/UDP source & destination ports. */
        if ( u->raddr.sin_port )
        {
            dpf_mkeq(f, 32, 0, ~0, (ntohs(u->raddr.sin_port) << 16) | 
                     ntohs(u->laddr.sin_port));
        }
        else
        {
            dpf_mkeq(f, 16, 2, 0xffff, ntohs(u->laddr.sin_port));
        }

    retry:
        if ( (err = ace_install_connection_filter(sk->protinfo.af_user.device,
                                                  sk->protinfo.af_user.usd_id,
                                                  f)) )
        {
            if ( (err == -EAGAIN) && (retries++ < 30) )
            {    
                TRC(printk(__FUNCTION__":retrying filter initialisation..\n"));
                current->state = TASK_INTERRUPTIBLE;
                schedule_timeout(HZ/5); /* wait 200ms */
                current->state = TASK_RUNNING;
                goto retry;
            }
            TRC(printk(__FUNCTION__":error installing filter (%d)\n", -err));
            kfree(f);
            return(err);
        }

        kfree(f);
        return(0);
    }

    /* Invalid command argument. */
    default:
    {
        TRC(printk(__FUNCTION__":returning, invalid command argument %d\n",
                   cmd));
        return(-EINVAL);
    }
    }

    TRC(printk(__FUNCTION__":leaving, succeeded\n"));
    return(0);
}


/******************************************************************************
 * user_listen:
 *   Listen for incoming connections.
 */
static int user_listen(struct socket *sock, int len)
{
    struct sock *sk;
    int err, retries=0;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(sock != NULL);
    sk = sock->sk;

    /*
     * By setting the remote address to zero, we disable packet filtering
     * for the remote address, allowing us to listen for incoming SYNs.
     */
    sk->protinfo.af_user.raddr.sin_addr.s_addr = 0;
    sk->protinfo.af_user.raddr.sin_port        = 0;
    memset(sk->protinfo.af_user.rether, 0, ETH_ALEN);

    /* Set up a new connection in the ethernet device. */
    TRC(printk(__FUNCTION__":SETTING UP NEW LISTENING CONNECTION...\n"));
    TRC(printk(__FUNCTION__": -> local address = "));
    PRINT_IP_AND_PORT(sk->protinfo.af_user.laddr.sin_addr.s_addr,
                      sk->protinfo.af_user.laddr.sin_port);
    TRC(printk(__FUNCTION__": -> local ethernet address = "));
    PRINT_ETHER(sk->protinfo.af_user.lether);

 retry:
    if ( (err = acenic_new_connection(sk->protinfo.af_user.device,
#ifndef DYN_BUFS
                                      sk->protinfo.af_user.physarea,
#else
                                      sk->protinfo.af_user.nic_ctrl,
#endif
                                      AF_USER_PHYSAREA_SIZE,
                                      usd_setup_callback,
                                      sk,
                                      usd_rx_callback,
                                      usd_tx_callback,
                                      usd_teardown_callback,
                                      sk->protinfo.af_user.rether,
                                      sk->protinfo.af_user.laddr.
                                        sin_addr.s_addr,
                                      sk->protinfo.af_user.raddr.
                                        sin_addr.s_addr,
                                      sk->protinfo.af_user.laddr.sin_port,
                                      sk->protinfo.af_user.raddr.sin_port,
                                      (sock->type == SOCK_DGRAM) ? 
                                        IPPROTO_UDP : IPPROTO_TCP)) )
    {        
        if ( (err == -EAGAIN) && (retries++ < 30) )
        {    
            TRC(printk(__FUNCTION__":retrying connection creation...\n"));
            current->state = TASK_INTERRUPTIBLE;
            schedule_timeout(HZ/5); /* wait 200ms */
            current->state = TASK_RUNNING;
            goto retry;
        }
        TRC(printk(__FUNCTION__":couldn't create connection: %d\n", -err));
        return err;
    }

    /* Wait for the device to confirm the connection. */
    TRC(printk(__FUNCTION__":waiting for connection callback...\n"));
    if ( wait_event_timeout(sk->protinfo.af_user.connection_wait_queue, 
                            sk->protinfo.af_user.usd_id != 0,
                            SETUP_CALLBACK_WAIT_IN_SECS * HZ) )
    {
        TRC(printk(__FUNCTION__":leaving, didn't get callback!\n"));
        return(-1);
    }
    TRC(printk(__FUNCTION__":got the connection callback!\n"));

    /* Check that the device did not reject the connection for any reason. */
    if ( sk->protinfo.af_user.usd_id == -1 ||
         sk->protinfo.af_user.usd_id == 0 )
    {
        TRC(printk(__FUNCTION__":leaving, card was unable to connect\n"));
        return(-EINVAL);
    }       
    TRC(printk(__FUNCTION__":got usd_id=%d\n", sk->protinfo.af_user.usd_id));

    TRC(printk(__FUNCTION__":leaving\n"));
    return(0);
}


/******************************************************************************
 * --- PRIVATE FUNCTION DEFINITIONS ---
 */

/******************************************************************************
 * vm_flags:
 *   Stolen from mm/mmap.c
 */
static inline unsigned long vm_flags(unsigned long prot, unsigned long flags)
{
#define _trans(x,bit1,bit2) \
((bit1==bit2)?(x&bit1):(x&bit1)?bit2:0)

	unsigned long prot_bits, flag_bits;
	prot_bits =
		_trans(prot, PROT_READ, VM_READ) |
		_trans(prot, PROT_WRITE, VM_WRITE) |
		_trans(prot, PROT_EXEC, VM_EXEC);
	flag_bits =
		_trans(flags, MAP_GROWSDOWN, VM_GROWSDOWN) |
		_trans(flags, MAP_DENYWRITE, VM_DENYWRITE) |
		_trans(flags, MAP_EXECUTABLE, VM_EXECUTABLE);
	return prot_bits | flag_bits;
#undef _trans
}


/******************************************************************************
 * get_buffers:
 *   Map <len> bytes in kernel address space, starting at <kern_addr>, into
 *   the current process's address space. The start of the mapping in
 *   user space is returned in <user_addr>.
 *
 *   NB. Neither <kern_addr> nor <len> need be page-aligned. This function
 *       will "do the right thing" :-)
 *
 *   Returns 0 on success, else -(errno).
 */
static int get_buffers(void *kern_addr, int len, unsigned long *user_addr)
{
    struct mm_struct      *mm  = current->mm;
    struct vm_area_struct *vma;
    void *aligned_kern_addr = (void *)(((u32)kern_addr) & PAGE_MASK);
    int   aligned_len       = (len + (((u32)kern_addr) & ~PAGE_MASK) + 
                               PAGE_SIZE - 1) & PAGE_MASK;

    TRC(printk(__FUNCTION__":entered\n"));

    ASSERT(user_addr != NULL);

    /* Will we end up locking too much memory? */
    if ( ((mm->locked_vm << PAGE_SHIFT) + aligned_len) > 
         current->rlim[RLIMIT_MEMLOCK].rlim_cur )
    {
        TRC(printk(__FUNCTION__":leaving, would lock too much\n"));
        return(-EAGAIN);
    }

    /* Get an unmapped area of virtual address space. */
    *user_addr = get_unmapped_area(*user_addr, aligned_len);
    if ( *user_addr == 0 )
    {
        TRC(printk(__FUNCTION__":leaving, could not get an unmapped area\n"));
        return(-ENOMEM);
    }
    TRC(printk(__FUNCTION__":got unmapped area at %p\n", (void *)*user_addr));

    /* Allocate and initialise a VM structure. */
    vma = kmem_cache_alloc(vm_area_cachep, SLAB_KERNEL);
    if ( vma == NULL )
    {
        TRC(printk(__FUNCTION__":leaving, could not kmem_cache_alloc\n"));
        return(-ENOMEM);
    }
    vma->vm_mm        = mm;
    vma->vm_start     = *user_addr;
    vma->vm_end       = *user_addr + aligned_len;
    vma->vm_flags     = vm_flags(PROT_WRITE|PROT_READ,0) | mm->def_flags;
    vma->vm_flags    |= VM_MAYREAD | VM_MAYWRITE;
    vma->vm_flags    |= VM_SHARED  | VM_MAYSHARE | VM_LOCKED;
    vma->vm_page_prot = protection_map[vma->vm_flags & 0x0f];
    vma->vm_ops       = NULL;
    vma->vm_pgoff     = 0;
    vma->vm_file      = NULL;
    vma->vm_private_data = 0;

    /*
     * Ensure the virtual address space we allocated is fully unmapped.
     * Yes, I know we asked for an unmapped area originally, but this is the
     * way mm/mmap.c does it!
     */
    if ( do_munmap(*user_addr, aligned_len) )
    {
        TRC(printk(__FUNCTION__":leaving, could not clear old mem map\n"));
	kmem_cache_free(vm_area_cachep, vma);
	return(-ENOMEM);
    }

    /* Check we haven't exhausted the address space. */
    if ((mm->total_vm << PAGE_SHIFT) + aligned_len
        > current->rlim[RLIMIT_AS].rlim_cur)
    {
        TRC(printk(__FUNCTION__":leaving, not enough address space\n"));
	kmem_cache_free(vm_area_cachep, vma);
	return(-ENOMEM);
    }        

    /* Now map the virtual address range onto the socket's physical buffers. */
    if ( remap_page_range(vma->vm_start,
                          virt_to_phys(aligned_kern_addr),
                          vma->vm_end - vma->vm_start,
                          vma->vm_page_prot) )
    {
        TRC(printk(__FUNCTION__":leaving, could not remap page range\n"));
        kmem_cache_free(vm_area_cachep, vma);
        return(-EAGAIN);
    }

    /*
     * Insert the info into the kernel structures, and merge VM segments.
     * This may result in our vma being merged, so we save off the start
     * address for future reference.
     */
    *user_addr = vma->vm_start; /* can user_addr have changed?? */
    insert_vm_struct(mm, vma);
    merge_segments(mm, vma->vm_start, vma->vm_end);   
 
    /* Tidy up all the loose ends. */
    mm->total_vm  += aligned_len >> PAGE_SHIFT;
    mm->locked_vm += aligned_len >> PAGE_SHIFT;
    make_pages_present(*user_addr, *user_addr + aligned_len);
    TRC(printk(__FUNCTION__":made pages present from %p to %p\n",
               (void *)*user_addr, (void *)*user_addr + aligned_len));

    /* Function parameters may not have been page-aligned. */
    *user_addr += ((u32)kern_addr) & ~PAGE_MASK;

    TRC(printk(__FUNCTION__":leaving, succeeded\n"));
    return(0);
}


/******************************************************************************
 * usd_setup_callback:
 *   Called by ethernet device when a new connection has been set up.
 */
static void usd_setup_callback(void *argument, int usd_id)
{
    struct sock *sk = argument;

    TRC(printk(__FUNCTION__":entered for id=%d\n", usd_id));

    /* Let the user process check for error value. */
    sk->protinfo.af_user.usd_id = (usd_id != 0) ? usd_id : -1;
    wake_up_interruptible(&(sk->protinfo.af_user.connection_wait_queue));

    TRC(printk(__FUNCTION__":leaving\n"));
}


/******************************************************************************
 * usd_irq_callback:
 *   Sends a signal back to user-land when the network card triggers an
 *   event. A code value is returned specifying the conenction id and the
 *   type of teh event (RX or TX).
 *
 *   If it's a receive event, the code has bit 16 set (RX_EVENT), otherwise
 *   this bit will be clear.
 */
static void usd_irq_callback(void *argument, int rx_event)
{
    struct sock *sk = argument;
    struct siginfo info;
    
    ASSERT(sk != NULL);

    STAMP(TS_IRQ_CALLBACK);

    if ( sk->protinfo.af_user.callback_task )
    {
        int ret;

        /*
         * Setting the siginfo struct correctly is very important: Linux 2.2.x
         * is particularly fussy. si_signo must be set or queued signals can
         * never be found again. si_code must be set to SI_KERNEL or the
         * permissions check may fail if we interrupt the execution of someone
         * else's process.
         */
        info.si_value.sival_int = 
            sk->protinfo.af_user.usd_id | (rx_event ? RX_EVENT : 0);
        info.si_code            = SI_KERNEL;
        info.si_signo           = AF_USER_CALLBACK_SIGNAL;
        info.si_errno           = 0;
        
        if ( (ret = send_sig_info(AF_USER_CALLBACK_SIGNAL, 
                                  &info, 
                                  sk->protinfo.af_user.callback_task) < 0) )
        {
            /*
             * XXX Argh! Should probably have a standard signal allocated to
             * fall back on in this case. Application would interpret the
             * fallback signal to mean that it should do a paranoid check on
             * all connections it has open.
             */
                printk("XXXXXXXXXX DROPPED AN RT SIGNAL: error %d\n", -ret);
        }
    }
}


/******************************************************************************
 * usd_{rx|tx}_callback:
 *   These are called from the network driver to indicate events to be
 *   passed back to user-land. These actually get usd_irq_callback to do the
 *   work.
 */
static void usd_rx_callback(void *argument)
{
    usd_irq_callback(argument, 1);
}
static void usd_tx_callback(void *argument)
{
    usd_irq_callback(argument, 0);
}


/******************************************************************************
 * usd_teardown_callback:
 *   Called by ethernet device when a connection has been torn down.
 */
static void usd_teardown_callback(void *argument)
{
    struct sock *sk = argument;

    ASSERT(sk != NULL);

    TRC(printk("***"__FUNCTION__"***:entered for id=%d\n",
               sk->protinfo.af_user.usd_id));

    /*
     * Setting the usd id to zero tells the waiting thread that the
     * connection has been torn down.
     */
    sk->protinfo.af_user.usd_id = 0;
    wake_up_interruptible(&(sk->protinfo.af_user.connection_wait_queue));
}


/******************************************************************************
 * dummy_callback:
 *   We don't want those annoying automatic callbacks!
 */
static void dummy_state_change(struct sock *sk) 
{ TRC(printk("***"__FUNCTION__"***:entered\n")); }
static void dummy_write_space(struct sock *sk) 
{ TRC(printk("***"__FUNCTION__"***:entered\n")); }
static void dummy_error_report(struct sock *sk) 
{ TRC(printk("***"__FUNCTION__"***:entered\n")); }


/******************************************************************************
 * dummy_data_ready:
 *   Discard any incoming skbuff's from standard kernel mechanisms.
 */
static void dummy_data_ready(struct sock *sk, int len)
{ 
    int             err;
    struct sk_buff *skb;

    ASSERT(sk != NULL);
    TRC(printk("***"__FUNCTION__"***:entered\n"));
    while ( (skb = skb_recv_datagram(sk, 0, 1, &err)) != NULL )
    {
        TRC(printk(__FUNCTION__":freeing datagram\n"));
        skb_free_datagram(sk, skb);
    }
}


/******************************************************************************
 * --- SUPPORT FOR MAPPING COMMONLY-USED KERNEL STRUCTURES INTO USERLAND ---
 */

static void timermap_init(void)
{
    struct proc_dir_entry *p;
    p = create_proc_entry("afuser_tmap", S_IFREG | S_IRUGO, &proc_root);
    if ( !p )
    {
        TRC(printk(__FUNCTION__":couldn't create proc struct\n"));
        return;
    }
    p->size = 80;
    p->get_info = timermap_read;
}

static int timermap_read(char *buffer,
                         char **buffer_location,
                         off_t offset,
                         int buffer_length)
{
#define NUM_TIMERMAPS 2
    if ( (offset != 0) || (buffer_length < 4*NUM_TIMERMAPS) ) return(0);
    *buffer_location = buffer;
    if ( get_buffers((void *)&jiffies, 
                     sizeof(unsigned long), 
                     (unsigned long *)(buffer+0)) ||
         get_buffers((void *)&xtime,   
                     sizeof(struct timeval), 
                     (unsigned long *)(buffer+4)) ) return(0);
    return(4*NUM_TIMERMAPS);
}

static void timermap_close(void)
{
    remove_proc_entry("afuser_tmap", &proc_root);
}


/******************************************************************************
 * --- SUPPORT FOR A /proc FS TEST DATA ENTRY ---
 */
#ifdef TEST_PROC

static struct timer_list test_timer;

static void test_do_tick(unsigned long data)
{
    int i;
    if ( numsocks < 2 || (test_curr - test_buffer) >= ((TEST_BUFLEN-100)/2) )
    {
        goto out;
    }
    *(test_curr++) = current->pid;
    for ( i = 0; i < 8; i++ )
    {
        struct ace_private *ap;
        usd2_ctrl_t *c;
        int tx_len;
        if (!socklist[i] || (socklist[i]->state != TCP_ESTABLISHED)) continue;
        ap     = socklist[i]->protinfo.af_user.device->priv;
        c      = &(ap->regs->usd2_ctrl[socklist[i]->protinfo.af_user.usd_id]);
        tx_len = (c->tx_prod-c->tx_cons) & (TX2_RING_ENTRIES-1);
        *(test_curr++) = c->tx_cons;
    }
 out:
    test_timer.expires = jiffies + 1;
    add_timer(&test_timer);
}

static void test_init(void)
{
    test_buffer = (u16 *) 
        bigphysarea_alloc_pages(TEST_BUFLEN/PAGE_SIZE, 0, GFP_KERNEL);
    test_curr = test_buffer;
    proc_register(&proc_root, &test_file);

    init_timer(&test_timer);
    test_timer.function = test_do_tick;
    test_timer.expires  = jiffies + 1;
    add_timer(&test_timer);
}

static int test_read(char *buffer,
                     char **buffer_location,
                     off_t offset,
                     int buffer_length,
                     int zero)
{
    int len = (test_curr - test_buffer) * 2 - offset;
    if ( (offset < 0) || (len <= 0) || (buffer_length <= 0) )
    {
        test_curr = test_buffer;
        return(0);
    }
    if ( buffer_length < len ) len = buffer_length;
    *buffer_location = buffer;
    memcpy(buffer, ((char *)test_buffer) + offset, len);
    return(len);
}

static void test_close(void)
{
    proc_unregister(&proc_root, test_file.low_ino);
    bigphysarea_free_pages((caddr_t)test_buffer);
}

#endif


/******************************************************************************
 * --- MODULE SUPPORT ROUTINES ---
 */
#ifdef MODULE

int init_module(void)
{
    user_proto_init(NULL);
    return(0);
}

void cleanup_module(void)
{
    (void) sock_unregister(AF_USER);
    timermap_close();
#ifdef TEST_PROC
    test_close();
#endif
}
#endif
