/******************************************************************************
 * acenic.h
 * 
 * All modifications copyright (c) 1999-2000, University of Cambridge,
 * by Ian Pratt and Keir Fraser (iap10@cl.cam.ac.uk, kaf24@cl.cam.ac.uk)
 * 
 * Original modifications of Jes Sorenson's source by Tim Deegan.
 * 
 * Please note the original copyright and licence in `acenic.c'. 
 */

#ifndef _ACENIC_H_
#define _ACENIC_H_

#include "acenic_endpoint.h"

#define ACENIC_DEBUGGING_IOCTLS
#define USD_CHANNELS            64

#define NUM_USD_MBOXS           8
#define USD_MBOX_BASE           (16 - NUM_USD_MBOXS)

/*
 * Addressing:
 *
 * The Tigon uses 64-bit host addresses, regardless of their actual
 * length, and it expects a big-endian format. For 32 bit systems the
 * upper 32 bits of the address are simply ignored (zero), however for
 * little endian 64 bit systems (Alpha) this looks strange with the
 * two parts of the address word being swapped.
 *
 * The addresses are split in two 32 bit words for all architectures
 * as some of them are in PCI shared memory and it is necessary to use
 * readl/writel to access them.
 *
 * The addressing code is derived from Pete Beckman's work, but
 * modified to deal properly with readl/writel usage.
 */



#ifdef __KERNEL__

static inline void set_aceaddr(aceaddr *aa, volatile void *addr)
{
	unsigned long baddr = virt_to_bus((void *)addr);
#if (BITS_PER_LONG == 64)
	aa->addrlo = baddr & 0xffffffff;
	aa->addrhi = baddr >> 32;
#else
    /* Don't bother setting zero every time */
	aa->addrlo = baddr;
#endif
	mb();
}


static inline void *get_aceaddr(aceaddr *aa)
{
	unsigned long addr;
	mb();
#if (BITS_PER_LONG == 64)
	addr = (u64)aa->addrhi << 32 | aa->addrlo;
#else
	addr = aa->addrlo;
#endif
	return bus_to_virt(addr);
}

#endif /* __KERNEL__ */


static inline void set_aceaddr_bus(aceaddr *aa, volatile void *addr)
{
	unsigned long baddr = (unsigned long)addr;
#if (BITS_PER_LONG == 64)
	aa->addrlo = baddr & 0xffffffff;
	aa->addrhi = baddr >> 32;
#else
    /* Don't bother setting zero every time */
	aa->addrlo = baddr;
#endif
	mb();
}


static inline void *get_aceaddr_bus(aceaddr *aa)
{
	unsigned long addr;
	mb();
#if (BITS_PER_LONG == 64)
	addr = (u64)aa->addrhi << 32 | aa->addrlo;
#else
	addr = aa->addrlo;
#endif
	return (void *)addr;
}

/* A mailbox is 64 bits.  Writing to the second 32 bits triggers
 * an event on the card */
typedef struct mbox
{
    u32 hi;
    u32 lo;
} mbox_t;


/* USD2 ********************************************************************/


typedef u16 index_t;

/* This is the control block in the 2KB NIC shared memory section that
is used to control a USD2 channel. The host should really only ever
write to it. Support sub-word stores. */

typedef struct
{
#ifdef __LITTLE_ENDIAN
  index_t tx_ref;       
  index_t tx_prod;    // 1+ ptr to packet Q'd for TX
  index_t rx_ref;       
  index_t free_prod;  // 1+ ptr to Q'd Free Buffer descr
  index_t rx_prod;    // Don't Poll from Host!
  index_t tx_cons;    // Don't Poll from Host!
#else
  index_t tx_prod;    // 1+ ptr to packet Q'd for TX
  index_t tx_ref;       
  index_t free_prod;  // 1+ ptr to Q'd Free Buffer descr
  index_t rx_ref;       
  index_t tx_cons;    // Don't Poll from Host!
  index_t rx_prod;    // Don't Poll from Host!
#endif

} usd2_ctrl_t;  // 12 bytes total

/* USD2 ********************************************************************/

struct ace_regs {
    u32	pad0[16];	/* PCI control registers */

    u32	HostCtrl;	/* 0x40 */
    u32	LocalCtrl;

    u32	pad1[2];

    u32	MiscCfg;	/* 0x50 */

    u32	pad2[2];

    u32	PciState;

    u32	pad3[2];	/* 0x60 */

    u32	WinBase;
    u32	WinData;

    u32	pad4[12];	/* 0x70 */

    u32	DmaWriteState;	/* 0xa0 */
    u32	pad5[3];
    u32	DmaReadState;	/* 0xb0 */

    u32	pad6[26];

    u32	AssistState;

    u32	pad7[8];	/* 0x120 */

    u32	CpuCtrl;	/* 0x140 */
    u32	Pc;

    u32	pad8[3];

    u32	SramAddr;	/* 0x154 */
    u32	SramData;

    u32	pad9[49];

    u32	MacRxState;	/* 0x220 */

    u32	pad10[7];

    u32	CpuBCtrl;	/* 0x240 */
    u32	PcB;

    u32	pad11[3];

    u32	SramBAddr;	/* 0x254 */
    u32	SramBData;

    u32	pad12[105];

    u32	pad13[32];	/* 0x400 */
    u32	Stats[32];

    mbox_t Mb[16];      /* 0x500 */
#define CmdPrd Mb[1].lo

    mbox_t UsdMb[16];	/* Another 16 mboxes, for the USD channels */

    u32	MacAddrHi;	/* 0x600 */
    u32	MacAddrLo;
    u32	InfoPtrHi;
    u32	InfoPtrLo;
    u32	MultiCastHi;	/* 0x610 */
    u32	MultiCastLo;
    u32	ModeStat;
    u32	DmaReadCfg;
    u32	DmaWriteCfg;	/* 0x620 */
    u32	TxBufRat;
    u32	EvtCsm;
    u32	CmdCsm;
    u32	TuneRxCoalTicks;/* 0x630 */
    u32	TuneTxCoalTicks;
    u32	TuneStatTicks;
    u32	TuneMaxTxDesc;
    u32	TuneMaxRxDesc;	/* 0x640 */
    u32	TuneTrace;
    u32	TuneLink;
    u32	TuneFastLink;
    u32	TracePtr;	/* 0x650 */
    u32	TraceStrt;
    u32	TraceLen;
    u32	IfIdx;
    u32	IfMtu;		/* 0x660 */
    u32	MaskInt;
    u32	LnkState;
    u32	FastLnkState;
    //u32	pad_14[2];	/* 0x670 */
    u32 PerfTestPtrHi;
    u32 PerfTestPtrLo;

    u32	NewRXFilterHi;
    u32	NewRXFilterLo;
    u32	NewRXFilterLen; 
    u32	pad_15;                 /* 0x684 */

    u32	Interrupt;

    u32	rx_interrupts[(USD_CHANNELS+31)/32];
    u32	tx_interrupts[(USD_CHANNELS+31)/32];

    u32	pad_16[29 - (((USD_CHANNELS+31)/32)*2)];
    u32	CmdRng[64];	/* 0x700 */
    u32	Window[64];  /* 0x800 : USD2 reduce from 512 to 64 */

    usd2_ctrl_t	 usd2_ctrl[224]; /* 0x900 XXXX Debug size */
    /* end at 0x800 + 2KB = 0xFFF */
};

#define ACE_WINDOW_SIZE	0x800

#define ACE_JUMBO_MTU 9000
#define ACE_STD_MTU 1500

#define ACE_MTU ACE_JUMBO_MTU  /* HACK ! */

#define ACE_TRACE_SIZE 0x8000

/*
 * Host control register bits.
 */
	
#define IN_INT		0x01
#define CLR_INT		0x02
#define BYTE_SWAP	0x10
#define WORD_SWAP	0x20
#define MASK_INTS	0x40

/*
 * Local control register bits.
 */

#define EEPROM_DATA_IN		0x800000
#define EEPROM_DATA_OUT		0x400000
#define EEPROM_WRITE_ENABLE	0x200000
#define EEPROM_CLK_OUT		0x100000

#define EEPROM_BASE		0xa0000000

#define EEPROM_WRITE_SELECT	0xa0
#define EEPROM_READ_SELECT	0xa1

#define SRAM_BANK_512K		0x200


/*
 * Misc Config bits
 */

#define SYNC_SRAM_TIMING	0x100000


/*
 * CPU state bits.
 */

#define CPU_RESET		0x01
#define CPU_TRACE		0x02
#define CPU_PROM_FAILED		0x10
#define CPU_HALT		0x00010000
#define CPU_HALTED		0xffff0000


/*
 * PCI State bits.
 */

#define DMA_READ_MAX_4		0x04
#define DMA_READ_MAX_16		0x08
#define DMA_READ_MAX_32		0x0c
#define DMA_READ_MAX_64		0x10
#define DMA_READ_MAX_128	0x14
#define DMA_READ_MAX_256	0x18
#define DMA_READ_MAX_1K		0x1c
#define DMA_WRITE_MAX_4		0x20
#define DMA_WRITE_MAX_16	0x40
#define DMA_WRITE_MAX_32	0x60
#define DMA_WRITE_MAX_64	0x80
#define DMA_WRITE_MAX_128	0xa0
#define DMA_WRITE_MAX_256	0xc0
#define DMA_WRITE_MAX_1K	0xe0
#define MEM_READ_MULTIPLE	0x00020000
#define DMA_WRITE_ALL_ALIGN	0x00800000
#define READ_CMD_MEM		0x06000000
#define WRITE_CMD_MEM		0x70000000


/*
 * Mode status
 */

#define ACE_BYTE_SWAP_DATA	0x10
#define ACE_WARN		0x08
#define ACE_WORD_SWAP		0x04
#define ACE_NO_JUMBO_FRAG	0x200
#define ACE_FATAL		0x40000000


/*
 * DMA config
 */

#define DMA_THRESH_8W		0x80


/*
 * Tuning parameters
 */

#define TICKS_PER_SEC		1000000


/*
 * Link bits
 */

#define LNK_PREF		0x00008000
#define LNK_10MB		0x00010000
#define LNK_100MB		0x00020000
#define LNK_1000MB		0x00040000
#define LNK_FULL_DUPLEX		0x00080000
#define LNK_HALF_DUPLEX		0x00100000
#define LNK_TX_FLOW_CTL_Y	0x00200000
#define LNK_NEG_ADVANCED	0x00400000
#define LNK_RX_FLOW_CTL_Y	0x00800000
#define LNK_NIC			0x01000000
#define LNK_JAM			0x02000000
#define LNK_JUMBO		0x04000000
#define LNK_ALTEON		0x08000000
#define LNK_NEG_FCTL		0x10000000
#define LNK_NEGOTIATE		0x20000000
#define LNK_ENABLE		0x40000000
#define LNK_UP			0x80000000


/*
 * Event definitions
 */

#define EVT_RING_ENTRIES	256
#define EVT_RING_SIZE	(EVT_RING_ENTRIES * sizeof(struct event))

struct event {
#ifdef __LITTLE_ENDIAN
	u32	idx:12;
	u32	code:12;
	u32	evt:8;
#else
	u32	evt:8;
	u32	code:12;
	u32	idx:12;
#endif
	u32     pad;
};


/*
 * Events
 */
#define E_FW_RUNNING		0x01
#define E_LNK_STATE		0x06
#define E_C_LINK_UP		0x01
#define E_C_LINK_DOWN		0x02
#define E_ERROR			0x07
#define E_C_ERR_INVAL_CMD	0x01
#define E_C_ERR_UNIMP_CMD	0x02
#define E_C_ERR_BAD_CFG		0x03
#define E_C_ERR_BAD_USD		0x04
#define E_USD_CTXT_DELETED	0x0a
#define E_USD_CTXT_ADDED	0x0b

#define E_USD_FILTER_INSTALLED	0x0c
#define E_C_ERR_INVALID_FILTER	0x0d


/*
 * Commands
 */

#define CMD_RING_ENTRIES	64

struct cmd {
#ifdef __LITTLE_ENDIAN
	u32	idx:12;
	u32	code:12;
	u32	evt:8;
#else
	u32	evt:8;
	u32	code:12;
	u32	idx:12;
#endif
};

/*
 * Commands definitions
 */

#define C_HOST_STATE		0x01
#define C_C_STACK_UP		0x01
#define C_C_STACK_DOWN		0x02
#define C_SET_PROMISC_MODE	0x0a
#define C_C_PROMISC_ENABLE	0x01
#define C_C_PROMISC_DISABLE	0x02
#define C_SET_MAC_ADDR		0x0c
#define C_SET_MULTICAST_MODE	0x0e
#define C_C_MCAST_ENABLE	0x01
#define C_C_MCAST_DISABLE	0x02
#define C_OPEN_USD_CONNECTION	0x15
#define C_CLOSE_USD_CONNECTION	0x16
#define C_USD_INSTALL_FILTER	0x17


/*
 * USD command scheme: 1 32-bit entry in the 2k shared memory window
 * for each channel, and the top 16 mailboxes are used to indicate
 * that a command has been sent.  These numbers must match the ones
 * in usd_conn.h in the firmware.
 */

#define USD_CMD_BASE            0x800


/*
 * This struct is shared with the NIC firmware.
 */
struct ring_ctrl {
	aceaddr	rngptr;
#ifdef __LITTLE_ENDIAN
	u16	flags;
	u16	max_len;
#else
	u16	max_len;
	u16	flags;
#endif
	u32	pad;
};


struct ace_mac_stats {
	u32 excess_colls;
	u32 coll_1;
	u32 coll_2;
	u32 coll_3;
	u32 coll_4;
	u32 coll_5;
	u32 coll_6;
	u32 coll_7;
	u32 coll_8;
	u32 coll_9;
	u32 coll_10;
	u32 coll_11;
	u32 coll_12;
	u32 coll_13;
	u32 coll_14;
	u32 coll_15;
	u32 late_coll;
	u32 defers;
	u32 crc_err;
	u32 underrun;
	u32 crs_err;
	u32 pad[3];
	u32 drop_ula;
	u32 drop_mc;
	u32 drop_fc;
	u32 drop_space;
	u32 coll;
	u32 kept_bc;
	u32 kept_mc;
	u32 kept_uc;
};

/* The General Info Area now that we're in USD happyland */
struct ace_connection_info {
    /* These parts must match the 'struct conn_ctxt_host_part' in the FW */
    aceaddr range_base;
    u32     range_length;
    u32     source_ip_addr;
    u32     dest_ip_addr;
#ifdef __LITTLE_ENDIAN
    u16     dest_port; /* TCP or UDP port number */
    u16     source_port;
#else
    u16     source_port; /* TCP or UDP port number */
    u16     dest_port;
#endif

    /* TX2/RX2 stuff */
#ifdef __LITTLE_ENDIAN
    u16      tx2_ring_size;          // must both be power of 2
    u16      rx2_ring_size;
#else
    u16      rx2_ring_size;
    u16      tx2_ring_size;
#endif
    u32      pad;
    aceaddr  tx2_ring_ptr;           // PA ptr to the TX2 ring
    aceaddr  tx2_csm_ptr;            // PA ptr to the TX2 consumer
    aceaddr  rx2_ring_ptr;           // PA ptr to the RX2 ring

    char     shared_end[0];
    /* Everything below here is only of interest to the Host */
    void  (*rx_irq_callback_fn)(void *); 
    void  (*tx_irq_callback_fn)(void *); 
    void  (*setup_callback_fn)(void *, int);
    void  (*teardown_callback_fn)(void *); 
    void   *callback_arg;
};

struct ace_info {
	union {
	    struct tg_stats { 
		/*
		 * MAC stats, taken from RFC 1643, ethernet-like MIB
		 */
		u32 dot3StatsAlignmentErrors;
		u32 dot3StatsFCSErrors;
		u32 dot3StatsSingleCollisionFrames;
		u32 dot3StatsMultipleCollisionFrames;
		u32 dot3StatsSQETestErrors;
		u32 dot3StatsDeferredTransmissions;
		u32 dot3StatsLateCollisions;
		u32 dot3StatsExcessiveCollisions;
		u32 dot3StatsInternalMacTransmitErrors;
		u32 dot3StatsCarrierSenseErrors;
		u32 dot3StatsFrameTooLongs;
		u32 dot3StatsInternalMacReceiveErrors;
		/*
		 * interface stats, taken from RFC 1213, MIB-II, 
		 * interfaces group
		 */
		u32 ifIndex;
		u32 ifType;
		u32 ifMtu;
		u32 ifSpeed;
		u32 ifAdminStatus;
#define IF_ADMIN_STATUS_UP	1
#define IF_ADMIN_STATUS_DOWN	2
#define IF_ADMIN_STATUS_TESTING	3
		u32 ifOperStatus;
#define IF_OPER_STATUS_UP	1
#define IF_OPER_STATUS_DOWN	2
#define IF_OPER_STATUS_TESTING	3
#define IF_OPER_STATUS_UNKNOWN	4
#define IF_OPER_STATUS_DORMANT	5
		u32 ifLastChange;
		u32 ifInDiscards;
		u32 ifInErrors;
		u32 ifInUnknownProtos;
		u32 ifOutDiscards;
		u32 ifOutErrors;
		u32 ifOutQLen;	/* deprecated */
	        u32 ifPhysAddressHi;
	        u32 ifPhysAddressLo;
		u8  ifDescr[32];
		u32 alignIt;   	/* align to 64 bit for U64s following */
		/*
		 * more interface stats, taken from RFC 1573, MIB-IIupdate, 
		 * interfaces group
		 */
		u64 ifHCInOctets;
		u64 ifHCInUcastPkts;
		u64 ifHCInMulticastPkts;
		u64 ifHCInBroadcastPkts;
		u64 ifHCOutOctets;
		u64 ifHCOutUcastPkts;
		u64 ifHCOutMulticastPkts;
		u64 ifHCOutBroadcastPkts;
		u32 ifLinkUpDownTrapEnable;
		u32 ifHighSpeed;
		u32 ifPromiscuousMode; 
		u32 ifConnectorPresent; /* follow link state */
		/*
		 * pad out to 1024 bytes...
		 */
		u32 res[200];
	    } nic_stats;

	    u32 stats[256];
	} s;
	struct ring_ctrl	evt_ctrl;
	struct ring_ctrl	cmd_ctrl;
        aceaddr evt_prd_ptr;
        aceaddr stats2_ptr;
        aceaddr connection_zero;
};


#ifdef __KERNEL__

/*
 * Struct private for the AceNIC.
 */

struct ace_private
{
	struct ace_regs		*regs;		/* register base */
	volatile __u32		*sgt;
	struct sk_buff		*pkt_buf;	/* Receive buffer */

	struct event		evt_ring[EVT_RING_ENTRIES];
	struct ace_info		*info;

	spinlock_t		lock;
        volatile u32	        harvest_lock;  /* used on tx */

        volatile u32            filter_lock;   /* used on connection deln */
        int                     filter_id;

	volatile u32		evt_prd
				__attribute__ ((aligned (L1_CACHE_BYTES)));
	struct net_device	*next
				__attribute__ ((aligned (L1_CACHE_BYTES)));
        volatile u_long         in_interrupt;
	unsigned char		*trace_buf;
	int			fw_running, fw_up, promisc, mcast_all;
	int			version;
	int			flags;
	u16			vendor;
	u16			pci_command;
	struct pci_dev		*pdev;
	char			name[24];

	struct net_device_stats stats;

        /* State for setup and teardown of USD channels */
        struct ace_connection_info *conn_state[USD_CHANNELS];

        /* TX2/RX2 endpoint for default kernel traffic */
        u32			tx2_full;
        u16    c0_tx_producer, c0_old_tx_consumer;
	struct sk_buff	       *c0_tx2_skbuff[TX2_RING_ENTRIES];

        u32    tx_int_mask [(USD_CHANNELS+31)/32];

        u16    c0_free_producer, c0_rx_consumer;

        u32    rx_int_mask [(USD_CHANNELS+31)/32];

	struct sk_buff	       *c0_rx2_skbuff[RX2_RING_ENTRIES];

        struct usd_endpoint     chan0;

    /* XXX debug vars */
	struct timer_list	watch_timer;
        u16    DEBUG_last_irq_tx_cons;
};

#endif /* __KERNEL__ */

#endif /* _ACENIC_H_ */
