/*
 * COPYRIGHT NOTICE
 * Copyright (c) Alteon Networks, Inc. 1996
 * All rights reserved
 */
/*
 * FILE ring.h
 *
 * $Source: /Nfs/stives/grp56/gige/src-current/acenic-fw-12.3.10/nic/fw2/common/RCS/ring.h,v $
 * $Revision: 1.11 $ $Locker:  $
 * $Date: 2000/01/27 18:07:40 $ $Author: kaf24 $ $State: Exp $
 */

#ifndef _RING_H_
#define _RING_H_

#include "timer.h"      /* for TG_TICK_XX */
#include "dma.h"


/* timeout for must_enq_event() */
#define ONE_MSEC 1000


__inline__ static U32
get_next_event_index(U32 index)
{
    return ((index == (EVENT_RING_ENTRIES - 1))? 0: (index + 1));
}

__inline__ static U32
enq_event(struct tg_event *ep)
{
    U32 event_prod, event_prod_next;

    event_prod = nicfp->event_prod_index;
    event_prod_next = get_next_event_index(event_prod);

    if (event_prod_next == tsp->gen_com.event_cons_index) {
        nicfp->stats.nicEventRingFull++;

        NIC_UTRACE("evtRngF", 0x90100,
	    ep->TG_EVENT_w0,
	    event_prod,
	    tsp->gen_com.event_cons_index,
	    0);
        return (ERROR);
    }

    /* copy event to ring */
    nicp->event_ring[event_prod] = *ep; /* struct copy */

    /* dma event to the host */
    if (!(*nicfp->q_dma_to_host_stub)
            (nicfp->event_ring_ptr + (event_prod * sizeof(tg_event_t)),
            (U32)&nicp->event_ring[event_prod],
            sizeof(tg_event_t),
            TIGON_TYPE_EVENT,
	    0,
            event_prod_next, /* dummy */
            nicfp->dma_to_host_bd_state)) {

        NIC_UTRACE("QevtF", 0x90200,
	    ep->TG_EVENT_w0,
	    event_prod,
	    trp->host_dma_assist.write_chan_producer,
	    trp->host_dma_assist.write_chan_consumer);
        return (ERROR);
    }

    /* update producer */
    nicfp->event_prod_index = event_prod_next;

    /* send the new producer to the host */
    if (!q_dma_to_host_index
	    (nicfp->event_producer_ptr,
	    sizeof(U64),
	    TIGON_TYPE_EVENT_PROD,
            0,
	    event_prod_next,
	    nicfp->dma_to_host_bd_state)) {

        nicfp->stats.nicEventProducerRingFull++;

        NIC_UTRACE("QevtP_F", 0x90300,
	    ep->TG_EVENT_w0,
	    event_prod,
	    trp->host_dma_assist.write_chan_producer,
	    trp->host_dma_assist.write_chan_consumer);
        return (ERROR);
    }

    NIC_TRACE(TRACE_TYPE_EVENT, "QevtOk", 0x90400,
	ep->TG_EVENT_w0,
	event_prod,
	tsp->gen_com.event_cons_index,
	0);

    return (OK);
}

/*
 * common routine for must_enq_event() and must_enq_interrupt_event()
 */
__inline__ static U32
must_enq_event_2(struct tg_event *ep, bool intr_host)
{
    U32 event_prod, event_prod_next;
    U32 end_time;
    bool successQ;

    event_prod = nicfp->event_prod_index;
    event_prod_next = get_next_event_index(event_prod);

    if (event_prod_next == tsp->gen_com.event_cons_index) {
        nicfp->stats.nicEventRingFull++;

        NIC_UTRACE("MevtRngF", 0x90500,
	    ep->TG_EVENT_w0,
	    event_prod,
	    tsp->gen_com.event_cons_index,
	    0);
        return (ERROR);
    }

    /* copy event in ring */
    nicp->event_ring[event_prod] = *ep; /* struct copy */

    /* dma event to the host */
    end_time = trp->gen_control.timer + ONE_MSEC;
    successQ = FALSE;

    /* computation way is to handle the timer wrapping around */
    while (end_time - trp->gen_control.timer <= ONE_MSEC) {
        if ((*nicfp->q_dma_to_host_stub)
                (nicfp->event_ring_ptr + (event_prod * sizeof(tg_event_t)),
                (U32)&nicp->event_ring[event_prod],
                sizeof(tg_event_t),
		TIGON_TYPE_EVENT, 0,
                event_prod_next, /* dummy */
		nicfp->dma_to_host_bd_state)) {

            successQ = TRUE;
            break;
        }
    }

    if (!successQ) {
	nicfp->stats.nicEnqEventDelayed++;

        NIC_UTRACE("MQevtF", 0x90600,
	    ep->TG_EVENT_w0,
	    event_prod,
	    trp->host_dma_assist.write_chan_producer,
	    trp->host_dma_assist.write_chan_consumer);
        return (ERROR);
    }

    /* update producer */
    nicfp->event_prod_index = event_prod_next;

    /* send the new producer to the host */
    end_time = trp->gen_control.timer + ONE_MSEC;
    successQ = FALSE;

    /* computation way is to handle the timer wrapping around */
    while (end_time - trp->gen_control.timer <= ONE_MSEC) {
        if (q_dma_to_host_index
		(nicfp->event_producer_ptr,
		 sizeof(U64),
		 TIGON_TYPE_EVENT_PROD, 0,
		 event_prod_next,
		 (intr_host)?
		    (nicfp->dma_to_host_bd_state | TG_DMA_STATE_SET_INTERRUPT):
		    nicfp->dma_to_host_bd_state)) {

	    nicfp->stats.nicInterrupts++;
            successQ = TRUE;
            break;
        }
    }

    if (!successQ) {
	nicfp->stats.nicEnqEventDelayed++;

        NIC_UTRACE("MQevP_F", 0x90700,
	    ep->TG_EVENT_w0,
	    event_prod,
	    trp->host_dma_assist.write_chan_producer,
	    trp->host_dma_assist.write_chan_consumer);
        return (ERROR);
    }

    NIC_TRACE(TRACE_TYPE_EVENT, "MQevtOk", 0x90800,
	ep->TG_EVENT_w0,
	event_prod,
	tsp->gen_com.event_cons_index,
	0);

    return (OK);
}

/*
 * enqueue an event 
 * no interrupt is generated when this dma completes
 */
__inline__ static U32
must_enq_event(struct tg_event *ep)
{
    return (must_enq_event_2(ep, FALSE));
}

/*
 * enqueue an event and interrupt host when this dma completes
 */
__inline__ static U32
must_enq_interrupt_event(struct tg_event *ep)
{
    return (must_enq_event_2(ep, TRUE));
}

/*
 * Misc. functions
 */


__inline__ static void
set_event_handler(U32 event_num, U32 func)
{
    ev_handler[(event_num + 1) << 1] =
	(0x08 << 24) | ((0x0fffffff & (func)) >> 2);
    return;
}
/*
 * lock among processors
 */

__inline__ static U32 
get_cpu_id(void)
{
    return (nicfp->cpu_id);
}

__inline__ static void 
set_cpu_id(U32 id)
{
    nicfp->cpu_id = id;
    return;
}

__inline__ static void
grab_spin_lock(void)
{
    if (get_cpu_id()) {
        do {
            trp->gen_control.semaphore_b = 0;
        } while (! trp->gen_control.semaphore_b);
    } else {
        do {
            trp->gen_control.semaphore_a = 0;
        } while (! trp->gen_control.semaphore_a);
    }
    return;
}

__inline__ static bool
try_spin_lock(void)
{
    if (get_cpu_id()) {
	trp->gen_control.semaphore_b = 0;
        return ((trp->gen_control.semaphore_b != 0));
    } else {
	trp->gen_control.semaphore_a = 0;
        return ((trp->gen_control.semaphore_a != 0));
    }
}

__inline__ static void
free_spin_lock(void)
{
    if (get_cpu_id()) {
	if (trp->gen_control.semaphore_b)
	    trp->gen_control.semaphore_b = 0;
    } else {
	if (trp->gen_control.semaphore_a)
	    trp->gen_control.semaphore_a = 0;
    }
    return;
}

__inline__ static void
clear_event_reg(U32 bits)
{
    grab_spin_lock();
    trp->gen_control.event &= ~bits;
    free_spin_lock();
    return;
}

__inline__ static void
set_event_reg(U32 bits)
{
    grab_spin_lock();
    trp->gen_control.event |= bits;
    free_spin_lock();
    return;
}

__inline__ static void
clear_event_reg_b(U32 bits)
{
    grab_spin_lock();
    trp->gen_control.event_b &= ~bits;
    free_spin_lock();
    return;
}

__inline__ static void
set_event_reg_b(U32 bits)
{
    grab_spin_lock();
    trp->gen_control.event_b |= bits;
    free_spin_lock();
    return;
}

/* Indivisible counter operations. */
#define inc_count(c)  \
({                    \
    typeof (c) __r;   \
    grab_spin_lock(); \
    __r = ++(c);      \
    free_spin_lock(); \
    __r;              \
})

#define dec_count(c)  \
({                    \
    typeof (c) __r;   \
    grab_spin_lock(); \
    __r = --(c);      \
    free_spin_lock(); \
    __r;              \
})

#endif /* _RING_H_ */
