/******************************************************************************
 * event.h
 * 
 * An event mechanism for communication between the two processors.
 * 
 * Copyright (c) 2000, University of Cambridge
 *
 * Please read the FIRMWARE-LICENSE file distributed with this source file
 * for details of the license agreement.
 */

/* Event masks and sizes */
#define EVENT_ID_MASK              (15<<28)
#define EVENT_DATA_MASK            (~EVENT_ID_MASK)

/* Event definitions -- for CPU A */
#define EVA_ADD_CONTEXT_FINAL      (0<<28)
#define EVA_DEL_CONTEXT_FINAL      (1<<28)

/* Event definitions -- for CPU B */
#define EVB_ADD_CONTEXT            (0<<28)
#define EVB_DEL_CONTEXT            (1<<28)
#define EVB_SET_TX_BW              (2<<28)

/* Event handlers */
void event_handler_A(void);
void event_handler_B(void);

/* Event constructors */
void send_event_to_A(U32 event);
void send_event_to_B(U32 event);
void send_event_to_A_with_args(U32 event, U32 *args, int numargs);
void send_event_to_B_with_args(U32 event, U32 *args, int numargs);


/******************************************************************************
 * Send events with no arguments.
 */

/* Send an event to CPU A, from CPU B */
extern __inline__ void send_event_to_A(U32 event) 
{
    U16 prod = nicp->event_ring_a.prod;

    while ( prod == (nicp->event_ring_a.cons + CPU_EVENT_RING_SIZE) )
    {
        /*
         * While waiting for other CPU to clear space in his event ring, we
         * should make sure ours is kept empty. Otherwise there would be a
         * deadlock possibility if both rings filled at the same time.
         * 
         * Obviously, spinning really sucks. But it is highly unlikely that
         * these event rings will ever fill, and if they do then they should
         * be emptied pretty swiftly (high priority s/w events)
         */
        event_handler_B();
    }
    nicp->event_ring_a.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = event;
    
    nicp->event_ring_a.prod = prod;
    set_event_reg(TG_FW_EVENT_USD_EVENT_A);
}


/* Send an event to CPU B, from CPU A */
extern __inline__ void send_event_to_B(U32 event)
{
    U16 prod = nicp->event_ring_b.prod;

    while ( prod == (nicp->event_ring_b.cons + CPU_EVENT_RING_SIZE) )
    {
        /* See comment in `send_event_to_A' */
        event_handler_A();
    }
    nicp->event_ring_b.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = event;

    nicp->event_ring_b.prod = prod;
    set_event_reg_b(TG_FW_EVENT_USD_EVENT_B);
}


/******************************************************************************
 * Send events with arguments.
 */

/* Send an event to CPU A, from CPU B */
extern __inline__ void send_event_to_A_with_args(U32 event,
                                                 U32 *args, int numargs)
{
    U16 prod = nicp->event_ring_a.prod;
    int i = 0;

    while ( prod == (nicp->event_ring_a.cons + CPU_EVENT_RING_SIZE) )
    {
        /*
         * While waiting for other CPU to clear space in his event ring, we
         * should make sure ours is kept empty. Otherwise there would be a
         * deadlock possibility if both rings filled at the same time.
         * 
         * Obviously, spinning really sucks. But it is highly unlikely that
         * these event rings will ever fill, and if they do then they should
         * be emptied pretty swiftly (high priority s/w events)
         */
        event_handler_B();
    }
    nicp->event_ring_a.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = event;
    
    while ( numargs-- )
    {
        while ( prod == (nicp->event_ring_a.cons + CPU_EVENT_RING_SIZE) )
        {
            event_handler_B();
        }
        nicp->event_ring_a.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = args[i++];
    }

    nicp->event_ring_a.prod = prod;
    set_event_reg(TG_FW_EVENT_USD_EVENT_A);
}


/* Send an event to CPU B, from CPU A */
extern __inline__ void send_event_to_B_with_args(U32 event, 
                                                 U32 *args, int numargs)
{
    U16 prod = nicp->event_ring_b.prod;
    int i = 0;

    while ( prod == (nicp->event_ring_b.cons + CPU_EVENT_RING_SIZE) )
    {
        /* See comment in `send_event_to_A' */
        event_handler_A();
    }
    nicp->event_ring_b.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = event;

    while ( numargs-- )
    {
        while ( prod == (nicp->event_ring_b.cons + CPU_EVENT_RING_SIZE) )
        {
            event_handler_A();
        }
        nicp->event_ring_b.ring[prod++ & (CPU_EVENT_RING_SIZE-1)] = args[i++];
    }

    nicp->event_ring_b.prod = prod;
    set_event_reg_b(TG_FW_EVENT_USD_EVENT_B);
}
