//
// Kiwi Scientific Acceleration
// (C) 2009 DJ Greaves - University of Cambridge, Computer Laboratory
//
// Demonstration of driving the 16x2 LCD display panel found on FPGA development boards.
//

using System;
using KiwiSystem;

class KiwiLCD1602Driver
{

/*
   // Put the following padring code in a Xilinx top-level file.
   // Instance of the Kiwi-generated LCD driver
   wire lcd_data4_openable; 
   wire [3:0] lcd_data4_in, lcd_data4_out;
   bufif1(LCD_DB4_LS, lcd_data4_out[0], lcd_data4_openable);  
   bufif1(LCD_DB5_LS, lcd_data4_out[1], lcd_data4_openable);  
   bufif1(LCD_DB6_LS, lcd_data4_out[2], lcd_data4_openable);  
   bufif1(LCD_DB7_LS, lcd_data4_out[3], lcd_data4_openable);  
   IBUF lcd_in40(lcd_data4_in[0], LCD_DB4_LS);
   IBUF lcd_in41(lcd_data4_in[1], LCD_DB5_LS);
   IBUF lcd_in42(lcd_data4_in[2], LCD_DB6_LS);
   IBUF lcd_in43(lcd_data4_in[3], LCD_DB7_LS);
   
   kiwi_lcd1602_driver  the_lcd_driver(  
			 .LCD_RS_LS(LCD_RS_LS),
			 .LCD_E_LS(LCD_E_LS),
			 .LCD_RW_LS(LCD_RW_LS),
			 .lcd_data4_in(lcd_data4_in),
			 .lcd_data4_out(lcd_data4_out),
			 .lcd_data4_openable(lcd_data4_openable)
			
   );
   

*/

// The standard 1602a DisplayTech LCD is documented here  https://www.openhacks.com/uploadsproductos/eone-1602a1.pdf
// http://embeddedlifehelp.blogspot.co.uk/2012/03/16x2-lcd-programming-for-beginners-made.html
//  https://learningmsp430.wordpress.com/2013/11/16/16x2-lcd-interfacing-in-4-bit-mode/

  // Use 4-bit, tri-state connection to the LCD panel.
  [Kiwi.OutputBitPort("LCD_RS_LS")] static bool LCD_RS;    
  [Kiwi.OutputBitPort("LCD_RW_LS")] static bool LCD_RW;   
  [Kiwi.OutputBitPort("LCD_E_LS")] static bool LCD_E;    
  [Kiwi.OutputBitPort("lcd_data4_openable")] static bool lcd_data4_openable;    
  [Kiwi.InputWordPort(3, 0, "lcd_data4_in")] static byte lcd_data4_in;
  [Kiwi.OutputWordPort(3, 0, "lcd_data4_out")] static byte lcd_data4_out;

  public static void PortIdle()
  { 
    lcd_data4_openable = true;
    LCD_RS = false;
    LCD_RW = false;
    LCD_E = false;
    lcd_data4_out = 0;
  }

  // To send a byte on the 4-bit bus, send high nibble first.
  // To read a byte on the 4-bit bus, get high nibble first.
  static byte readByte(bool dataf)
  {
    LCD_RS = dataf; 	// Low for control, high for readback of data.
    lcd_data4_openable = false;    LCD_RW = true;   
    Kiwi.Pause();    LCD_E = true;
    Kiwi.Pause();    byte r = (byte)(lcd_data4_in & 0xF);  LCD_E = false;
    Kiwi.Pause();    LCD_E = true;
    Kiwi.Pause();    r =  (byte)((r << 4) | (lcd_data4_in & 0xF)) ;    LCD_E = false;
    Kiwi.Pause();
    lcd_data4_openable = true;  // Re-enable outputs for next write.
    return r;
  }

  // To send a byte on the 4-bit bus, send high nibble first.
  static void sendByte(bool dataf, byte data)
  {
    LCD_RS = dataf; 	// Low for control, high for readback of data.
    lcd_data4_openable = true;     LCD_RW = false;    lcd_data4_out = (byte)((data >> 4) & 0xF);
    Kiwi.Pause();    LCD_E = true;
    Kiwi.Pause();    LCD_E = false;
    Kiwi.Pause();    lcd_data4_out = (byte)((data >> 0) & 0xF) ;  
    Kiwi.Pause();    LCD_E = true;
    Kiwi.Pause();    LCD_E = false;
    Kiwi.Pause();  
  }


  static void WaitNotBusy()
  {
    // Have two pauses to give it a chance to go busy.
    Kiwi.Pause();
    Kiwi.Pause();
    // Busy flag is bit seven of the status register - poll to be not busy.
    while (true)
      {
	byte status_reg = readByte(false); 
	Kiwi.Pause();
	if (((int)(status_reg) & 128)==0) break;
      }
  }

  static void wait_delay_40() 
  {
    for (int k=5; k != 0; k--) Kiwi.Pause();   // Wait 5*period for 4.1 ms or 100 us
  }

  static void LcdCmdD(byte cmd, int delay)      // Send a command byte using a delay.
  { sendByte(false, cmd);
    Console.WriteLine("LCD sent cmd byte using delay timing alue={0:X}", cmd);
  }

  static void LcdCmd(byte cmd)      // Send a command byte using polled busy.
  {
    WaitNotBusy();    
    wait_delay_40();
    sendByte(false, cmd);
    Console.WriteLine("LCD sent cmd byte value={0:X}", cmd);
  }

  public static void sendDataByte(byte data)
  {
    WaitNotBusy();    

    wait_delay_40();
    sendByte(true, data);
    Console.WriteLine("LCD sent data byte (datareg={1}) value={0:X}", data, LCD_RS);
  }


  public static void Reset()
  {
    lcd_data4_openable = false;
    Kiwi.Pause();
    PortIdle();
    wait_delay_40();
	// Wait >4.1ms and >100 us in the gaps between the first 2 '3' operations, then used BF polling instead of delays.
    LcdCmdD(0x33, 2000); // Reset code 30
    wait_delay_40();

    LcdCmdD(0x32, 2000); // Reset code 20
    wait_delay_40();

    LcdCmdD(0x28, 2000);   // 4-bit interface, 2 lines.
    wait_delay_40();
  }


  public static void Setup()
  { 
    LcdCmd(0x06);   // Entry mode, ID=1, S=0;
 //     wait_delay_40();

    //LcdCmd(0x0c);   // Display: disp on, cursor off, blink off
    LcdCmd(0x0f);   // Display: disp on, cursor on, blink on.
 //     wait_delay_40();

    LcdCmd(0x01);   // Home
 //     wait_delay_40();

    LcdCmd(0x80);   // Set data data RAM location 0 of top line write
 //     wait_delay_40();
  }

  public static void WriteString(string msg)
  {
    char [] cdata = msg.ToCharArray();
    for (int x=0; x<msg.Length; x++)      sendDataByte((byte)cdata[x]);
  }
}


class tester
{

  [Kiwi.OutputBitPort("done")] static bool done;  
  [Kiwi.InputBitPort("select")] static bool select;    
  [Kiwi.OutputWordPort("codeword")] static int codeword;    

  
  public static void Main()
  {
    select = false;
    RunHW();
  }

  [Kiwi.HardwareEntryPoint()]
  public static void RunHW()
  {
    codeword = 5;
    done = false;
    Console.WriteLine("Hello from LCD 1602 driver");
    Kiwi.Pause();
    Kiwi.Pause();
    Kiwi.Pause();
    KiwiLCD1602Driver.PortIdle();    
    codeword = 6;
    KiwiLCD1602Driver.Reset();
    codeword = 7;
    KiwiLCD1602Driver.Setup();

    Kiwi.Pause();
    codeword = 8;

    string msg = (select) ? "Hello World": "David Greaves";
    KiwiLCD1602Driver.WriteString(msg);

    Kiwi.Pause();
    codeword = 7;

    Kiwi.Pause();
    codeword = 9;
    while(true)
      {
	done = true;
	Kiwi.Pause();
    codeword = 10;
      }
  }
}


// eof



