// cbguart1.v  (C) 1997 Mixerton ST
// +    
// + This file contains:
// +   A very simple UART with 
// +     8 bits of data 
// +     no parity options
// +     one stop bit
// +     Transmission of bit on D0 (lsb) first
// +


//
// This design could/should be optimised to allow greater throughput.
//

module CBGUART1
  (
   input 	reset,
   input 	baudck, // Input at 16 times the baud rate (153.6 KHz for 9600b = 50e6/325.5)
   input 	clk, // Bus clock.

   // Serial connections   
   input 	rxdata, // RX data, polarity is such that start bit is a zero.
   output 	txdata,
   input 	cts,
   output 	rts,


   input 	rx_read, // Parallel data transfer qualifier
   output [7:0] rx_parout,
   output 	rx_av,
   
   
   input 	tx_write, // Parallel data transfer qualifier
   input [7:0] 	tx_parin,
   output 	tx_av
   );
   
  UARTRX uartrx(baudck, rxdata, rx_av, rx_read, rx_parout, clk, reset);
  UARTTX uarttx(baudck, txdata, tx_av, tx_write, tx_parin, clk, reset);

endmodule
//
//
//===========================================================
//
//

module UARTRX(baudck, rxdata, rxav, read, parout, busck, reset);

  input baudck;   // Input at 16 times the baud rate (153.6 KHz = 9600b)
  input rxdata;   // RX data, polarity is such that start bit is a zero.
  input reset;

  output rxav;

  input busck;		// Bus clock.
  input read;		// Data transfer qualifier 

  output [7:0] parout;

//-----------------------------
// Composite input decision flip-flop

  reg dec_a, dec_b, dec;

  always @(posedge baudck)
	begin
	dec_a <= rxdata;
	dec_b <= dec_a;
	dec <= (dec_a & dec_b) | (dec & ~(dec_a | dec_b));
	end

//-----------------------------
// Input sequencer.
// 
// The sequencer starts at 0 and counts 16 baudck periods per bit
// for a total of 9.5 bits (152 cycles).
//


  reg [3:0] rseq;
  reg running;
  reg [3:0] phase;

  wire startbit = ~running & ~dec;
  wire sample = (rseq == 4'd6);
  wire false_start = sample & (phase == 0) & dec;

  wire stopbit = sample & (phase == 4'd9);
  wire next_is_stopbit = sample & (phase == 4'd8);
  always @(posedge baudck)
	begin
	running <= startbit | (running & ~(stopbit | false_start));
	rseq <= (running) ? rseq+1'd1: 4'd0;
	if (sample | ~running) phase <= (running) ? phase + 4'd1: 4'd0;
	end

//---------------------------------------
// Serial to parallel conversion
// Shift into msb to handle lsb first.

  reg [7:0] rxshift;

  always @(posedge baudck) if (sample) rxshift <= { dec, rxshift[7:1] };

//---------------------------------------
// Output holding register

  reg [7:0] parout;

  always @(posedge baudck) if (stopbit) parout <= rxshift;

//---------------------------------------
// rxav and place for overrun logic


  URXSYNCH rxsynch(.inck(baudck), .write(sample & stopbit), 
	.reset(reset), .outck(busck), .rxav(rxav), .outt(read));
 
endmodule
//
//
//===========================================================
//
//

module UARTTX(baudck, txdata, txav, write, parin, busck, reset);

  input baudck;   	// Input at 16 times the baud rate (153.6 KHz = 9600b)
  output txdata;
  input reset;
  output txav;

  input busck;		// Bus clock for parallel transfers.
  input write;		// Data transfer qualifier 

  input [7:0] parin;
//----------------------------------------------------------
// Free running divide by 16 to get actual baud rate clock
  reg  [3:0] t_div16;
  always @(posedge baudck) t_div16 <= t_div16 + 4'd1;
  wire txck = t_div16[3];
//----------------------------------------------------------
// Input holding register

  reg [7:0] txhold;

  wire start;		// Start pulse to uart tx logic
  wire idle;		// Idle signal from tx logic
  wire ready;		// High when able to take another byte
  always @(posedge busck) if (write) txhold <= parin;



  UTXSYNCH txsynch(.inck(busck), .write(write), .av(txav), .reset(reset), 
		.outck(txck), .start(start), .idle(ready));
  
//----------------------------------------------------------
// Divide by ten tx framer - stays at zero when idle.

  reg [3:0] txframe;
  
  assign idle = (txframe == 4'd0);

  assign ready = idle; 
  // No pipelines in the holding reg supported, since ready is simply
  // tied to idle.

  wire move = start | ~idle;
  wire limit = (txframe == 4'd9);
  always @(posedge txck) if (move) txframe <= (limit) ? 4'd0 : txframe + 4'd1;

//----------------------------------------------------------
// Serial to parallel.  Shift right to send lsb first

  reg [8:0] txshift;
  wire [8:0] newd;
  assign newd[8:1] = txhold;
  assign newd[0] = 0;  // This is the start bit.
  always @(posedge txck) 
	begin
	txshift = (start & idle) ? newd: { 1'd1, txshift[8:1] };
	if (reset) txshift = 9'h1FF;
	end

  assign txdata = txshift[0] | idle | reset;

endmodule 
//
//
//===========================================================
//
// Clock domain crossing
module UTXSYNCH(inck, write, av, outck, start, idle, reset);

// Do not glitch idle while out-clock is low.

   input reset;           // Epynonymous input
   input inck;		// Data src clock
   input outck;		// Data sink clock
   input write;		// Data src write stobe 
   output av;		// Not logged output to src
   input  idle;		// Not busy input from sink
   output start;		// Start signal to sink;
   
   reg 	out_tog, s_in_tog;
   reg in_tog, old_tog;
   reg oldidle;
   
   
   always @(posedge outck) 
     begin
	oldidle <= idle;
	out_tog = out_tog ^ (~oldidle & idle);
     end
   
   always @(posedge inck) begin
      in_tog = in_tog ^ write;
   end
   
   assign av = ~(out_tog ^ in_tog);
   
   always @(posedge outck) begin	
      s_in_tog <= in_tog;
      old_tog <= s_in_tog;
   end

  assign start = old_tog ^ s_in_tog;
  
endmodule
//
//
//===========================================================
//
// Clock domain crossing for received data
//
module URXSYNCH(inck, write, outck, rxav, outt, reset);

// Do not glitch idle while out-clock is low.

 input reset;           // Epynonymous input
 input inck;		// Data src clock
 input outck;		// Data sink clock
 input write;		// Data src write stobe 
 output rxav;		// Not logged output to src
 input outt;		// Read request

 reg out_tog;
 reg in_tog;


 always @(posedge outck) begin
    out_tog = out_tog ^ (outt & rxav);
 end

 always @(posedge inck) begin
    in_tog <= in_tog ^ write;
 end

 assign rxav = ~(out_tog ^ in_tog);
  
endmodule
//
//
//===========================================================
//
// end of cbg uart.cv



`ifdef NOTDEF
// uart test wrapper
module UART_SIMSYS();

  wire baudclk, txdata, rxav, txav, write, busclk;

  reg read, rxdata;

  wire [7:0] parin, parout;
  wire reset;


  CLK5MHz(busclk);

  CLK1MHz(baudclk);


  reg [7:0] datasrc;
  reg go;
  always @(posedge busclk)
	begin
	if (go)	datasrc <= datasrc + 123;
	
	go <= ~go & txav;
	read <= rxav;	
	
	end

  always @(posedge baudclk) rxdata <= txdata;



  assign write = go;
  assign parin = datasrc;

  RESET(reset);

  UART uart(baudclk, rxdata, txdata, rxav, txav, write, read, parin, parout,
  busclk, reset);




endmodule
`endif // unmatched `else, `elsif or `endif

// eof


