package uk.org.pjt.jmaxellanotostream;

import java.io.IOException;
import java.io.InputStream;
import java.util.TooManyListenersException;
import java.util.logging.Logger;

import javax.comm.CommPortIdentifier;
import javax.comm.NoSuchPortException;
import javax.comm.PortInUseException;
import javax.comm.SerialPort;
import javax.comm.SerialPortEvent;
import javax.comm.SerialPortEventListener;
import javax.comm.UnsupportedCommOperationException;


/**
 * Modified by Phil Tuddenham from parts of the R3 toolkit 
 * (Copyright Stanford University, written by Ron Yeh and available 
 * from http://hci.stanford.edu/paper/ under the BSD licence).
 */
public class PenStreamingConnection implements SerialPortEventListener {

	public int x, y, force;
	
	private static final Logger logger = Logger.getLogger("uk.org.pjt.jmaxellanotostream");
	
	private static enum StreamingField {
		HEADER, SKIP16, X, Y, FORCE
	}
	
	// PenUP Identifier
	private static final byte ID_PEN_UP = 0x04;

	// SimpleCoord Identifier
	private static final byte ID_SIMPLE_COORD = 0x03;

	// length of the PenUP Packet
	private static final byte LENGTH_PEN_UP = 0x00;

	// length of the Simple Coordinate Packet
	private static final byte LENGTH_SIMPLE_COORD = 0x15;



	private byte bCurrent;
	private byte bLast;
	private byte bLastLast;
	private StreamingField nextUp = StreamingField.HEADER;
	private int numBytesCoord;
	private boolean penUp=true;

	private final PenListener pl;
	private final SerialPort serialPort;
	private final InputStream inputStream;
	private final byte[] readBuffer;
	private final boolean skipIntermediatePackets, pollMode;

	
	/**
	 * If skipIntermediatePackets is true then for any number of successive samples
	 * that are in the queue when it is polled, only the most recent sample will be
	 * reported. Samples interleaved with pen-ups are still reported. So for example,
	 * (pen is down to begin with) SSSSUSSUSS is reported as USUS. Another example,
	 * (pen is up to begin with) SSSSUSSUSS is reported as SUSUS
	 * @param port
	 * @param pl
	 * @param skipIntermediatePackets
	 * @param pollMode
	 * @return
	 * @throws NoSuchPortException
	 * @throws PortInUseException
	 * @throws IOException
	 * @throws TooManyListenersException
	 * @throws UnsupportedCommOperationException
	 */
	public static PenStreamingConnection createInstanceFromPortName(String port, PenListener pl, boolean skipIntermediatePackets, boolean pollMode) 
	throws NoSuchPortException, PortInUseException, IOException, TooManyListenersException, UnsupportedCommOperationException
	{
		if(!CommPortIdentifier.getPortIdentifiers().hasMoreElements()) {
			throw new IllegalStateException(
					"No serial or parallel ports found. "
					+"This is probably because javax.comm cannot find the files it needs. "
					+"This is a well known problem and it tends to look in silly places for them. "
					+" Please check your javax.comm installation as follows:  "
					+"1. win32comm.dll should be in the working directory when the java program runs.  " 
					+"2. javax.comm.properties should be in the same directory as comm.jar, "
					+"or if that doesn't work, put the properties file in Java's jre/lib directory.");
		}
		CommPortIdentifier portID = (CommPortIdentifier) CommPortIdentifier.getPortIdentifier(port);
		if(portID.getPortType() != CommPortIdentifier.PORT_SERIAL) {
			throw new IllegalStateException("Port was found but is not a serial port");
		}	
		return new PenStreamingConnection(portID, pl, skipIntermediatePackets, pollMode);
	}
	
	
	public PenStreamingConnection(CommPortIdentifier portID, PenListener pl, boolean skipIntermediatePackets, boolean pollMode) 
	throws PortInUseException, IOException, TooManyListenersException, UnsupportedCommOperationException
	{
		this.pl = pl;
		this.serialPort = (SerialPort) portID.open("StreamingPen", 2000);
		this.serialPort.setSerialPortParams(
				9600, SerialPort.DATABITS_8, SerialPort.STOPBITS_1,	SerialPort.PARITY_NONE
			);
		this.inputStream = serialPort.getInputStream();
		this.readBuffer = new byte[500];
		this.skipIntermediatePackets = skipIntermediatePackets;
		this.pollMode = pollMode;
		if(!pollMode) {
			this.serialPort.addEventListener(this);
			this.serialPort.notifyOnDataAvailable(true);
		} else {
			// nothing
		}
	}

	public void close() {
		this.serialPort.close();
	}
	
	
	/**
	 * We process one byte at a time.
	 * 
	 * @param streamedByte
	 */
	private void handleByte(byte streamedByte) {

		// we got a new byte, so we push the others back
		bLastLast = bLast;
		bLast = bCurrent;
		bCurrent = streamedByte;

		//System.out.println(bCurrent);
		//System.out.println(nextUp);

		// looking for the header portion of the data
		if (nextUp == StreamingField.HEADER) {
			if (bCurrent == LENGTH_SIMPLE_COORD && bLast == 0x00 && bLastLast == ID_SIMPLE_COORD) {
				// System.out.print("SAMPLE: ");
				// we are now in the sample mode
				nextUp = StreamingField.SKIP16;
				numBytesCoord = 0;
			} else if (bCurrent == LENGTH_PEN_UP && bLast == 0x00 && bLastLast == ID_PEN_UP) {
				// System.out.println("PEN UP");
				this.pl.penUp();
				penUp = true;
			}
		} else if (nextUp == StreamingField.SKIP16) {
			// skip 16 bytes
			numBytesCoord++;
			if (numBytesCoord == 16) {
				nextUp = StreamingField.X;
				numBytesCoord = 0;
			}
			
		} else if (nextUp == StreamingField.X) { // 2 bytes long, X Coordinate
			numBytesCoord++;
			x = x << 8; // shift left by one byte
			x = x | (bCurrent & 0xFF); // attach the byte

			if (numBytesCoord == 2) {
				nextUp = StreamingField.Y;
				numBytesCoord = 0;
			}
			
		} else if (nextUp == StreamingField.Y) { // 2 bytes long, Y Coordinate
			numBytesCoord++;
			y = y << 8;
			y = y | (bCurrent & 0xFF);

			if (numBytesCoord == 2) {
				nextUp = StreamingField.FORCE;
				numBytesCoord = 0;
			}
		} else if (nextUp == StreamingField.FORCE) {
			force = (((int) bCurrent) & 0xFF);
			
			this.pl.sample(x,y,force);
			

			// reset our values
			x = 0;
			y = 0;
			force = 0;
			penUp = false;

			// look for the header of the next sample
			nextUp = StreamingField.HEADER;
		}
	}

	/**
	 * Whenever data is available, send bytes one in a row to the processor.
	 * 
	 * @see javax.comm.SerialPortEventListener#serialEvent(javax.comm.SerialPortEvent)
	 */
	public void serialEvent(SerialPortEvent event) {
		assert !this.pollMode;
		
		switch (event.getEventType()) {

		case SerialPortEvent.BI:
			// fall through
		case SerialPortEvent.OE:
			// fall through
		case SerialPortEvent.FE:
			// fall through
		case SerialPortEvent.PE:
			// fall through
		case SerialPortEvent.CD:
			// fall through
		case SerialPortEvent.CTS:
			// fall through
		case SerialPortEvent.DSR:
			// fall through
		case SerialPortEvent.RI:
			// fall through
		case SerialPortEvent.OUTPUT_BUFFER_EMPTY:
			break;

		case SerialPortEvent.DATA_AVAILABLE: // there is data to process!
			try {
				this.processDataFromInputStream();
			} catch(IOException e) {
				throw new RuntimeException(e);
			}
			break;
		}
	}
	
	public void poll() {
		if(!this.pollMode) {
			throw new IllegalStateException("Not in polling mode.");
		}
		try {
			this.processDataFromInputStream();
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
	}
		
	
	private void processDataFromInputStream() throws IOException {
		while (inputStream.available() > 0) {
			int numBytes = inputStream.read(readBuffer);

			// process the byte of data
			for (int i = 0; i < numBytes; i++) {
				if(this.nextUp==StreamingField.SKIP16 && this.numBytesCoord==0) {
					if(this.skipIntermediatePackets && !penUp && (numBytes-i)>=(16+2+2+1)+(3+16+2+2+1)) {
						/*
						 * We have the rest of this packet in the pipeline, followed by
						 * at least one more packet so let's skip the rest of this 
						 * packet. (We can't skip the next packet yet because we need to work out
						 * whether it's a pen-up or not.) Also if the pen is currently up then we want
						 * to report this packet because it's the first one of the pen being down.
						 * The rest of this packet is (16+2+2+1) bytes and we are processing the first,
						 * so skip (16+2+2+1)-1 bytes
						 */
						i+=(16+2+2+1)-1;
						this.bCurrent=0;
						this.bLast=0;
						this.bLastLast=0;
						this.numBytesCoord=0;
						this.nextUp = StreamingField.HEADER;
					} else if ((numBytes-i)>=16) {
						/* we want to skip 16 and we're processed the first, 
						 * so skip 15 more.
						 */
						i+=15;
						this.bCurrent=0;
						this.bLast=0;
						this.bLastLast=0;
						this.numBytesCoord=0;
						this.nextUp = StreamingField.X;
					} else {
						handleByte(readBuffer[i]);
					}
				} else {
					handleByte(readBuffer[i]);
				}
			}
		}
	}
}
