/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.server;

import java.io.EOFException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.net.Socket;
import java.net.SocketException;
import java.util.logging.Level;
import java.util.logging.Logger;

import t3.remotehrd.protocol.HelloMessageToServer;
import t3.remotehrd.protocol.Ready;

/*
 * All methods are thread safe
 * If not explicitly using synchronized then it's because NetworkSender is.
 */

/**
 * Represents, at the server end, a client connected to the server.
 * @author pjt40
 *
 */
public class Client {
	private ObjectInputStream ois;
	private NetworkSender networkSender;
	private Socket socket;
	private RemoteHRDServer server;
	private Object closingSynchronizer = new Object();
	private boolean closed;
	public final int clientId;

	private static final Logger logger = Logger.getLogger("t3.remotehrd.server");
	
	Client(Socket s, RemoteHRDServer server, boolean alwaysServerPush) throws IOException{
		this.closed = false;
		this.server = server;
		this.socket = s;

        // care!
        // protocol is: 
        //      client creates outputstream
        //      client sends hello
        //      server creates outputstream
        
		this.ois = new ObjectInputStream( Client.this.socket.getInputStream());
		HelloMessageToServer helloMessage;
		try {
			helloMessage = (HelloMessageToServer) ois.readObject();
		} catch(ClassNotFoundException e) {
			throw new IOException(e.getMessage());
		}
		this.clientId = helloMessage.clientId;
		this.networkSender = new NetworkSender(Client.this.socket.getOutputStream(), this, alwaysServerPush);
		this.receiverAndProcessorThread.setPriority(Thread.MAX_PRIORITY);
	}
	
	private Thread receiverAndProcessorThread = new Thread("Client's receiver and processor thread") {
		public void run() {
			logger.info("Connected to client "+Client.this.socket+" as "+Client.this);
			try {
				logger.fine("Sending entire state to client.");
				
				Client.this.server.sendEntireStateToClientAndThenLinkIn(Client.this);
				
				logger.fine("Sent state.");
				while(true) {
					Object msg = ois.readObject();
					logger.fine("Received message from client: "+msg);
					if(msg instanceof Ready) {
						networkSender.receivedClientReadyMessage();
					} else {
						Client.this.server.receivedMessageFromClientOrInputSource(msg);
					}
				}
			} catch(EOFException e) {
				// client left
				Client.this.close();
			} catch(Throwable e) {
				//includes exceptions and assertionerrors
				Client.this.errorWithThisClient(e);
			}
		}
	};
	
	
	/**
	 * Reports a specified exception to do with this client, and closes
	 * the client without stopping the server. This method is thread-safe; 
	 * it can be called from any thread.
	 * @param e
	 */
	public void errorWithThisClient(Throwable e) {
		// indicates an error with this client, not with whole server
		// should be callable from any thread
		synchronized(closingSynchronizer) {
			if(!this.closed) {
				if(
					(e instanceof SocketException)
					&& e.getMessage().equals("Connection reset")
				) {
					if(logger!=null) {
						logger.log(Level.INFO,"Client terminated connection "+this);
					}
				} else {
					if(logger!=null) {
						logger.log(Level.SEVERE,"Caught exception and about to close client ",e);
					}
					e.printStackTrace();
				}
				this.close();
			} else {
				// we're already closed
				// chances are the error was because we were closing
				// so just ignore it
			}
		}
	}
	
	/**
	 * Ensures that no more messages are sent to this client and closes the
	 * client's socket. This method can be called more than once with no ill
	 * effects. This method is thread-safe; it can be called from any thread.
	 */
	public void close() {
		// should be callable from any thread
		// might be called more than once
		// eg an error in the outputstream so another thread calls errorWithThisClient
		// which then calls close. The receiver/processor thread for this client then
		// throws an exception next time it tries to read from its input stream
		try {
			synchronized(closingSynchronizer) {
				if(!this.closed) {
					this.server.clientHasStopped(this);
					this.closed = true;
					if(!this.socket.isClosed()) {
						this.socket.close();
					}
					this.server.callBacks.remoteServerCallBack_clientLeft(this.clientId);
					logger.info("Closed connection to client "+this);
				} else {
					// already closed
				}
			}
		} catch(Throwable e) {
			this.server.stateManager.fatalError(e);
		}
	}
	
	void addToSendQueue(Serializable msg) {
		logger.log(Level.FINE,"Adding message to client's send queue: {0}",msg);
		this.networkSender.addToSendQueue(msg);
	}
	
	void endOfUpdateFrame() {
		this.networkSender.endOfUpdateFrame();
	}
	
	void startReceiverAndProcessorThread() {
		this.receiverAndProcessorThread.start();
	}
	
}
