/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.client;


import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.media.opengl.GL;

import t3.hrd.input.KeyboardInput;
import t3.hrd.input.PointInputDeviceState;
import t3.hrd.input.ShapeInputDeviceState;
import t3.hrd.renderer.HRDRendererCallBacks;
import t3.hrd.renderer.Projector;
import t3.hrd.state.ScaRotTraTransformImmutable;
import t3.hrd.state.StateManager;
import t3.hrd.util.FPSTimer;
import t3.remotehrd.protocol.BurstTerminator;
import t3.remotehrd.protocol.OpCreateCursor;
import t3.remotehrd.protocol.OpCreateLink;
import t3.remotehrd.protocol.OpCreateTile;
import t3.remotehrd.protocol.OpCreateUnwarpedRect;
import t3.remotehrd.protocol.OpCursorPos;
import t3.remotehrd.protocol.OpCursorShape;
import t3.remotehrd.protocol.OpDestroyCursor;
import t3.remotehrd.protocol.OpDestroyLink;
import t3.remotehrd.protocol.OpDestroyTile;
import t3.remotehrd.protocol.OpDestroyUnwarpedRect;
import t3.remotehrd.protocol.OpDestroyUnwarpedRects;
import t3.remotehrd.protocol.OpLogMessage;
import t3.remotehrd.protocol.OpReorderTilesAndLinks;
import t3.remotehrd.protocol.OpSetCursorOptions;
import t3.remotehrd.protocol.OpSetLinkAff;
import t3.remotehrd.protocol.OpSetTileAff;
import t3.remotehrd.protocol.OpSetTileVisibility;
import t3.remotehrd.protocol.OpUpdateTileContents;
import t3.remotehrd.protocol.OpUpdateTileContentsByCopyingFromOtherTile;
import t3.remotehrd.protocol.Ready;

class HRDRendererCallbacksForRemoteHRDClient implements HRDRendererCallBacks{

	private static final Logger logger = Logger.getLogger("t3.remotehrd.client");
	
	public final RemoteHRDClient r;
	private static final int SIMULATE_DELAY_MS_OR_ZERO=0;
    private final List<Object> receivedMessagesInThisBurst = new LinkedList<Object>();
	private final FPSTimer fpsTimer = new FPSTimer("NWK UPDATE RATE per sec = ","",100000.0,0.3,2000,8000);
    private final FPSTimer fpsShapeTimer = new FPSTimer("Hand shape output rate = ","",100000.0,0.3,2000,8000);
    	
    
	HRDRendererCallbacksForRemoteHRDClient(RemoteHRDClient r) {
		this.r = r;
	}
	public void callback_keyboardInput(KeyboardInput k, Object client) { 
		if(r.callBacks.remoteClientCallBack_keyboardInput(k)) {
			r.sendMessageToServer(k);
		}
	}
	public void callback_pointInputDeviceStateChanged(PointInputDeviceState pids, Object client) { 
		if(r.callBacks.remoteClientCallBack_pointInputDeviceStateChanged(pids)) {
			r.sendMessageToServer(pids);
		}		
	}	     
    public void callback_shapeInputDeviceStateChanged(ShapeInputDeviceState sids, Object client) {
        r.sendMessageToServer(sids);
    }
	public void callback_closingBeforeLastOGL() {}
	public void callback_closingAfterWindowsClosed() { r.callBacks.remoteClientCallBack_closingAfterWindowsClosed(); }
	public void callback_mouseClicked(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mouseClicked(e, p);  }
    public void callback_mouseEntered(MouseEvent e, Projector p){ r.callBacks.remoteClientCallBack_mouseEntered(e, p);  }
    public void callback_mouseExited(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mouseExited(e, p);  }
    public void callback_mousePressed(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mousePressed(e, p);  }
    public void callback_mouseReleased(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mouseReleased(e, p);  }
    public void callback_mouseMoved(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mouseMoved(e, p);}
    public void callback_mouseDragged(MouseEvent e, Projector p) { r.callBacks.remoteClientCallBack_mouseDragged(e, p); }
	public boolean callback_needToRepaintOverlay(Projector p) { return r.callBacks.remoteClientCallBack_needToRepaintOverlay(p); }
	public void callback_oglPaintOverlay(GL gl, Projector p) { r.callBacks.remoteClientCallBack_oglPaintOverlay(gl,  p); }
	
	public void callback_oncePerFrame() {
		try {
			synchronized (r.hrdRenderer.stateManager) {
                if( this.readInDataFromNetwork() ) {
                    this.processBurst(r.hrdRenderer.stateManager);
                }
			}
			r.callBacks.remoteClientCallBack_oncePerFrame();
		} catch(Throwable e) {
			// we're not allowed to declare that we throw exceptions so we'll throw them as runtime excpetions
			throw new RuntimeException(e);
		}
	}
	
	private boolean readInDataFromNetwork() throws IOException, ClassNotFoundException {
        // reads in data from network
        // stops when no more data available, or when end of burst is reached, which ever
        // comes first.
        // returns true if end of burst is reached
        long t1 = System.currentTimeMillis();
        boolean burstTermReceived = false;
        boolean anyMessagesReceivedInThisCall = false;
        while(r.bis.available()>0 && !burstTermReceived) {
            Object lastMessageReceived = r.obis.readObject();
            anyMessagesReceivedInThisCall = true;
            burstTermReceived = burstTermReceived || (lastMessageReceived instanceof BurstTerminator);
            logger.log(Level.FINE,"Received object from server: {0}",lastMessageReceived);
            this.receivedMessagesInThisBurst.add(lastMessageReceived);
        }
        if(anyMessagesReceivedInThisCall) {
            //System.out.println("Received BT in this read? "+burstTermReceived+" and time taken " +(System.currentTimeMillis()-t1));
        } else {
            // nothing was received this time so no point outputting stuff
        }
        return burstTermReceived;
    }
    
    
	private void processBurst(StateManager sm) 
	{        
        if(this.receivedMessagesInThisBurst.size()>1) {
            // send a ready message to the server.
            // ie client is sending a request for more messages
            // advantage of this is that in the mean time the server has been storing them up
            // and can coalesce messages
            //System.out.println("Server push now false");
            
            if(!(
                    this.receivedMessagesInThisBurst.size()==2 
                    && this.receivedMessagesInThisBurst.get(0) instanceof OpCursorPos
            )) {
                fpsTimer.oneFrame();
            } else {
                // not really a netwrok update as it was just cursors
            }
            
            if(SIMULATE_DELAY_MS_OR_ZERO==0) {
                this.r.sendMessageToServer(new Ready());
            } else {
                //simulate long rtt
                Thread t = new Thread() {
                    public void run() {
                        try {
                            Thread.currentThread().sleep(SIMULATE_DELAY_MS_OR_ZERO);
                            HRDRendererCallbacksForRemoteHRDClient.this.r.sendMessageToServer(new Ready());                             
                        } catch(Exception e) {}
                    }
                };
                t.start();
            }
            logger.fine("Sent ready message to server");
        } else {
            // do nothing:
            // we just got a burst terminator, nothing else
            // so the server has no more messages to send
            // we won't send a ready message because we're now in server push mode
            // server will send a message when it has one
            // if it doesn't have one to send to us then we'll block next time
            // we call ois.readObject(), until it does have one to send to us
            //System.out.println("Server push now true");
        }
        
       for(Object o: this.receivedMessagesInThisBurst) {
			logger.log(Level.FINE,"Processing object from server: {0}",o);
			if(o instanceof OpUpdateTileContents) {
				OpUpdateTileContents oo = (OpUpdateTileContents) o;
				BufferedImage img = oo.reconstructBufferedImage();
				Rectangle r= oo.r;
				int tid = oo.elId;
                int chint = oo.compressionHint;
				oo = null; // explicit deallocation
				sm.getTileByIdForOp(tid).opUpdateContents(r.x,r.y,img, chint);
			} else if(o instanceof OpUpdateTileContentsByCopyingFromOtherTile) {
				OpUpdateTileContentsByCopyingFromOtherTile oo = (OpUpdateTileContentsByCopyingFromOtherTile) o;
				sm.getTileByIdForOp(oo.elId).opUpdateContentsByCopyingFromOtherTile(
					oo.sx, oo.sy, oo.dx, oo.dy, oo.w, oo.h, sm.getTileByIdForOp(oo.sTileId));
			} else if(o instanceof OpSetTileAff) {
				OpSetTileAff oo = (OpSetTileAff) o; 
				sm.getTileByIdForOp(oo.elId).opSetAff(oo.DESKcentreX,oo.DESKcentreY,oo.DESKwidth, oo.DESKheight, oo.thetaClockwise);
			} else if(o instanceof OpCreateTile) {
				OpCreateTile oo = (OpCreateTile) o; 
				sm.opCreateTile(oo.elId,oo.w,oo.h,oo.flags);
			} else if(o instanceof OpDestroyTile) {
				OpDestroyTile oo = (OpDestroyTile) o; 
				sm.opDestroyTile(sm.getTileByIdForOp(oo.elId));
			} else if(o instanceof OpSetTileVisibility) {
				OpSetTileVisibility oo = (OpSetTileVisibility) o; 
				sm.getTileByIdForOp(oo.elId).opSetVisibility(oo.v);
			} else if(o instanceof OpCreateUnwarpedRect) {
				OpCreateUnwarpedRect oo = (OpCreateUnwarpedRect) o; 
				sm.getTileByIdForOp(oo.elId).opCreateUnwarpedRect(oo.r);
			} else if(o instanceof OpDestroyUnwarpedRect) {
				//OpDestroyUnwarpedRect oo = (OpDestroyUnwarpedRect) o; 
				//sm.getTileByIdForOp(oo.elId).opDestroyUnwarpedRect(oo.r);
				// deprecated
				assert false;
			} else if(o instanceof OpDestroyUnwarpedRects) {
				OpDestroyUnwarpedRects oo = (OpDestroyUnwarpedRects) o; 
				sm.getTileByIdForOp(oo.elId).opDestroyAllUnwarpedRectsIntersecting(oo.r);
			} else if(o instanceof OpCreateLink) {
				OpCreateLink oo = (OpCreateLink) o; 
				sm.opCreateLink(oo.elId,oo.displayType,oo.color,
						new ScaRotTraTransformImmutable(oo.ma),
						new ScaRotTraTransformImmutable(oo.mb)
					);
			} else if(o instanceof OpSetLinkAff) {
				OpSetLinkAff oo = (OpSetLinkAff) o; 
				sm.getLinkByIdForOp(oo.elId).opSetAff(
						new ScaRotTraTransformImmutable(oo.ma),
						new ScaRotTraTransformImmutable(oo.mb)
					);
			} else if(o instanceof OpDestroyLink) {
				OpDestroyLink oo = (OpDestroyLink) o; 
				sm.opDestroyLink(sm.getLinkByIdForOp(oo.elId));
			} else if(o instanceof OpReorderTilesAndLinks) {
				OpReorderTilesAndLinks oo = (OpReorderTilesAndLinks) o; 
				sm.opReorderTilesAndLinksById(oo.elOrder);
			} else if (o instanceof BurstTerminator) {
				// we'll deal with it later
			} else if (o instanceof OpCreateCursor) {
				OpCreateCursor oo = (OpCreateCursor) o;
				sm.opCreateCursor(oo.cursorId);
				//cm.opSetCursorOptions(oo.cursorId, Color.YELLOW, pjt.hrd.commonlogic.Cursor.DISPLAYTYPE_TRAIL);
			} else if (o instanceof OpDestroyCursor) {
				OpDestroyCursor oo = (OpDestroyCursor) o;
				sm.opDestroyCursor(sm.getCursorByIdForOp(oo.cursorId));
			} else if (o instanceof OpCursorPos) {
				OpCursorPos oo = (OpCursorPos) o;
                sm.getCursorByIdForOp(oo.cursorId).opCursorPos(oo.DESKx, oo.DESKy, oo.visible);
			} else if (o instanceof OpCursorShape) {
				OpCursorShape oo = (OpCursorShape) o;
                fpsShapeTimer.oneFrame();
				sm.getCursorByIdForOp(oo.cursorId).opCursorShape(oo.polygonsForShapeCursor);
			} else if (o instanceof OpSetCursorOptions) {
				OpSetCursorOptions oo = (OpSetCursorOptions) o;
				sm.getCursorByIdForOp(oo.cursorId).opSetCursorOptions(oo.col, oo.displayType);
            } else if (o instanceof OpLogMessage) {
                OpLogMessage oo = (OpLogMessage) o;
                sm.opLogMessage(oo.message);
			} else {
				r.callBacks.remoteClientCallBack_receivedMessageFromServer(o);
			}
		}

		this.receivedMessagesInThisBurst.clear();
        
		// we don't need to worry about reseting the objectinput stream
        // it's done for us by the other end.   
            
		return;
	}		
	
}
