/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios.commonbehaviours;


import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.ScaRotTraTransformImmutableUniformScale;
import t3.portfolios.PointInputDevice;
import t3.portfolios.PointInputDeviceTypeAndButton;
import t3.portfolios.PointInputDeviceTypeAndButtonSet;
import t3.portfolios.Portfolio;
import t3.portfolios.PortfolioCommonBehaviour;
import t3.portfolios.PortfolioEvent;
import t3.portfolios.swing.SwingFramePortfolio;
import Jama.Matrix;

/**
 * This class is like RotateNTranslateWithTranslateOnlyRegion but additionally
 * any portfolios in the set containers act as containers. Portfolios dragged into
 * containers will become children of the containers until dragged out. 
 * <p>
 * Portfolios dragged into containers are shrunk (scaled down from 1.0 to scaleInContainerDESKtoPORT).
 * This happens while the pen drags a distance scaleChangePerMM. Portfolios dragged into 
 * recycleBinPortfolioOrNull (which must also be in the containers set) are destroyed.
 * <p>
 * Rule re containing and shrinking:
 * If point of contact goes in then whole lot shrinks.
 * If whole thing comes out then whole thing unshrinks.
 * 
 * @author pjt40
 *
 */
public class RNTWTORAndRegroupAndShrinkAndDestroy  implements PortfolioCommonBehaviour {
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios.commonbehaviours");
	
	public final double PORTw, PORTh, translateOnlyProportion;
	public final boolean guessPORTdimensionsFromOurTile;
    public final Set<Portfolio> containers;
    public final Portfolio recycleBinPortfolio;
    public final double scaleInContainerDESKtoPORT;
    public final double scaleChangePerMM;
    public final Set<Portfolio> portfoliosBeingDragged = new HashSet<Portfolio>();
    
	public final PointInputDeviceTypeAndButtonSet pidtbs;

	public RNTWTORAndRegroupAndShrinkAndDestroy(
			PointInputDeviceTypeAndButtonSet pidtbs, 
			boolean guessPORTdimensionsFromOurTile, 
			double PORTw, 
			double PORTh,
			double translateOnlyProportion,
            double scaleInContainerPORTtoDESK,
            double scaleChangePerMM,
            Portfolio recycleBinPortfolioOrNull
		) {
		this.pidtbs = pidtbs;
		this.guessPORTdimensionsFromOurTile = guessPORTdimensionsFromOurTile;
		this.PORTw = PORTw;
		this.PORTh = PORTh;
		this.translateOnlyProportion = translateOnlyProportion;
        this.containers = new HashSet<Portfolio>();
        this.scaleChangePerMM=scaleChangePerMM;
        this.scaleInContainerDESKtoPORT = 1.0/scaleInContainerPORTtoDESK;
        this.recycleBinPortfolio = recycleBinPortfolioOrNull;
	}

    
	public boolean customProcessEventForThisPortfolioNotChildren(Portfolio p, PortfolioEvent e, boolean bubbled) {
		if (
			e.pointInputDevice != null
			&& e.positionAndButtonsKnown
			&& this.pidtbs.contains(
				new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
			)
		) {
			if(
                    e.eventType==e.EVENT_PID_PRESS 
                    && !this.portfoliosBeingDragged.contains(p)
			) {
                p.getParent().bringChildToFront(p);
                p.portfolioServer.enterFDOPmodeForPointInputDevice(p, e.pointInputDevice, e.pidButton, e.getPORTx(p), e.getPORTy(p));
                this.portfoliosBeingDragged.add(p);
                return true;
			} else {
				// we still don't want any events for this button to get through to the underlying
				// portfolio.
				return true;
			}
		} else {
			return false;
		}
	}

	public boolean customProcessFDOPevent(Portfolio p, PortfolioEvent e, double PORTxWhenEnteredFDOPmode, double PORTyWhenEnteredFDOPmode) {
        if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
		)) {
			
			double PORTcentreX, PORTcentreY;
			if(p.childrenTopToBottomReadOnly.size()==0) {
				PORTcentreX=0.0;
				PORTcentreY=0.0;
			} else {
				double DESKcentreX = p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles().getCenterX();
				double DESKcentreY = p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles().getCenterY();
				double[] PORTcentre = JOGLHelper.getDFromM( 
                        p.getmDESKtoPORTReadOnly().times(
							JOGLHelper.getMFromD(DESKcentreX, DESKcentreY, 1.0)
						)
					);
				PORTcentreX = PORTcentre[0]/PORTcentre[2];
				PORTcentreY = PORTcentre[1]/PORTcentre[2];
			}	
			
			double PORTwidth, PORTheight;
			if(this.guessPORTdimensionsFromOurTile) {
				if(!p.hasTile()) {
					throw new IllegalStateException("Cannot guess width and height of portfolio if it has no tile.");
				}
				PORTwidth = p.getTileWidthInPORT();
				PORTheight = p.getTileHeightInPORT();
			} else {
				PORTwidth = this.PORTw;
				PORTheight = this.PORTh;
			}

            // check if we should/shouldn't be in a container
            boolean curInContainer = this.containers.contains(p.getParent());
            if(!curInContainer) {
                // check if we should be in a container but we're not
                for(Portfolio c: this.containers) {
                    if(e.isPointOnTile(c)) {
                        // we should be in c!
                        p.unhookFromParentAndMakeChildOf(c);
                        curInContainer = true;
                        break;
                    }
                }
            } else {
                // check if we shouldn't be in a container but we are
                if(
                        !p.getParent().getGpDESKoutlineOfOurTile().contains(p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles())
                    && !p.getParent().getGpDESKoutlineOfOurTile().intersects(p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles())
                 ) {
                    p.unhookFromParentAndMakeChildOf(p.portfolioServer.rootPortfolio);
                    curInContainer = false;
                }
            }
            
            ScaRotTraTransformImmutableUniformScale curDESKtoPORT = 
                new ScaRotTraTransformImmutableUniformScale(p.getmDESKtoPORTReadOnly());
            double newScaleDESKtoPORT = curDESKtoPORT.getScale();
            if(curInContainer) {
                if(newScaleDESKtoPORT<this.scaleInContainerDESKtoPORT) {
                    // increase it
                    double dist = Math.abs(e.DESKxOld-e.DESKx)+Math.abs(e.DESKyOld-e.DESKy);
                    newScaleDESKtoPORT+=this.scaleChangePerMM*dist;
                }
                if(newScaleDESKtoPORT>this.scaleInContainerDESKtoPORT) {
                    newScaleDESKtoPORT = this.scaleInContainerDESKtoPORT;
                }
            } else {
                if(newScaleDESKtoPORT>1.0) {
                    // decrease it
                    double dist = Math.abs(e.DESKxOld-e.DESKx)+Math.abs(e.DESKyOld-e.DESKy);
                    newScaleDESKtoPORT-=this.scaleChangePerMM*dist;
                }
                if(newScaleDESKtoPORT<1.0) {
                    newScaleDESKtoPORT = 1.0;
                }                
            }
            
			
			ScaRotTraTransformImmutableUniformScale idealNewDESKtoPORT;
			
			if(
				Math.abs(PORTxWhenEnteredFDOPmode-PORTcentreX)/PORTwidth>this.translateOnlyProportion
				|| Math.abs(PORTyWhenEnteredFDOPmode-PORTcentreY)/PORTheight>this.translateOnlyProportion
			) {
				// now lets work out our rotn
				// point W is PORTwhenenteredfdopmode
				// point N is PORTcursornow
				// point O is origin
				// we want to find alpha, the angle between WO and NO
				// WO.NO = |WO| |NO| cos alpha
				double WOdx = PORTcentreX-PORTxWhenEnteredFDOPmode;
				double WOdy = PORTcentreY-PORTyWhenEnteredFDOPmode;
				double NOdx = PORTcentreX-e.getPORTx(p);
				double NOdy = PORTcentreY-e.getPORTy(p);
				double WOdotNO = WOdx*NOdx+WOdy*NOdy;
				double modWO = Math.sqrt(WOdx*WOdx+WOdy*WOdy);
				double modNO = Math.sqrt(NOdx*NOdx+NOdy*NOdy);
				double cosAlpha = WOdotNO/(modWO*modNO);
				if(cosAlpha>1.0) {
					// correct for numerical errors
					cosAlpha = 1.0;
				} else {
					// no need to correct
				}
				double alpha = Math.acos(cosAlpha);
				// 0<=alpha<pi
				
				// now we need to work out whether alpha is clockwise or not
				// clockwise iff WOdy*NOdx-WOdx*NOdy>0
				// beta is the clockwise angle from WO to NO		
				double beta = (WOdy*NOdx-WOdx*NOdy>0.0) 
					? alpha
					: 2.0*Math.PI-alpha;
					
                // now rotate by beta clockwise 
                idealNewDESKtoPORT = 
                    new ScaRotTraTransformImmutableUniformScale(
                        newScaleDESKtoPORT,
                        curDESKtoPORT.getThetaClockwise()-beta,
                        curDESKtoPORT.getTx(),
                        curDESKtoPORT.getTy()
                    );
			} else {
                idealNewDESKtoPORT =  new ScaRotTraTransformImmutableUniformScale(
                                newScaleDESKtoPORT,
                                curDESKtoPORT.getThetaClockwise(),
                                curDESKtoPORT.getTx(),
                                curDESKtoPORT.getTy()
                            );
			}
			
			
			// now where do we end up if we use this on our pen coords
			Matrix PORTpoint = idealNewDESKtoPORT.getMatrix2dHomogReadOnly().times( 
					JOGLHelper.getMFromD(e.DESKx, e.DESKy,1.0)
				);
			
			// now adjust based on this
			// we can do the translation comparison in portfolio space because 
			// the translation is applied last, after the rotn and scale.
			double tx = ( PORTpoint.get(0,0)/PORTpoint.get(2,0)-PORTxWhenEnteredFDOPmode );
			double ty = ( PORTpoint.get(1,0)/PORTpoint.get(2,0)-PORTyWhenEnteredFDOPmode );
			
			ScaRotTraTransformImmutableUniformScale s = new ScaRotTraTransformImmutableUniformScale(
				idealNewDESKtoPORT.getScale(),
				idealNewDESKtoPORT.getThetaClockwise(),
				idealNewDESKtoPORT.getTx() - tx,
				idealNewDESKtoPORT.getTy() - ty
				);

			
			p.setDESKtoPORT(s.getMatrix2dHomogReadOnly());
            
			return true;
		} else {
			return false;
		}
	}

	public boolean customProcessEndOfFDOPmode(final Portfolio p, PointInputDevice pid, int button) {
		if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(pid.pointInputDeviceType, button)
		)) {
            this.portfoliosBeingDragged.remove(p);
            
            if(p.getParent() == this.recycleBinPortfolio) {
                p.setVisibleWhenParentVisible(false);
                SwingFramePortfolio.runLaterFromSwingThreadUsesPortfolios(p.portfolioServer, new Runnable(){ public void run() {
                    p.destroyThisAndAllDescendants();
                }});
            } else {
                // do nothing
            }

			return true;
		} else {
			return false;
		}
	}
		
	
}


