/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;


import java.awt.Color;
import java.awt.geom.Rectangle2D;
import java.util.logging.Logger;

import t3.hrd.state.Link;
import t3.hrd.state.ScaRotTraTransformImmutable;

/**
 * 
 * Represents a link between two rectangles each in a specified portfolio.
 *  
 * @author pjt40
 *
 */
public class PortfolioLink {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios");

	public final Portfolio sourcePortfolio;
	public final Portfolio destPortfolio;
	public final Rectangle2D.Double sourcePORTrect, destPORTrect;
	private final Link link;
	
	// uniform
	private final ScaRotTraTransformImmutable m2hStandardRectToSourcePORTrect;
	private final ScaRotTraTransformImmutable m2hStandardRectToDestPORTrect;
	private boolean destroyed;
	
	/**
	 * Creates a link between two portfolios.
	 * 
	 * @param displayType
	 * @param c
	 * @param sourcePortfolio
	 * @param destPortfolio
	 */
	public PortfolioLink(int displayType, Color c, Portfolio sourcePortfolio, Portfolio destPortfolio) {
		this(
			displayType,
			c, 
			sourcePortfolio, 
			destPortfolio, 
			new Rectangle2D.Double(
				-sourcePortfolio.getTileWidthInPORT()/2.0,
				-sourcePortfolio.getTileHeightInPORT()/2.0,
				sourcePortfolio.getTileWidthInPORT(), 
				sourcePortfolio.getTileHeightInPORT()
			),
			new Rectangle2D.Double(
				-destPortfolio.getTileWidthInPORT()/2.0,
				-destPortfolio.getTileHeightInPORT()/2.0,
				destPortfolio.getTileWidthInPORT(), 
				destPortfolio.getTileHeightInPORT()
			)
		);
			
		
	}
	
	/**
	 * Creates a link between two rectangles, each in the PORT space of a portfolios.
	 * 
	 * @param displayType
	 * @param c
	 * @param sourcePortfolio
	 * @param destPortfolio
	 * @param sourcePORTrect
	 * @param destPORTrect
	 */
	public PortfolioLink(int displayType, Color c, Portfolio sourcePortfolio, Portfolio destPortfolio, Rectangle2D.Double sourcePORTrect, Rectangle2D.Double destPORTrect) {
		this.sourcePortfolio = sourcePortfolio;
		this.sourcePORTrect = sourcePORTrect;
		this.m2hStandardRectToSourcePORTrect =
			new ScaRotTraTransformImmutable(
					sourcePORTrect.width, 
					sourcePORTrect.height,
					0.0,
					sourcePORTrect.x,
					sourcePORTrect.y
				);
		this.destPortfolio = destPortfolio;
		this.destPORTrect = destPORTrect;
		this.m2hStandardRectToDestPORTrect = 
			new ScaRotTraTransformImmutable(
				destPORTrect.width, 
				destPORTrect.height,
				0.0,
				destPORTrect.x,
				destPORTrect.y
			);
		this.sourcePortfolio.notifyCreatedLinkFromThis(this);
		this.destPortfolio.notifyCreatedLinkToThis(this);
		this.link = sourcePortfolio.portfolioServer.stateManager.opCreateLinkAutogenerateId(displayType, c, new ScaRotTraTransformImmutable(1,1,0,0,0), new ScaRotTraTransformImmutable(1,1,0,0,0));
		this.notifyOfSourceOrDestPortfolioMove();
		this.notifyOfSourceOrDestPortfolioVisChange();
		
	}
	
	void notifyOfSourceOrDestPortfolioMove() {
		this.link.opSetAff(
			new ScaRotTraTransformImmutable(
				sourcePortfolio.gettPORTtoDESK().getMatrix2dHomogReadOnly().times(
					this.m2hStandardRectToSourcePORTrect.getMatrix2dHomogReadOnly()
				)
			),
			new ScaRotTraTransformImmutable(
				destPortfolio.gettPORTtoDESK().getMatrix2dHomogReadOnly().times(
					this.m2hStandardRectToDestPORTrect.getMatrix2dHomogReadOnly()
				)
			)
		);
	}
	
	void notifyOfSourceOrDestPortfolioVisChange() {
		// nothing yet		
	}
	
	Link getLink() {
		return this.link;
	}
	
	/**
	 * Destroys this link.
	 */
	public final void destroy() {
		this.checkNotDestroyed();
		this.sourcePortfolio.notifyDestroyedLinkFromThis(this);
		this.destPortfolio.notifyDestroyedLinkToThis(this);
		this.link.stateManager.opDestroyLink(this.link);
		this.destroyed = true;
	}	
	
	private void checkNotDestroyed() {
		if(this.destroyed) {
			throw new IllegalStateException("PortfolioLink is destroyed.");
		}
	}
	
	/**
	 * Returns true iff the link has been destroyed.
	 * @return
	 */
	public final boolean isDestroyed() {
		return this.destroyed;
	}
	
	
	
}
