/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.color.ColorSpace;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferInt;
import java.awt.image.Raster;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Logger;

import javax.imageio.ImageIO;

import t3.hrd.state.JOGLHelper;
import Jama.Matrix;

class IdentifyUnwarpedRects {
	
	
	
	/************ Testing **********************/

	public static void main(String[] args) { 
		Matrix mScalept96 = JOGLHelper.get2hmScale(0.96, 0.96);
		Matrix mScalept975 = JOGLHelper.get2hmScale(0.975, 0.975);		
		Matrix mScale1pt01 = JOGLHelper.get2hmScale(1.01, 1.01);	
		Matrix mScale1pt005 = JOGLHelper.get2hmScale(1.005, 1.005);
		Matrix mScale1pt05 = JOGLHelper.get2hmScale(1.05, 1.05);
		Matrix mScale1pt1 = JOGLHelper.get2hmScale(1.1, 1.1);
		Matrix mRotate0pt521Deg = JOGLHelper.get2hmRotation(0.521*(2.0*Math.PI/360.0));
		Matrix mRotate1pt041Deg = JOGLHelper.get2hmRotation(1.041*(2.0*Math.PI/360.0));	
		Matrix mScale1pt013Rotate0pt75Deg = JOGLHelper.get2hmScale(1.013, 1.013).times(JOGLHelper.get2hmRotation(0.75*(2.0*Math.PI/360.0)));	
		Matrix mScale1pt100Rotate1pt00Deg = JOGLHelper.get2hmScale(1.100, 1.100).times(JOGLHelper.get2hmRotation(1.00*(2.0*Math.PI/360.0)));	
		
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s0pt960", mScalept96);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s0pt975", mScalept975);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt010", mScale1pt01);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt005", mScale1pt005);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt050", mScale1pt05);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt100", mScale1pt1);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-r0pt521Deg", mRotate0pt521Deg);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-r1pt041Deg", mRotate1pt041Deg);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt013-r0pt750Deg", mScale1pt013Rotate0pt75Deg);
		testIt2("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4-s1pt100-r1pt000Deg", mScale1pt100Rotate1pt00Deg);
		
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s0pt960", mScalept96);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s0pt975", mScalept975);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt010", mScale1pt01);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt005", mScale1pt005);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt050", mScale1pt05);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt100", mScale1pt1);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-r0pt521Deg", mRotate0pt521Deg);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-r1pt041Deg", mRotate1pt041Deg);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt013-r0pt750Deg", mScale1pt013Rotate0pt75Deg);
		testIt2("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5-s1pt100-r1pt000Deg", mScale1pt100Rotate1pt00Deg);
		
		
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s0pt960", mScalept96);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s0pt975", mScalept975);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt010", mScale1pt01);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt005", mScale1pt005);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt050", mScale1pt05);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt100", mScale1pt1);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-r0pt521Deg", mRotate0pt521Deg);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-r1pt041Deg", mRotate1pt041Deg);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt013-r0pt750Deg", mScale1pt013Rotate0pt75Deg);
		testIt2("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1-s1pt100-r1pt000Deg", mScale1pt100Rotate1pt00Deg);
		
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s0pt960", mScalept96);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s0pt975", mScalept975);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt010", mScale1pt01);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt005", mScale1pt005);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt050", mScale1pt05);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt100", mScale1pt1);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-r0pt521Deg", mRotate0pt521Deg);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-r1pt041Deg", mRotate1pt041Deg);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt013-r0pt750Deg", mScale1pt013Rotate0pt75Deg);
		testIt2("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2-s1pt100-r1pt000Deg", mScale1pt100Rotate1pt00Deg);
		
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s0pt960", mScalept96);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s0pt975", mScalept975);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt010", mScale1pt01);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt005", mScale1pt005);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt050", mScale1pt05);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt100", mScale1pt1);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-r0pt521Deg", mRotate0pt521Deg);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-r1pt041Deg", mRotate1pt041Deg);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt013-r0pt750Deg", mScale1pt013Rotate0pt75Deg);
		testIt2("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3-s1pt100-r1pt000Deg", mScale1pt100Rotate1pt00Deg);
		
		testIt("identifyunwarpedrects/source-1.png","identifyunwarpedrects/1");
		testIt("identifyunwarpedrects/source-2.bmp","identifyunwarpedrects/2");
		testIt("identifyunwarpedrects/source-3.bmp","identifyunwarpedrects/3");
		testIt("identifyunwarpedrects/source-4.png","identifyunwarpedrects/4");
		testIt("identifyunwarpedrects/source-5.png","identifyunwarpedrects/5");
		/*byte b = -128;
		byte w = 127;
		byte[] bb = new byte[] {
		 b, b, b, b, b,
		 b, b, b, w, b,
		 b, b, w, b, b,
		 b, b, w, b, b,
		 b, b, w, b, b,
		 b, b, w, b, b,
		 b, b, w, b, b,
		 b, b, b, w, b,
		 b, b, b, b, b
		};
		byte[] contrast = new byte[bb.length];
		contrast(bb,contrast, 5, bb.length/5, 5, 200);
		System.out.println("done!");*/
		
	}	
	
	
	public static void testIt(String s, String pre) {
		try {
			IdentifyUnwarpedRects i = new IdentifyUnwarpedRects(1000*1000);
			BufferedImage si = ImageIO.read(new File(s));
			int w= si.getWidth(), h =si.getHeight();
			BufferedImage bi = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			
			
			
			bi.createGraphics().drawImage(si,0,0,null,null);
			try {
				Thread.sleep(2000);
			} catch(InterruptedException e) {}
			
			
			ArrayList<UnwarpedRect> lWords = new ArrayList<UnwarpedRect>();
			ArrayList<UnwarpedRect> lLines = new ArrayList<UnwarpedRect>();
			for(int j=0; j<10; j++) {
				lWords.clear();
				lLines.clear();
				i.doItFromBufferedImageSourceUsingOurBuffers(
						lWords,
						lLines,
						bi,
						0,
						0,
						w,
						h,
						RECOMMENDED_WIDTH_MIN,
						RECOMMENDED_WIDTH_MAX,
						RECOMMENDED_HEIGHT_MIN,
						RECOMMENDED_HEIGHT_MAX,
						RECOMMENDED_CONTRAST_THRESH,
						true,
						true
					);
			}
			ImageIO.write( i.havingDoneItNowGetGrayImage(w,h), "png", new File(pre+"-g.png"));
			ImageIO.write( i.havingDoneItNowGetContrastImage(w,h), "png", new File(pre+"-c.png"));
			ImageIO.write( i.havingDoneItNowGetMorphedContrastImageCopy(w,h), "png", new File(pre+"-m.png"));
			drawRectsInto( lWords,bi.createGraphics());
			ImageIO.write( bi, "png", new File(pre+"-rw.png"));
			bi.createGraphics().drawImage(si,0,0,null,null);
			drawRectsInto( lLines,bi.createGraphics());
			ImageIO.write( bi, "png", new File(pre+"-rl.png"));
			
			bi.createGraphics().drawImage(si,0,0,null,null);			
			try {
				Thread.sleep(2000);
			} catch(InterruptedException e) {}
			List<UnwarpedRect> l= i.doItHereldFromBufferedImageSourceUsingOurBuffers(
					bi,
					0,
					0,
					w,
					h,
					RECOMMENDED_CONTRAST_THRESH,
					true,
					true
				);
			ImageIO.write( i.havingDoneItNowGetGrayImage(w,h), "png", new File(pre+"-hereld-g.png"));
			ImageIO.write( i.havingDoneItNowGetContrastImage(w,h), "png", new File(pre+"-hereld-c.png"));
			ImageIO.write( i.havingDoneItNowGetMorphedContrastImageCopy(w,h), "png", new File(pre+"-hereld-m.png"));
			drawRectsInto(l,bi.createGraphics());
			ImageIO.write( bi, "png", new File(pre+"-hereld-r.png"));
			
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	public static void testIt2(String s, String pre, Matrix m2h) {
		System.out.println("TestIt2: "+s+" "+pre);
		try {
			IdentifyUnwarpedRects i = new IdentifyUnwarpedRects(1000*1000);
			BufferedImage si = ImageIO.read(new File(s));
			int w= si.getWidth(), h =si.getHeight();
			BufferedImage bi = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			bi.createGraphics().drawImage(si,0,0,null,null);			
			
			ArrayList<UnwarpedRect> lWords = new ArrayList<UnwarpedRect>();
			ArrayList<UnwarpedRect> lLines = new ArrayList<UnwarpedRect>();
			i.doItFromBufferedImageSourceUsingOurBuffers(
					lWords,
					lLines,
					bi,
					0,
					0,
					w,
					h,
					RECOMMENDED_WIDTH_MIN,
					RECOMMENDED_WIDTH_MAX,
					RECOMMENDED_HEIGHT_MIN,
					RECOMMENDED_HEIGHT_MAX,
					RECOMMENDED_CONTRAST_THRESH,
					true,
					true
				);
			
			List<UnwarpedRect> lhereld = i.doItHereldFromBufferedImageSourceUsingOurBuffers(
					bi,
					0,
					0,
					w,
					h,
					RECOMMENDED_CONTRAST_THRESH,
					false,
					false
				);
			
			BufferedImage biwb = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			warpBilinear(biwb,bi,m2h);
			ImageIO.write(biwb,"png",new File(pre+"-wbilin.png"));
			
			BufferedImage biwl = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			BufferedImage biww = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			BufferedImage biwh = new BufferedImage(w,h,BufferedImage.TYPE_INT_ARGB);
			biwl.createGraphics().drawImage(biwb,0,0,null,null);
			biww.createGraphics().drawImage(biwb,0,0,null,null);
			biwh.createGraphics().drawImage(biwb,0,0,null,null);
			
			drawBlankPolys(biwl,bi,m2h,lLines);
			drawBlankPolys(biww,bi,m2h,lWords);
			// we don't draw blank poly's for hereld's. drawBlankPolys(biwh,bi,m2h,lhereld);
			
			/*ImageIO.write(biwmine, "png", new File(pre+"-wblanked.png"));*/
			cutAndPaste(biwl,bi,m2h,lLines,false);
			cutAndPaste(biww,bi,m2h,lWords,false);
			cutAndPaste(biwh,bi,m2h,lhereld,false);
			
			ImageIO.write(biwl,"png",new File(pre+"-wl.png"));
			ImageIO.write(biww,"png",new File(pre+"-ww.png"));
			ImageIO.write(biwh,"png",new File(pre+"-wh.png"));
			
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	
	
	
	
	
	
	public static void drawBlankPolys(BufferedImage dest, BufferedImage src, Matrix m2hSourceToDest, List<UnwarpedRect> l) {
		AffineTransform a = new AffineTransform(
			m2hSourceToDest.get(0,0),
			m2hSourceToDest.get(1,0),
			m2hSourceToDest.get(0,1),
			m2hSourceToDest.get(1,1),
			m2hSourceToDest.get(0,2),
			m2hSourceToDest.get(1,2)
		);
		Graphics2D g = dest.createGraphics();
		for(UnwarpedRect ur: l) {
			if(!ur.filteredOut) {
				g.setColor(new Color(src.getRGB(ur.bgx,ur.bgy)));
				g.fill(a.createTransformedShape(new Rectangle(ur.r.x-1,ur.r.y-1,ur.r.width+2, ur.r.height+2)));
			}
		}
	}
	
	public static void cutAndPaste(BufferedImage dest, BufferedImage src, Matrix m2hSourceToDest, List<UnwarpedRect> l,boolean highlight) {
		for(UnwarpedRect ur: l) {
			if(!ur.filteredOut) {
				Rectangle r = ur.r;
				Matrix m2hDestCentre = m2hSourceToDest.times(JOGLHelper.getMFromD(r.getCenterX(),r.getCenterY(),1.0));
				double destCentreX = m2hDestCentre.get(0,0)/m2hDestCentre.get(2,0);
				double destCentreY = m2hDestCentre.get(1,0)/m2hDestCentre.get(2,0);
				int destX = (int)Math.round(destCentreX - (r.getWidth()/2.0));
				int destY = (int)Math.round(destCentreY - (r.getHeight()/2.0));
				dest.createGraphics().drawImage(src,destX,destY,destX+r.width,destY+r.height,r.x,r.y,r.x+r.width,r.y+r.height,null,null);
				if(highlight) {
					Graphics2D g = dest.createGraphics();
					JOGLHelper.drawRectangleCorrectSize(g,new Rectangle(destX,destY,r.width,r.height));
				}
			}
		}
	}
	
	public static void warpBilinear(BufferedImage dest, BufferedImage src, Matrix m2hSourceToDest) {
		Matrix m2hDestToSource = m2hSourceToDest.inverse();
		for(int destX=0; destX<dest.getWidth(); destX++) {
			for(int destY=0; destY<dest.getHeight(); destY++) {
				Matrix m2hSource = m2hDestToSource.times(JOGLHelper.getMFromD(destX,destY,1.0));
				final double sourceX = m2hSource.get(0,0)/m2hSource.get(2,0);
				final double sourceY = m2hSource.get(1,0)/m2hSource.get(2,0);
				
				// 1 2
				//  p
				// 3 4
				// x increases left to right
				// y increases downwards
				
				final int x1 = (int)Math.floor(sourceX);
				final int x2 = (int)Math.ceil(sourceX);
				final int x3 = x1;
				final int x4 = x2;
				final int y1 = (int)Math.floor(sourceY);
				final int y2 = y1;
				final int y3 = (int)Math.ceil(sourceY);
				final int y4 = y3;
				final double w1,w2,w3,w4;
				
				if(x1>=0 && x2<dest.getWidth() && y1>=0 && y3<dest.getHeight()) {
					
					if(sourceX==x1 && sourceX==x2 && sourceY==y1 && sourceY==y3) {
						w1=1.0;
						w2=w3=w4=0.0;
					} else if(sourceX==x1 && sourceX==x2) {
						w1 = (1.0-Math.abs(sourceY-(double)y1));
						w2 = 0.0;
						w3 = (1.0-Math.abs(sourceY-(double)y3));
						w4 = 0.0;
	
					} else if(sourceY==y1 && sourceY==y3) {
						w1 = (1.0-Math.abs(sourceX-(double)x1));
						w2 = (1.0-Math.abs(sourceX-(double)x2));
						w3 = 0.0;
						w4 = 0.0;
					} else {
						w1 = (1.0-Math.abs(sourceX-(double)x1))*(1.0-Math.abs(sourceY-(double)y1));
						w2 = (1.0-Math.abs(sourceX-(double)x2))*(1.0-Math.abs(sourceY-(double)y2));
						w3 = (1.0-Math.abs(sourceX-(double)x3))*(1.0-Math.abs(sourceY-(double)y3));
						w4 = (1.0-Math.abs(sourceX-(double)x4))*(1.0-Math.abs(sourceY-(double)y4));
					}
					
					int destr = (int)Math.round(
						w1*(double)((src.getRGB(x1,y1)&0xff0000)>>>16)
						+w2*(double)((src.getRGB(x2,y2)&0xff0000)>>>16)
						+w3*(double)((src.getRGB(x3,y3)&0xff0000)>>>16)
						+w4*(double)((src.getRGB(x4,y4)&0xff0000)>>>16)
					);
					assert destr>=0 ;
					destr=Math.min(255,destr);
					
					int destg = (int)Math.round(
							w1*(double)((src.getRGB(x1,y1)&0x00ff00)>>>8)
							+w2*(double)((src.getRGB(x2,y2)&0x00ff00)>>>8)
							+w3*(double)((src.getRGB(x3,y3)&0x00ff00)>>>8)
							+w4*(double)((src.getRGB(x4,y4)&0x00ff00)>>>8)
						);
					assert destg>=0;
					destg=Math.min(255,destg);
					
					int destb = (int)Math.round(
							 w1*(double)((src.getRGB(x1,y1)&0xff)>>>0)
							+w2*(double)((src.getRGB(x2,y2)&0xff)>>>0)
							+w3*(double)((src.getRGB(x3,y3)&0xff)>>>0)
							+w4*(double)((src.getRGB(x4,y4)&0xff)>>>0)
						);
					assert destb>=0;					
					destb=Math.min(255,destb);
					
					int destrgb = 0xff000000|(destr<<16)|(destg<<8)|(destb<<0);
					
					dest.setRGB(destX, destY, destrgb);
					//System.out.println(destX+" "+destY+" "+destrgb+" "+destb);
				} else {

					dest.setRGB(destX, destY, 0xff000000);
				}
			}
		}
	}

	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	/********* class ****************/
	
	
	
	public static class UnwarpedRect {
		public final Rectangle r;
		public final int bgx, bgy;
		public boolean filteredOut;
		public UnwarpedRect(Rectangle r, int bgx, int bgy) {
			this.bgx = bgx;
			this.bgy = bgy;
			this.r = r;
			this.filteredOut = false;
		}		
	}
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios");
	
	
	/*
	 * Colour spaces:
	 * 
	 * cs_gray - one byte per pixel, black=0, 127 is dark gray, -128 is lighter gray, white=-1
	 * mygs - one byte per pixel, black=-128, -1 is dark gray, 0 is lighter gray, white=127
	 * abgr byte array - 4 bytes per pixel in abgr format with 0=black 127=light gray, -128 is darker gray, -1 is white
	 * ARGBIntArray - one int (ie 4 bytes) per pixel in argb format
	 * 
	 */
	
	public static final int RECOMMENDED_CONTRAST_THRESH = (int)Math.round(255.0*0.6);

	public static final int 
	RECOMMENDED_WIDTH_MAX = 140, 
	RECOMMENDED_WIDTH_MIN = 4,
	RECOMMENDED_HEIGHT_MAX = 30, 
	RECOMMENDED_HEIGHT_MIN = 4;
	
	private final byte[] grayImageByteArray, contrastImageByteArray, morphedContrastImageCopyByteArray, morphedContrastImageByteArray;
	
	
	public IdentifyUnwarpedRects(int bufferSize) {
		grayImageByteArray = new byte[bufferSize];
		contrastImageByteArray = new byte[bufferSize];
		morphedContrastImageCopyByteArray = new byte[bufferSize];
		morphedContrastImageByteArray = new  byte[bufferSize];
	}

	public BufferedImage havingDoneItNowGetGrayImage(int w, int h) {
		return getCsgrayBIFromMyGrayByteArray(
			this.grayImageByteArray, 
			w, 
			h, 
			w
		);
	}
	
	public BufferedImage havingDoneItNowGetContrastImage(int w, int h) {
		return getCsgrayBIFromMyGrayByteArray(
			this.contrastImageByteArray, 
			w, 
			h, 
			w
		);
	}
	
	public BufferedImage havingDoneItNowGetMorphedContrastImageCopy(int w, int h) {
		return getCsgrayBIFromMyGrayByteArray(
			this.morphedContrastImageCopyByteArray, 
			w, 
			h, 
			w
		);
	}

	
	
	
	public void doItFromBufferedImageSourceUsingOurBuffers(
			ArrayList<UnwarpedRect> lWords, ArrayList<UnwarpedRect> lLines,
			BufferedImage colbi,
			int subx, int suby, int subw, int subh,
			int minw, int maxw, int minh, int maxh, int contrastThresh,
			boolean timing,
			boolean preserveMorphedContrastImageForDebugAfter
		) {
			final int dilateLookBehind = 1;
			final int dilateLookAhead = 1;
			
			assert(colbi.getWidth()*colbi.getHeight()<=grayImageByteArray.length);
			
			long startTime = System.currentTimeMillis();
			myGSfromColouredBufferedImage(colbi,subx,suby,subw,subh,this.grayImageByteArray);
			long gstime = System.currentTimeMillis();
			
			contrastUnderlyingPxMustBeForeground(this.grayImageByteArray, this.contrastImageByteArray, subw, subh, subw, contrastThresh);
			long ctime = System.currentTimeMillis();

			horizDilateByN(dilateLookBehind, dilateLookAhead, this.contrastImageByteArray,this.morphedContrastImageByteArray, subw, subh, subw);
			
			long dtime = System.currentTimeMillis();
			
			
			if(preserveMorphedContrastImageForDebugAfter) {
				System.arraycopy(this.morphedContrastImageByteArray, 0, this.morphedContrastImageCopyByteArray,0,subw*subh);
			} else {
				// don't bother
			}
					
			long edgestarttime  = System.currentTimeMillis();
			getEdgesWithColourAndSizeCheckPinchHoriz(this.morphedContrastImageByteArray, this.contrastImageByteArray, this.grayImageByteArray, subw, subh, subw, lWords, minw, maxw, minh, maxh,dilateLookAhead,dilateLookBehind);
			long edgeendtime  = System.currentTimeMillis();
			
			filterContainedRectangles(lWords);
			for(UnwarpedRect ur: lWords) {
				lLines.add(
					new UnwarpedRect((Rectangle)ur.r.clone(),ur.bgx,ur.bgy)
				);
			}
			mergeRectangles(lLines,4,this.grayImageByteArray,subw);
			patchUpLeftAlignment(lLines, this.grayImageByteArray, subw);
			
			
			long filterEdgetime  = System.currentTimeMillis();
					
			if(timing) {
				logger.info("Timing:"
						+" PerMillionPx: "+(filterEdgetime-startTime)*1000000/(subw*subh)
						+" Total: "+(filterEdgetime-startTime)
						+" Gs: "+(gstime-startTime)
						+" Con: "+(ctime-gstime)
						+" Dil: "+(dtime-ctime)
						+" Ed: "+(edgeendtime-edgestarttime)
						+" Fil: "+(filterEdgetime-edgeendtime));
			}
			
			return;
		}		
	
	
	
	
	
	
	
	
	
	
	
	

	
	
	
	
	
	
	
	
	/***********IMAGE CONVERSION AND DEBUG *************/
	
	public static BufferedImage getCsgrayBIFromMyGrayByteArray(byte[] a, int w, int h, int vstride) {
		// convert from mygray colourspace
		// where each byte is black=-128, -1 is dark gray, 0 is lighter gray, white=127
		// to cs_gray where black=0, 127 is dark gray, -128 is lighter gray, white=-1
		byte[] b = new byte[a.length];
		for(int i=0; i<a.length; i++) {
			b[i] = intToByteUnsigned( ((int)a[i])+128 );
		}
		a = null;
		return new BufferedImage(
				new ComponentColorModel(
					ColorSpace.getInstance(ColorSpace.CS_GRAY),
					new int[] {8},
					false,
					false,
					ComponentColorModel.OPAQUE,
					DataBuffer.TYPE_BYTE
				),
				Raster.createInterleavedRaster(
					new DataBufferByte(b, b.length),
					w,
					h,
					vstride,
					1,
					new int[] {0}, 
					null
				),
				true,
				null
			);
	}
	
	
	public static void drawRectsInto(List<UnwarpedRect> l, Graphics2D g) {
		for(UnwarpedRect ur: l) {
			if(!ur.filteredOut) {
				g.setColor(Color.RED);
				JOGLHelper.drawRectangleCorrectSize(g,ur.r);
			}
		}
		return;
	}
	
	private static void myGSfromColouredBufferedImage(
		BufferedImage colbi,
		int subx, int suby, int subw, int subh,
		byte[] gs
	) {
		assert colbi.getRaster().getSampleModelTranslateX()==0;
		assert colbi.getRaster().getSampleModelTranslateY()==0;
		assert colbi.getWidth() == colbi.getRaster().getWidth();
		assert colbi.getHeight() == colbi.getRaster().getHeight();
		
		int gsvstep = subw;
		
		if(colbi.getType()==BufferedImage.TYPE_4BYTE_ABGR) {
			byte[] col = ((DataBufferByte)colbi.getRaster().getDataBuffer()).getData();
			int colhstep = 4;
			int colvstep = 4*colbi.getWidth();
			myGsFromABGRByteArray(col, gs, colhstep, colvstep, gsvstep, subx, suby, subw, subh );
		} else if (colbi.getType()==BufferedImage.TYPE_INT_ARGB) {
			int[] col = ((DataBufferInt)colbi.getRaster().getDataBuffer()).getData();
			int colvstep = colbi.getWidth();
			myGsFromARGBIntArray(col, gs, colvstep, gsvstep, subx, suby, subw, subh );
		} else if (colbi.getType()==BufferedImage.TYPE_3BYTE_BGR) {
			byte[] col = ((DataBufferByte)colbi.getRaster().getDataBuffer()).getData();
			int colhstep = 3;
			int colvstep = 3*colbi.getWidth();
			myGsFromBGRByteArray(col, gs, colhstep, colvstep, gsvstep, subx, suby, subw, subh );
		} else {
			throw new RuntimeException("Bad image format");
		}
	}
			
	private static void myGsFromBGRByteArray(byte[] col, byte[] gs, int colhstep, int colvstep, int gsvstep, int subx, int suby, int subw, int subh ) {
		for(int y=suby; y<suby+subh; y++) {
			int indexInGS = (y-suby)*gsvstep-1;				
			int indexInCol = y*colvstep+(subx-1)*colhstep;			
			for(int x=subx; x<subx+subw; x++) {
				indexInCol+=colhstep;
				indexInGS+=1;
				int c0= byteToIntUnsigned(col[indexInCol+0]);
				int c1= byteToIntUnsigned(col[indexInCol+1]);
				int c2= byteToIntUnsigned(col[indexInCol+2]);
				int frameGrey = (c0+c1+c2)/3;
				// now frame grey between 0 and 255
				// but java bytes only store between -128 and 127
				gs[indexInGS] = (byte)(frameGrey-128);
			}
		}
	}
	
	private static void myGsFromARGBIntArray(int[] col, byte[] gs, int colvstep, int gsvstep, int subx, int suby, int subw, int subh ) {
		for(int y=suby; y<suby+subh; y++) {
			int indexInGS = (y-suby)*gsvstep-1;				
			int indexInCol = y*colvstep+(subx-1)*1;			
			for(int x=subx; x<subx+subw; x++) {
				indexInCol+=1;
				indexInGS+=1;
				int v= col[indexInCol];
				int frameGrey = 
					((v & 0xFF) 
					+ ((v & 0xFF00) >>> 8)
					+ ((v & 0xFF0000) >>> 16))/3;
				// now frame grey between 0 and 255
				// but java bytes only store between -128 and 127
				gs[indexInGS] = (byte)(frameGrey-128);
			}
		}
	}
	
	private static void myGsFromABGRByteArray(byte[] col, byte[] gs, int colhstep, int colvstep, int gsvstep, int subx, int suby, int subw, int subh ) {
		for(int y=suby; y<suby+subh; y++) {
			int indexInGS = (y-suby)*gsvstep-1;				
			int indexInCol = y*colvstep+(subx-1)*colhstep;			
			for(int x=subx; x<subx+subw; x++) {
				indexInCol+=colhstep;
				indexInGS+=1;
				int c0= byteToIntUnsigned(col[indexInCol+1]);
				int c1= byteToIntUnsigned(col[indexInCol+2]);
				int c2= byteToIntUnsigned(col[indexInCol+3]);
				int frameGrey = (c0+c1+c2)/3;
				// now frame grey between 0 and 255
				// but java bytes only store between -128 and 127
				gs[indexInGS] = (byte)(frameGrey-128);
			}
		}
	}
	
	private static int byteToIntUnsigned(byte b) {
		// 0=>0, 127=>127, -128=>128, -1=>255
	    return ((int)b) & 0xFF;
	}
	
	private static byte intToByteUnsigned(int i) {
		// the reverse
		// 0=>0, 127=>127, 128=>-128, 255=>-1
		return (byte)(i & 0xFF);
	}

	
	
	
	
	/********** ALGORITHMS ************/
	
	
	
	
		
	
	private static void contrastUnderlyingPxMustBeForeground(byte[] gs, byte[] contrast, int w, int h, int vstep, int contrastThresh) {
		// gs is in mygray colour space
		// sets contrast array with 1 byte per pixel, either 127 or 0.
		
		// a b c
		// d e f
		// g h i
		
		int bgDetectionState = 0;
		boolean bgDetectedAsWhiteNotBlack=true; 
		
		for(int y=1; y<h-1; y++) {
			int index = y*vstep;
			int minl=0,minm=0,minr=0,maxl=0,maxm=0,maxr=0;
			int ce=0, cf=0;
			for(int x=1; x<w-1; x++) {
				index++;
				if(x==1) {
					int ca = gs[index-vstep-1];
					int cb = gs[index-vstep];
					int cd = gs[index-1];
					ce = gs[index];
					int cg = gs[index+vstep-1];
					int ch = gs[index+vstep];
					minl=Math.min(ca, Math.min(cd, cg));
					maxl=Math.max(ca, Math.max(cd, cg));
					minm=Math.min(cb, Math.min(ce, ch));
					maxm=Math.max(cb, Math.max(ce, ch));
				} else {
					minl=minm;
					maxl=maxm;
					minm=minr;
					maxm=maxr;
					ce=cf;
				}
								
				
				int cc = gs[index-vstep+1];
				cf = gs[index+1];
				int ci = gs[index+vstep+1];
				minr=Math.min(cc, Math.min(cf, ci));
				maxr=Math.max(cc, Math.max(cf, ci));
					
				// test for foreground-ness
				boolean ceIsWhiteNotBlack = ce>0;
				boolean ceIsForeground = (ceIsWhiteNotBlack ^ bgDetectedAsWhiteNotBlack);
				
				if(ceIsForeground) {
					
					int iMin = 128+Math.min(minl, Math.min(minm, minr)); 
					int iMax = 128+Math.max(maxl, Math.max(maxm, maxr));
					// now imin and imax are in range 0 to 255.					
					int contrastAtCe = (iMax+iMin)>0 ? (255*(iMax-iMin))/(iMax+iMin) : 0;
					boolean highContrastAtCe = contrastAtCe>contrastThresh;
					
					if(highContrastAtCe) {
						contrast[index] = 127;
						bgDetectionState=0;
					} else {
						contrast[index] = 0;
						bgDetectionState++;
						if(bgDetectionState==3) {
							// for bgDetectionState==N, we require (N+2) foreground pixels in a row
							// and ditto above and below.
							bgDetectedAsWhiteNotBlack = ceIsWhiteNotBlack;
							bgDetectionState=0;
						}
					}
					
				} else {
					contrast[index] = 0;	
					bgDetectionState=0;
				}
			}
		}
	}
	
	
	private static void contrastNoColourCheck(byte[] gs, byte[] contrast, int w, int h, int vstep, int contrastThresh) {
		// gs is in mygray colour space
		// sets contrast array with 1 byte per pixel, either 127 or 0.
		
		// a b c
		// d e f
		// g h i
		
		for(int y=1; y<h-1; y++) {
			int index = y*vstep;
			int minl=0,minm=0,minr=0,maxl=0,maxm=0,maxr=0;
			for(int x=1; x<w-1; x++) {
				index++;
				if(x==1) {
					int ca = gs[index-vstep-1];
					int cb = gs[index-vstep];
					int cd = gs[index-1];
					int ce = gs[index];
					int cg = gs[index+vstep-1];
					int ch = gs[index+vstep];
					minl=Math.min(ca, Math.min(cd, cg));
					maxl=Math.max(ca, Math.max(cd, cg));
					minm=Math.min(cb, Math.min(ce, ch));
					maxm=Math.max(cb, Math.max(ce, ch));
				} else {
					minl=minm;
					maxl=maxm;
					minm=minr;
					maxm=maxr;
				}
				
				int cc = gs[index-vstep+1];
				int cf = gs[index+1];
				int ci = gs[index+vstep+1];
				minr=Math.min(cc, Math.min(cf, ci));
				maxr=Math.max(cc, Math.max(cf, ci));
				
				int iMin = 128+Math.min(minl, Math.min(minm, minr)); 
				int iMax = 128+Math.max(maxl, Math.max(maxm, maxr));
				// now imin and imax are in range 0 to 255.
				
				int c = (iMax+iMin)>0 ? (255*(iMax-iMin))/(iMax+iMin) : 0;
				if(c>contrastThresh) {
					contrast[index] = 127;
				} else {
					contrast[index] = 0;
				}
			}
		}
	}
	
	
	private static void blackOutRect(byte[] closed,int w, int h, int vstep, Rectangle r) {
		for(int y=r.y; y<r.y+r.height; y++) {
			int startInd = vstep*y+r.x;
			int lastInd = vstep*y+r.x+r.width-1;
			int i=startInd;
			while(i<=lastInd) {
				closed[i]=0;
				i++;
			}
		}
	}
	
	
	
	private static void getEdgesWithColourAndSizeCheckPinchHoriz(byte[] morphedContrast,byte[] contrast, byte[] gs,int w, int h, int vstep, List<UnwarpedRect> l, int minw, int maxw, int minh, int maxh, int pinchLeft, int pinchRight) {
		
		// need to make sure there's a 1px black (ie 0) edge around the image
		// and need to make sure no negative pixels to start with 
				
		// need to set black border if reusing same contrast array
		// from a previous operation
		for(int ind=0; ind<vstep; ind++) {
			morphedContrast[ind]=0;
		}
		for(int ind=0; ind<h*vstep; ind+=vstep) {
			morphedContrast[ind]=0;
			morphedContrast[ind+w-1]=0;
		}
		for(int ind=(h-1)*vstep; ind<h*vstep; ind+=1) {
			morphedContrast[ind]=0;
		}
		
		int[] indexDeltas = { 1, -vstep+1, -vstep, -vstep-1, -1, vstep-1, vstep, vstep+1 };
		int[] xDeltas = { 1, 1, 0, -1, -1, -1, 0, 1 };
		int[] yDeltas = { 0, -1, -1, -1, 0, 1, 1, 1 };
		for(int y=1; y<h-1; y++) {
			int xyindex = y*vstep;
			int xyprev  = 0;
			for(int x=1; x<w-1; x++) {
				xyindex++;
				byte xycur = morphedContrast[xyindex];
				if(xycur>0 && xyprev==0) {
					
					// follow it
					
					int deltasIndex = 4;
					int xxyyindex=xyindex, xx=x, yy=y;
					int minx = x, miny=y, maxx=x, maxy=y;
					int newxxyyindex;
					boolean found=false;
					
					// check for single pixel and set secondxxyyIndex
					do {
						deltasIndex = (deltasIndex+1) & 7;
		                newxxyyindex = xxyyindex + indexDeltas[deltasIndex];
		                if( morphedContrast[newxxyyindex] != 0 ) {
		                	found=true;
		                    break;
		                } else {
		                	// do nothing
		                }
		            } while(deltasIndex!=4);	
					
					if(found) {
						int secondxxyyIndex = newxxyyindex;
						boolean first = true;
						deltasIndex = 4;
						while(true) {							
							do {
								deltasIndex = (deltasIndex+1) & 7;
				                newxxyyindex = xxyyindex + indexDeltas[deltasIndex];
				                if( morphedContrast[newxxyyindex] != 0 ) {
				                    break;
				                } else {
				                	// do nothing
				                }
				            } while(true);						
							
							morphedContrast[xxyyindex] = -1;
							
							// register the point
							if(xx<minx) { 
								minx=xx; 
							} else if (xx>maxx) {
								maxx=xx;
							}
							if(yy<miny) { 
								miny=yy; 
							} else if (yy>maxy) {
								maxy=yy;
							}
													
							if(!first && newxxyyindex==secondxxyyIndex && xxyyindex==xyindex  ) {
								break;
							} else {
								// do nothing
							}
							
							first=false;
							xx += xDeltas[deltasIndex];
							yy += yDeltas[deltasIndex];
							xxyyindex = newxxyyindex;
							deltasIndex = (deltasIndex + 4) & 7;
							if(secondxxyyIndex==-1) {
								secondxxyyIndex = xxyyindex;
							}
						}
					}
					
					// translate from minx, maxx, miny, maxy into outr rect coordinates
					// x coords get pinched inwards by 1px because contrast image was dilated horizontally
					// when java draws rectangles it draws the righthand edge at (x+width), ie 1 pixel out
					// so don't be surprised if diagrams look a little wrong.
					
					final int rectMinX = minx+pinchLeft; 
					final int rectMinY = miny;
					final int rectMaxX = maxx-pinchRight; 
					final int rectMaxY = maxy;
					final int rectW = rectMaxX-rectMinX+1;
					final int rectH = rectMaxY-rectMinY+1;
					
					
					// first check size
					if(rectW>=minw && rectW<=maxw && rectH>=minh && rectH<=maxh) {
						
						// do background colour check
						
						int bgx1=0, bgy1=0;
						int bgx2=0, bgy2=0;
						int bgx3=0, bgy3=0;
						boolean found1=false, found2=false, found3=false;
						
						// find a pixel of bg colour
						for(bgy1=rectMinY; bgy1<=rectMaxY; bgy1++) {
							for(bgx1=rectMinX; bgx1<=rectMaxX; bgx1++) {
								if(contrast[bgy1*vstep+bgx1]==0) {
									// we've found a pixel that is either background coloured
									// or foreground coloured with low contrast
									found1=true;
									break;
								} else {
									// carry on
								}
							}
							if(found1) {
								break;
							} else {
								//carry on
							}
						}

						// find another
						for(bgy2=rectMaxY; bgy2>=rectMinY; bgy2--) {
							for(bgx2=rectMaxX; bgx2>=rectMinX; bgx2--) {
								if(contrast[bgy2*vstep+bgx2]==0) {
									// we've found a pixel that is either background coloured
									// or foreground coloured with low contrast
									found2=true;
									break;
								} else {
									// carry on
								}
							}
							if(found2) {
								break;
							} else {
								//carry on
							}
						}
						
						
						// find another
						for(bgy3=rectMinY+rectH/2; bgy3<=rectMaxY; bgy3++) {
							for(bgx3=rectMinX+rectW/2; bgx3<=rectMaxX; bgx3++) {
								if(contrast[bgy3*vstep+bgx3]==0) {
									// we've found a pixel that is either background coloured
									// or foreground coloured with low contrast
									found3=true;
									break;
								} else {
									// carry on
								}
							}
							if(found3) {
								break;
							} else {
								//carry on
							}
						}
						
						if(
							found1 && found2 && found3 
							&& (gs[bgy1*vstep+bgx1])==(gs[bgy2*vstep+bgx2])
							&& (gs[bgy3*vstep+bgx3])==(gs[bgy2*vstep+bgx2])
						) {
							l.add(new UnwarpedRect(new Rectangle(rectMinX,rectMinY,rectW,rectH),bgx1,bgy1));
						} else {
							// failed backrgound colour check
						}
									
						
					} else {
						// do nothing - wrong size
					}
					
				} else {
					// do nothing
				}
				xyprev = xycur;
			}
		}
	}
	
	
	
	
	
	
	
	
	
	
	

	private static void getEdgesNoChecksNoPinch(byte[] morphedContrast,byte[] gs,int w, int h, int vstep, List<UnwarpedRect> l) {
		
		// need to make sure there's a 1px black (ie 0) edge around the image
		// and need to make sure no negative pixels to start with 
				
		// need to set black border if reusing same contrast array
		// from a previous operation
		for(int ind=0; ind<vstep; ind++) {
			morphedContrast[ind]=0;
		}
		for(int ind=0; ind<h*vstep; ind+=vstep) {
			morphedContrast[ind]=0;
			morphedContrast[ind+w-1]=0;
		}
		for(int ind=(h-1)*vstep; ind<h*vstep; ind+=1) {
			morphedContrast[ind]=0;
		}
		
		int[] indexDeltas = { 1, -vstep+1, -vstep, -vstep-1, -1, vstep-1, vstep, vstep+1 };
		int[] xDeltas = { 1, 1, 0, -1, -1, -1, 0, 1 };
		int[] yDeltas = { 0, -1, -1, -1, 0, 1, 1, 1 };
		for(int y=1; y<h-1; y++) {
			int xyindex = y*vstep;
			int xyprev  = 0;
			for(int x=1; x<w-1; x++) {
				xyindex++;
				byte xycur = morphedContrast[xyindex];
				if(xycur>0 && xyprev==0) {
					
					// follow it
					
					int deltasIndex = 4;
					int xxyyindex=xyindex, xx=x, yy=y;
					int minx = x, miny=y, maxx=x, maxy=y;
					int newxxyyindex;
					boolean found=false;
					
					// check for single pixel and set secondxxyyIndex
					do {
						deltasIndex = (deltasIndex+1) & 7;
		                newxxyyindex = xxyyindex + indexDeltas[deltasIndex];
		                if( morphedContrast[newxxyyindex] != 0 ) {
		                	found=true;
		                    break;
		                } else {
		                	// do nothing
		                }
		            } while(deltasIndex!=4);	
					
					if(found) {
						int secondxxyyIndex = newxxyyindex;
						boolean first = true;
						deltasIndex = 4;
						while(true) {							
							do {
								deltasIndex = (deltasIndex+1) & 7;
				                newxxyyindex = xxyyindex + indexDeltas[deltasIndex];
				                if( morphedContrast[newxxyyindex] != 0 ) {
				                    break;
				                } else {
				                	// do nothing
				                }
				            } while(true);						
							
							morphedContrast[xxyyindex] = -1;
							
							// register the point
							if(xx<minx) { 
								minx=xx; 
							} else if (xx>maxx) {
								maxx=xx;
							}
							if(yy<miny) { 
								miny=yy; 
							} else if (yy>maxy) {
								maxy=yy;
							}
													
							if(!first && newxxyyindex==secondxxyyIndex && xxyyindex==xyindex  ) {
								break;
							} else {
								// do nothing
							}
							
							first=false;
							xx += xDeltas[deltasIndex];
							yy += yDeltas[deltasIndex];
							xxyyindex = newxxyyindex;
							deltasIndex = (deltasIndex + 4) & 7;
							if(secondxxyyIndex==-1) {
								secondxxyyIndex = xxyyindex;
							}
						}
					}
					
					// translate from minx, maxx, miny, maxy into outr rect coordinates
					// x AND y coords get pinched inwards by 1px because contrast image was dilated
					// when java draws rectangles it draws the righthand edge at (x+width), ie 1 pixel out
					// so don't be surprised if diagrams look a little wrong if drawn naively
					// seems we have to move everything up by 1 to look correct.
					final int rectMinX = minx; 
					final int rectMinY = miny-1;
					final int rectMaxX = maxx; 
					final int rectMaxY = maxy-1;
					final int rectW = rectMaxX-rectMinX+1;
					final int rectH = rectMaxY-rectMinY+1;
					
					l.add(new UnwarpedRect(new Rectangle(rectMinX,rectMinY,rectW,rectH),0,0));
										
				} else {
					// do nothing
				}
				xyprev = xycur;
			}
		}
	}	
	
	
	
	private static void mergeRectangles(ArrayList<UnwarpedRect> l, int blurH, byte[] gs, int vstep) {
		
		/*
		 * If a rectangle has been compared to every rectangle in the list
		 * and cannot be merged with any then it can be safely ignored for
		 * the rest of the algorithm.
		 * 
		 * If we create a new rectangle by merging two rectangles, i and j,
		 * that we remove from the list then we must go back and compare 
		 * the new rectangle again to all the other rectangles in the list 
		 * that we can't ignore.
		 * 
		 * This is actually equivalent to removing i and replacing j with the
		 * new modified rectangle.
		 * 
		 * So, proceed as follows:
		 * 
		 * Anything preceeding i is safe to ignore - it won't mergeg with 
		 * anything that we have or could create.
		 * 
		 * For each rectangle i:
		 * 		Set changed = false
		 * 		For each rectangle j, where j starts just after the position of i:
		 * 			if j intersects with i
		 * 				set changed = true
		 * 				set j:= union (j,i)
		 * 				remove the last one returned by i
		 * 				jump out of the loop
		 * 		if changed
		 * 		else
		 * 			leave the last one returned by i as we can ignore it now.
		 * 
		 * We use ArrayLists so actually we don't want to remove items, so we set
		 * the filteredOut property. 
		 * 
		 */	
		
		ListIterator<UnwarpedRect> i = l.listIterator();
		while(i.hasNext()) {
			UnwarpedRect iu = i.next();
			if(!iu.filteredOut) {
				boolean changed=false;
				Rectangle ir = iu.r;
				byte iCol = gs[iu.bgy*vstep+iu.bgx];
				ListIterator<UnwarpedRect> j = l.listIterator(i.nextIndex());
				while(!changed && j.hasNext()) {
					UnwarpedRect ju = j.next();
					if(
						!ju.filteredOut
						&& rectsIntersectBlurH(ir,ju.r,blurH)
						&& iCol == gs[ju.bgy*vstep+ju.bgx]
					) {
						assert ju!=iu;
						ju.r.add(ir);
						iu.filteredOut=true;
						changed=true;
					} else {
						// do nothing
					}
				}
			}
		}
	}	
	
	
	private static boolean rectsIntersectBlurH(Rectangle r, Rectangle s, int blurH) {
		//return new Rectangle(r.x-2*blurH-1,r.y,r.width+4*blurH+2, r.height).intersects(s);
		
		int sw = s.width+2*blurH+2;
		int sh = s.height;
		int rw = r.width+2*blurH;
		int rh = r.height;
		if (rw <= 0 || rh <= 0 || sw <= 0 || sh <= 0) {
		    return false;
		}
		int sx = s.x-blurH-1;
		int sy = s.y;
		int rx = r.x-blurH;
		int ry = r.y;
		rw += rx;
		rh += ry;
		sw += sx;
		sh += sy;
		//      overflow || intersect
		return ((rw < rx || rw > sx) &&
			(rh < ry || rh > sy) &&
			(sw < sx || sw > rx) &&
			(sh < sy || sh > ry));
		
	}
	
	
	
	
	
	private static void filterContainedRectangles(ArrayList<UnwarpedRect> l) {
		ListIterator<UnwarpedRect> i = l.listIterator();
		while(i.hasNext()) {
			UnwarpedRect iu = i.next();
			Rectangle ir = iu.r;
			ListIterator<UnwarpedRect> j = l.listIterator();
			while(j.hasNext()) {
				Rectangle jr = j.next().r;
				if(jr.contains(ir) && jr!=ir) {
					iu.filteredOut=true;
					break;
				}
			}
		}	

	}
	
	private static class FourIntTuple {
		int a, b, c, d;
	}
	
	private static void patchUpLeftAlignment(ArrayList<UnwarpedRect> lLines, byte[] gs, int vstep) { 
		HashMap<Integer,FourIntTuple> leftToNWWsq = new HashMap<Integer,FourIntTuple>();
		final int pixelErrorB = 1;
		int nNotFiltered=0;
		for(UnwarpedRect ur: lLines) {
			if(!ur.filteredOut) {
				nNotFiltered++;
				// round to the left for odd pixels - helps with 1px ps->bmp rendering errors.
				int left = ur.r.x;
				int width = ur.r.width;
				FourIntTuple tit = leftToNWWsq.get(left);
				if(tit!=null) {
					tit.a++;
					tit.b+=width;
					tit.c=Math.max(width,tit.c);
					tit.d+=width*width;
				} else {
					tit = new FourIntTuple();
					tit.a=1;
					tit.b=width;
					tit.c=width;
					tit.d=width*width;
					leftToNWWsq.put(left,tit);
				}
			}
		}
		for(Map.Entry<Integer, FourIntTuple> entry: leftToNWWsq.entrySet()) {
			FourIntTuple tit = entry.getValue();
			if(tit.a>=3) {
				int left = entry.getKey();
				int meanW = tit.b/tit.a;
				int varW = tit.d/tit.a-meanW*meanW;
				int sdW = (int) Math.sqrt(varW);
				if(varW!=0 && sdW*3<meanW && tit.c<meanW+2*sdW) {
					for(UnwarpedRect ur: lLines) {
						if(
							!ur.filteredOut 
							&& ur.r.x>=left-pixelErrorB 
							&& ur.r.x<=left+pixelErrorB
							&& ur.r.x+ur.r.width<left+tit.c
						) {
							int newWidth = left+tit.c-ur.r.x;
							byte bg = gs[ur.bgy*vstep+ur.bgx];
							int sampleXstart = ur.r.x+ur.r.width;
							int sampleStep = (newWidth-ur.r.width)/5;
							int sampleY = ur.r.y+ur.r.height/2;
							if(
								   gs[sampleY*vstep+sampleXstart+sampleStep]==bg
								&& gs[sampleY*vstep+sampleXstart+2*sampleStep]==bg
								&& gs[sampleY*vstep+sampleXstart+3*sampleStep]==bg
								&& gs[sampleY*vstep+sampleXstart+4*sampleStep]==bg
							) {
								ur.r.width = newWidth;
							} else {
								// doesn't have same bg colour at sample points
								//System.out.println(bg);
								//System.out.println(gs[sampleY*vstep+sampleXstart+sampleStep]);
								//System.out.println("========");
							}
						} else {
							// either filtered out or x coord is nowhere near!
						}
					}
				} else {
					// sd is too big or max is too far from mean
				}
			} else {
				// not enough lines
			}
		}
	}
	
	
	private static void dilateOrErode(byte[] src, byte[] dest, int w, int h, int vstep, boolean dilateNotErode) {

		// a b c
		// d e f
		// g h i
		
		for(int y=1; y<h-1; y++) {
			int index = y*vstep;
			int ca=0, cb=0, cc=0, cd=0, ce=0, cf=0, cg=0, ch=0, ci=0;
			for(int x=1; x<w-1; x++) {
				index++;
				if(x==1) {
					ca = src[index-vstep-1];
					cb = src[index-vstep];
					cd = src[index-1];
					ce = src[index];
					cg = src[index+vstep-1];
					ch = src[index+vstep];
				} else {
					ca = cb; 
					cb = cc;
					cd=ce;
					ce=cf;
					cg=ch;
					ch=ci;
				}
				cc = src[index-vstep+1];
				cf = src[index+1];
				ci = src[index-vstep+1];
				
				if(dilateNotErode) {
                    if(ca>0 || cb>0 || cc>0 || cd>0 || ce>0 || cf>0 || cg>0 ||ch>0 || ci>0) {
                    	dest[index] = 127;
                    } else {
                    	dest[index] = 0;
                    }
				} else {              
                    if(ca==0 || cb==0 || cc==0 || cd==0 || ce==0 || cf==0 || cg==0 ||ch==0 || ci==0) {
                    	dest[index] = 0;
                    } else {
                    	dest[index] = 127;
                    }
				}
			}
		}
	}

	
	
	
	
	
	private static void horizDilate(byte[] src, byte[] dest, int w, int h, int vstep) {

		// d e f
		
		for(int y=1; y<h-1; y++) {
			int index = y*vstep;
			boolean  cdgz=false, cegz=false, cfgz=false;
			for(int x=1; x<w-1; x++) {
				index++;
				if(x==1) {
					cdgz = src[index-1]>0;
					cegz = src[index]>0;
				} else {
					cdgz=cegz;
					cegz=cfgz;
				}
				cfgz = src[index+1]>0;
				
				if(cdgz || cegz || cfgz) {
                	dest[index] = 127;
                } else {
                	dest[index] = 0;
                }
			}
		}
	}
	
	private static void horizDilateByN(final int lookBehind, final int lookAhead, byte[] src, byte[] dest, final int w, final int h, final int vstep) {

		// source and dest can be the same
		
		// eg lookBehind=3, lookAhead=3
		// ... a b c d e f g h ...
		//           *             the one we set, dest[index]
		//                 *       the one we read src[index+lookAhead]
		//               *         the last one we read
		
		// two dark pixels on the same row must be seperated 
		// by at least lookBehind+lookAhead+1 white pixels
		// in order to be not joined up by this operation.
		
		if(lookBehind==1 && lookAhead==1) {
			horizDilate(src,dest,w,h,vstep);
			return;
		}
		
		final int bufSize = lookBehind+1+lookAhead;
		boolean[] buf = new boolean[bufSize];
		int currentBufInd = 0;
		boolean lastRead;
		
		for(int y=0; y<h; y++) {
			int index = y*vstep;
			
			lastRead=false;
			for(int i=0; i<bufSize; i++) {
				buf[i]=false;
			}
			
			for(int x=0; x<w-lookAhead; x++) {
				
				index++;
				
				currentBufInd++;
				if(currentBufInd>=bufSize) { currentBufInd-=bufSize; }
				
				boolean thisRead = src[index+lookAhead]>0;
				
				buf[currentBufInd]=thisRead;
				
				if(thisRead || lastRead || or(buf)) {
					dest[index] = 127;
				} else {
					dest[index] = 0;
				}
               
				lastRead=thisRead;
			}
		}
	}
	
	
	static private boolean or(boolean[] buf) {
		final int n = buf.length;
		for(int i=0; i<n; i++) {
			if(buf[i]) { 
				return true; 
			}
		}
		return false;
	}
	
	
	
	
	
	
	
	
	
	
	
	

	/******************* OLD CODE FOR TESTING ONLY ******************/
	
	public List<UnwarpedRect> doItFromBufferedImageSourceUsingOurBuffersDilate1(
			BufferedImage colbi,
			int subx, int suby, int subw, int subh,
			int minw, int maxw, int minh, int maxh, int contrastThresh,
			boolean timing,
			boolean preserveMorphedContrastImageForDebugAfter
		) {
			final int dilateLookBehind = 1;
			final int dilateLookAhead = 1;
			
			assert(colbi.getWidth()*colbi.getHeight()<=grayImageByteArray.length);
			ArrayList<UnwarpedRect> l = new ArrayList<UnwarpedRect>(100);
			
			long startTime = System.currentTimeMillis();
			myGSfromColouredBufferedImage(colbi,subx,suby,subw,subh,this.grayImageByteArray);
			long gstime = System.currentTimeMillis();
			
			contrastUnderlyingPxMustBeForeground(this.grayImageByteArray, this.contrastImageByteArray, subw, subh, subw, contrastThresh);
			long ctime = System.currentTimeMillis();

			horizDilateByN(dilateLookBehind, dilateLookAhead, this.contrastImageByteArray,this.morphedContrastImageByteArray, subw, subh, subw);
			
			long dtime = System.currentTimeMillis();
			
			
			if(preserveMorphedContrastImageForDebugAfter) {
				System.arraycopy(this.morphedContrastImageByteArray, 0, this.morphedContrastImageCopyByteArray,0,subw*subh);
			} else {
				// don't bother
			}
					
			long edgestarttime  = System.currentTimeMillis();
			getEdgesWithColourAndSizeCheckPinchHoriz(this.morphedContrastImageByteArray, this.contrastImageByteArray, this.grayImageByteArray, subw, subh, subw, l, minw, maxw, minh, maxh,dilateLookAhead,dilateLookBehind);
			long edgeendtime  = System.currentTimeMillis();
			
			filterContainedRectangles(l);
			long filterEdgetime  = System.currentTimeMillis();
					
			if(timing) {
				logger.info("Timing:"
						+" PerMillionPx: "+(filterEdgetime-startTime)*1000000/(subw*subh)
						+" Total: "+(filterEdgetime-startTime)
						+" Gs: "+(gstime-startTime)
						+" Con: "+(ctime-gstime)
						+" Dil: "+(dtime-ctime)
						+" Ed: "+(edgeendtime-edgestarttime)
						+" Fil: "+(filterEdgetime-edgeendtime));
			}
			
			return l;
		}	
	
	
	public List<UnwarpedRect> doItFromBufferedImageSourceUsingOurBuffersDil3(
		BufferedImage colbi,
		int subx, int suby, int subw, int subh,
		int minw, int maxw, int minh, int maxh, int contrastThresh,
		boolean timing,
		boolean preserveMorphedContrastImageForDebugAfter
	) {
		final int dilateLookBehind = 3;
		final int dilateLookAhead = 3;
		
		assert(colbi.getWidth()*colbi.getHeight()<=grayImageByteArray.length);
		ArrayList<UnwarpedRect> l = new ArrayList<UnwarpedRect>(100);
		
		long startTime = System.currentTimeMillis();
		myGSfromColouredBufferedImage(colbi,subx,suby,subw,subh,this.grayImageByteArray);
		long gstime = System.currentTimeMillis();
		
		contrastUnderlyingPxMustBeForeground(this.grayImageByteArray, this.contrastImageByteArray, subw, subh, subw, contrastThresh);
		long ctime = System.currentTimeMillis();

		horizDilateByN(dilateLookBehind, dilateLookAhead, this.contrastImageByteArray,this.morphedContrastImageByteArray, subw, subh, subw);
		
		long dtime = System.currentTimeMillis();
		
		
		if(preserveMorphedContrastImageForDebugAfter) {
			System.arraycopy(this.morphedContrastImageByteArray, 0, this.morphedContrastImageCopyByteArray,0,subw*subh);
		} else {
			// don't bother
		}
				
		long edgestarttime  = System.currentTimeMillis();
		getEdgesWithColourAndSizeCheckPinchHoriz(this.morphedContrastImageByteArray, this.contrastImageByteArray, this.grayImageByteArray, subw, subh, subw, l, minw, maxw, minh, maxh,dilateLookAhead,dilateLookBehind);
		long edgeendtime  = System.currentTimeMillis();
		
		filterContainedRectangles(l);
		long filterEdgetime  = System.currentTimeMillis();
				
		if(timing) {
			logger.info("Timing:"
					+" PerMillionPx: "+(filterEdgetime-startTime)*1000000/(subw*subh)
					+" Total: "+(filterEdgetime-startTime)
					+" Gs: "+(gstime-startTime)
					+" Con: "+(ctime-gstime)
					+" Dil: "+(dtime-ctime)
					+" Ed: "+(edgeendtime-edgestarttime)
					+" Fil: "+(filterEdgetime-edgeendtime));
		}
		
		return l;
	}

	
	public List<UnwarpedRect> doItHereldFromBufferedImageSourceUsingOurBuffers(
			BufferedImage colbi,
			int subx, int suby, int subw, int subh, int contrastThresh,
			boolean timing,
			boolean preserveMorphedContrastImageForDebugAfter
		) {
		
		assert(colbi.getWidth()*colbi.getHeight()<=grayImageByteArray.length);
		ArrayList<UnwarpedRect> l = new ArrayList<UnwarpedRect>();
		byte[] temp = new byte[subw*subh];

		long startTime = System.currentTimeMillis();
		myGSfromColouredBufferedImage(colbi,subx,suby,subw,subh,this.grayImageByteArray);
		long gstime = System.currentTimeMillis();
		
		contrastNoColourCheck(this.grayImageByteArray, this.contrastImageByteArray, subw, subh, subw, contrastThresh);
		long ctime = System.currentTimeMillis();

		dilateOrErode(this.contrastImageByteArray,temp, subw, subh, subw,true);
		dilateOrErode(temp,this.morphedContrastImageByteArray, subw, subh, subw,false);
		
		long dtime = System.currentTimeMillis();
		
		
		if(preserveMorphedContrastImageForDebugAfter) {
			System.arraycopy(this.morphedContrastImageByteArray, 0, this.morphedContrastImageCopyByteArray,0,subw*subh);
		} else {
			// don't bother
		}
				
		long edgestarttime  = System.currentTimeMillis();
		getEdgesNoChecksNoPinch(this.morphedContrastImageByteArray, this.grayImageByteArray, subw, subh, subw, l);
		long edgeendtime  = System.currentTimeMillis();

		filterContainedRectangles(l);
		
				
		if(timing) {
			/*logger.info("Grayscale: "+(gstime-startTime));
			logger.info("Contrast: "+(ctime-gstime));
			logger.info("Dilate: "+(dtime-ctime));
			logger.info("Edge: "+(edgeendtime-edgestarttime));*/
		}
		
		return l;
	}
	
	
}



