/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;

import t3.hrd.util.FPSLimiter;

public abstract class AnimatorThread extends Thread {
    private volatile boolean reqToComplete = false;
    private final FPSLimiter fpsLimiter;
    private final PortfolioServer ps;
    private long startTime;
    private volatile boolean isCompleted = false;
    
    public final void reqToComplete() {
        this.reqToComplete = true;
    }
    
    
    
    public final boolean isCompleted() {
        return isCompleted;
    }
        
    public AnimatorThread(long targetMsBetweenFrames, PortfolioServer ps) {
        this.ps = ps;
        this.fpsLimiter = new FPSLimiter(targetMsBetweenFrames);
    }
    
    public abstract void runOnEachLoop();
    public void runOnComplete() { }
        
    private Runnable onEachLoop = new Runnable() { public void run() {  
        AnimatorThread.this.runOnEachLoop();
    } };
    
    public final void run() {
        try {
            this.setPriority(Thread.MIN_PRIORITY);
            this.startTime = System.currentTimeMillis();
            
            while(!this.reqToComplete) {
                ps.performActionAsynchronouslyByCurrentThread(this.onEachLoop);
                if(!this.reqToComplete) {
                    this.fpsLimiter.oneFrame();
                    Thread.currentThread().yield();
                } else {
                    // we'll be finished soon anyway
                }
            }
            ps.performActionAsynchronouslyByCurrentThread(new Runnable() { public void run() {
                AnimatorThread.this.runOnComplete();
            }});
            this.isCompleted = true; 
        } catch(InterruptedException e) {
            ps.fatalError(e);
        }
    }

    public final long getStartTime() {
        return startTime;
    }
    
    
    
}
