/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;

import java.util.List;

import t3.hrd.state.ScaRotTraTransformImmutableUniformScale;

/**
 * A thread that animates a set of portfolios by interpolating each from its current 
 * PORTtoPPORT transform to a specified transform. Call interrupt() to stop the 
 * thread animating.
 * 
 * @author pjt40
 *
 */
public class AnimatorJobsThread extends AnimatorThread {
	public static class AnimationJob {
		public final Portfolio p;
		public final ScaRotTraTransformImmutableUniformScale initialPORTtoPPORT;
		public final ScaRotTraTransformImmutableUniformScale finalPORTtoPPORT;
		
		/**
		 * @param p
		 * @param finalPORTtoPPORT
		 */
		public AnimationJob(Portfolio p, ScaRotTraTransformImmutableUniformScale finalPORTtoPPORT) {
			this.p = p;
			this.finalPORTtoPPORT = finalPORTtoPPORT;
			this.initialPORTtoPPORT = p.gettPORTtoPPORT();
		}
	}
	
	private final List<AnimationJob> animationJobs;
	private final long durationMillis;
    private Runnable onCompletion;
	
	
	/**
	 * @param jobs
	 * @param durationMillis
	 * @param intervalMillis
	 * @param onCompletion
	 */
	public AnimatorJobsThread(long targetMsBetweenFrames, PortfolioServer ps, List<AnimationJob> jobs, long durationMillis, Runnable onCompletion) {
        super(targetMsBetweenFrames, ps);
		this.animationJobs = jobs;
		this.durationMillis = durationMillis;
		this.onCompletion = onCompletion;
	}
    
    
    


    @Override
    public void runOnComplete() {
        this.onCompletion.run();
    }




    @Override
    public void runOnEachLoop() {
        double progresst = ((double)(System.currentTimeMillis()-this.getStartTime()))/this.durationMillis;
        if(progresst>=1.0) {
            progresst=1.0;
            this.reqToComplete();
        }
        final double progress = progresst;
        
        for(AnimationJob j:AnimatorJobsThread.this.animationJobs) {
            j.p.setPORTtoPPORT(
                new ScaRotTraTransformImmutableUniformScale(
                    (1.0-progress) * j.initialPORTtoPPORT.getScale()
                        + progress * j.finalPORTtoPPORT.getScale(),
                    (1.0-progress) * j.initialPORTtoPPORT.getThetaClockwise()
                        + progress * j.finalPORTtoPPORT.getThetaClockwise(),
                    (1.0-progress) * j.initialPORTtoPPORT.getTx()
                        + progress * j.finalPORTtoPPORT.getTx(),
                    (1.0-progress) * j.initialPORTtoPPORT.getTy()
                        + progress * j.finalPORTtoPPORT.getTy()
                )
            );                                  
        }
    }
	
}
