/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.util;


import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.geom.Area;
import java.awt.geom.Rectangle2D;
import java.lang.reflect.Constructor;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import Jama.Matrix;

import t3.hrd.input.PointInputDevice;
import t3.hrd.input.ShapeInputDevice;
import t3.hrd.renderer.ProjectorConfig;
import t3.hrd.state.JOGLHelper;
import t3.hrd.state.JOGLHelper.PropertiesException;


/**
 * Routines to create PointInputDevices and ProjectorConfigs from a Properties object.
 * The Properties format is described elsewhere.
 * 
 * @author pjt40
 *
 */
public class LoadConfigsFromProperties {

	private static final Logger logger = Logger.getLogger("t3.hrd.util");
	
    public static ShapeInputDevice loadShapeInputDeviceFromProperties(int clientId, Properties p) {
        try {
            if(p.containsKey("hrd.shapeInputDevice.className")) {
                Class c;
                String prefix = "hrd.shapeInputDevice.";
                String className = JOGLHelper.props_getString(p,prefix+"className");
                try {
                     c = Class.forName(className);
                } catch(Exception e) {
                    throw new RuntimeException("Class "+className+" not found.");
                }
                if(!ShapeInputDevice.class.isAssignableFrom(c)) {
                    throw new RuntimeException("Class "+className+" is not a subclass of PointInputDevice.");
                }
                Constructor con = c.getConstructor(new Class[] {int.class,Properties.class,String.class} );
                try {
                    return( (ShapeInputDevice)(con.newInstance(new Object[] { clientId, p, prefix})));
                } catch(Exception e) {
                    throw new RuntimeException(e);
                }
            } else {
                // no shape input device
                return null;
            }
        } catch(PropertiesException e) {
            throw new RuntimeException(e);
        } catch(NoSuchMethodException e) {
            throw new RuntimeException(e);
        }                
    }   
    
    public static Matrix loadScaRotTraMatrixFromProperties(Properties p, String prefix) throws PropertiesException {
        double sx = JOGLHelper.props_getDouble(p, prefix+"scaleX");
        double sy = JOGLHelper.props_getDouble(p, prefix+"scaleY");
        double rThetaClockwise = JOGLHelper.props_getDouble(p, prefix+"rotClockwiseDeg")*2*Math.PI/360.0;
        double tx = JOGLHelper.props_getDouble(p, prefix+"trX");
        double ty = JOGLHelper.props_getDouble(p, prefix+"trY");
        return JOGLHelper.get2hmTranslation(tx,ty).times(
                JOGLHelper.get2hmRotation(rThetaClockwise).times(
                    JOGLHelper.get2hmScale(sx,sy)
                )
        );
    }
    
    public static Rectangle2D loadRectangle2DFromProperties(Properties p, String prefix) throws PropertiesException {
        double x0 = JOGLHelper.props_getDouble(p, prefix+"0x");
        double y0 = JOGLHelper.props_getDouble(p, prefix+"0y");
        double x1 = JOGLHelper.props_getDouble(p, prefix+"1x");
        double y1 = JOGLHelper.props_getDouble(p, prefix+"1y");
        double minx = Math.min(x0,x1);
        double w = Math.max(x0,x1)-minx;
        double miny = Math.min(y0,y1);
        double h = Math.max(y0,y1)-miny;
        return new Rectangle2D.Double(minx,miny,w,h);                
    }
    
	public static LinkedList<PointInputDevice> loadPointInputDevicesFromProperties(int clientId, boolean accurate, Properties p) {
		LinkedList<PointInputDevice> r = new LinkedList<PointInputDevice>();
		try {
			int i=0;
			while(p.containsKey("hrd.pointInputDevices."+i+".className")) {
				Class c;
				String prefix = "hrd.pointInputDevices."+i+".";
				String className = JOGLHelper.props_getString(p,prefix+"className");
				int personId = JOGLHelper.props_getInt(p,prefix+"personId");
                Matrix mINPUTtoDESK = loadScaRotTraMatrixFromProperties(p,prefix+"INPUTtoDESK.");
				try {
					 c = Class.forName(className);
				} catch(Exception e) {
					throw new RuntimeException("Class "+className+" not found.");
				}
				if(!PointInputDevice.class.isAssignableFrom(c)) {
					throw new RuntimeException("Class "+className+" is not a subclass of PointInputDevice.");
				}
				Constructor con = c.getConstructor(new Class[] {int.class,int.class,Matrix.class, boolean.class, Properties.class,String.class} );
				try {
					r.add( (PointInputDevice)(con.newInstance(new Object[] { clientId, personId, mINPUTtoDESK, accurate, p, prefix })));
				} catch(Exception e) {
					throw new RuntimeException(e);
				}
				
				i++;
				
			}
		} catch(PropertiesException e) {
			throw new RuntimeException(e);
		} catch(NoSuchMethodException e) {
			throw new RuntimeException(e);
		}
		return r;
				
	}	
	
	public static ProjectorConfig getProjectorConfigForGraphicsDeviceIndex(List<ProjectorConfig> pcs, int i) {
		for(ProjectorConfig pc: pcs) {
			if(pc.graphicsDeviceIndex==i) {
				return pc;
			} else {
				continue;
			}
		}
		return null;
	}
	
	
    
    
    public static Area loadDESKvisibleAreaMaskFromProperties(Properties p) throws JOGLHelper.PropertiesException {
        return new Area(loadRectangle2DFromProperties(p,"hrd.projectorConfig.DESKvisibleRectMask."));
    }
    
	public static LinkedList<ProjectorConfig> loadProjectorConfigsFromProperties(Properties p) throws JOGLHelper.PropertiesException {
		LinkedList<ProjectorConfig> pcs = new LinkedList<ProjectorConfig>();
		List<Integer> graphicsDeviceIndexes = JOGLHelper.props_getIntList(p, "hrd.projectorConfig.graphicsDeviceIndexes");
		for(int graphicsDeviceOneBasedIndex: graphicsDeviceIndexes) {
			ProjectorConfig pc = loadProjectorConfigFromProperties(p, graphicsDeviceOneBasedIndex);
			pcs.add(pc);
		}
		return pcs;
	}
	
	public static void saveProjectorConfigsToProperties(LinkedList<ProjectorConfig> pcs, Properties p) throws JOGLHelper.PropertiesException {
		
		List<Integer> graphicsDeviceIndexes = new LinkedList<Integer>();
		for(ProjectorConfig pc: pcs) {
			graphicsDeviceIndexes.add(pc.graphicsDeviceIndex);
		}
		JOGLHelper.props_setList(graphicsDeviceIndexes, p, "hrd.projectorConfig.graphicsDeviceIndexes" );
		
		for(ProjectorConfig pc: pcs) {
			saveProjectorConfigToProperties(pc, p);
		}
	}	
		
	
	private static void saveProjectorConfigToProperties(ProjectorConfig pc, Properties p )  {
		
		GraphicsDevice graphicsDevice =  GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices()[pc.graphicsDeviceIndex];
		
		p.setProperty(
			"hrd.projectorConfig."+pc.graphicsDeviceIndex+".fullScreenExclusive",
			""+pc.window_fullScreenExclusive
			);
		
		for(int k=0; k<4; k++) {
			p.setProperty(
				"hrd.projectorConfig."+pc.graphicsDeviceIndex+".DESKcorner"+k+"x",
				""+(
					JOGLHelper.unhomogeniseD( JOGLHelper.getDFromM(pc.mDESKcornersOfProjectorLimitClockwiseInWOGL[k]))[0]
				)
			);
			p.setProperty(
					"hrd.projectorConfig."+pc.graphicsDeviceIndex+".DESKcorner"+k+"y",
					""+(
						JOGLHelper.unhomogeniseD( JOGLHelper.getDFromM(pc.mDESKcornersOfProjectorLimitClockwiseInWOGL[k]))[1]
					)
				);
		}
		
	}
	
	
	private static boolean SUPPRESS_GDI_ERROR = true;
	
	private static ProjectorConfig loadProjectorConfigFromProperties(Properties p, int graphicsDeviceIndex) throws JOGLHelper.PropertiesException {
		
		ProjectorConfig pc = new ProjectorConfig();
		GraphicsDevice[] ds = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices();   
		 
		if(graphicsDeviceIndex<0 || graphicsDeviceIndex>ds.length-1) { 
			if(SUPPRESS_GDI_ERROR) {
				logger.warning("Graphics device does not exist but continuing anyway...");
				pc.graphicsDeviceIndex = 0;
			} else {
				throw new JOGLHelper.PropertiesException("Graphics device index "+graphicsDeviceIndex+" out of bounds - there are only "+ds.length+" graphics devices.");
			}
		} else {
			pc.graphicsDeviceIndex = graphicsDeviceIndex;
		}
		
		pc.window_fullScreenExclusive = JOGLHelper.props_getBoolean(p,"hrd.projectorConfig."+graphicsDeviceIndex+".fullScreenExclusive");
		
		pc.mDESKcornersOfProjectorLimitClockwiseInWOGL = new Jama.Matrix[4];
		
		for(int k=0; k<4; k++) {
			pc.mDESKcornersOfProjectorLimitClockwiseInWOGL[k] =
				JOGLHelper.getMFromD(
					JOGLHelper.props_getDouble(p,"hrd.projectorConfig."+graphicsDeviceIndex+".DESKcorner"+k+"x"),
					JOGLHelper.props_getDouble(p,"hrd.projectorConfig."+graphicsDeviceIndex+".DESKcorner"+k+"y"),
					1
				);			
		}
		
		return pc;
	}
	
	
	

}
