/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;

/* 
 * Not thread-safe. Synchronize on its statemanager before you do anything.
 * toopt don't really need mdesktotile - set it to null on change and recalculate on getmdesktotile if necessary 
 */


import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Logger;

import Jama.Matrix;

/**
 * A Tile is a rectangular image with a width and height that is drawn on the desk 
 * with a specific width, height and orientation, and at a specific location. 
 * Create new Tiles by calling the appropriate
 * method on the a StateManager object. A Tile can be backed, split or unbacked. 
 * <p>
 * A backed tile is the simplest tile. It has a BufferedImage that stores its contents.
 * <p>
 * An unbacked tile has no BufferedImage, and if you call the method
 * to get the image then the system will create an empty bufferedimage, call 
 * callback_repaintUnbackedTile to paint into the empty image, and then return the result.
 * You can set up the StateManager to always produce unbacked tiles rather than backed tiles.
 * <p>
 * A split tile is created if you create a tile with a large number of pixels 
 * (depending on how you set up your StateManager). It has no bufferedImage itself
 * but refers to several smaller backed tiles.
 * <p>
 * A Tile uses the bufferedimage format TYPE_INT_ARGB (i.e. 32 bits per pixel) by default but 
 * you can specify a flag to change this to TYPE_USHORT_565RGB (i.e. 16 bits per pixel). Pixel
 * data is sent to the graphics card via OpenGL in the same format as the BufferedImage.
 * <p>
 * When stored as a texture for rendering, the texture has format RGBA8 (i.e. 32 bits
 * per pixel) by default but you can specify a flag to change this to RGB4 (i.e. 12 bits per
 * pixel).
 * <p>
 * TILE space is a 2D homogenous coordinate space corresponding to the coordinate space of the tile's image.
 * There are methods to obtain the corner points, outline and bounding box of the tile in TILE space
 * and also in DESK space (i.e. once the tile has been affine-transformed).
 * <p>
 * Threading notes: this class is not thread-safe. You must use some kind of locking scheme
 * if you use it in a multithreaded environment.
 * 
 * @author pjt40
 *
 */
public class Tile extends OrderedElement {

	
	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	// tileid for start of series of divtiles
	private static int nextDivTileId=3000;
	
	/**
	 * Tile not transformed - draw texture pixel for pixel into frame buffer
	 */
	public static final int FLAG_NO_TRANSFORM = 1;	
	public static final int FLAG_BI_USHORT565RGB_NOT_INTARGB = 2;
	public static final int FLAG_TEXTUREALWAYSRESIDENTWHENTILEVISIBLE = 4;
	public static final int FLAG_TEXTUREALWAYSRESIDENTEVENWHENTILEINVISIBLE= 8;	
	
	public static final boolean TILEtoDESKflipsVert = true;
	public static final boolean TILEtoDESKflipsHoriz = false;
	
	public static final int TILETYPE_BACKED= 0;
	public static final int TILETYPE_SPLIT = 1;
	public static final int TILETYPE_UNBACKED = 2;
	
	
	/**
	 * Width of the usable tile image in pixels.
	 */
	public final int tileWidth;
	
	
	/**
	 * Height of the usable tile image in pixels.
	 */
	public final int tileHeight;
	
	
	/**
	 * Width of the tile's buffered image in pixels. 
	 * Not set if the tile is unbacked or split. 
	 * May differ from tileWidth because some systems must use
	 * on powers of two for image dimensions
	 */
	public final int imageW;
	
	
	/**
	 * As imageW
	 */
	public final int imageH;
	
	public final int flags;	
	public final int tileType;
	
	private final Matrix[] mTILEcornersClockwise;
	
	/**
	 * UnwarpedRects associated with this tile, for reading only. 
	 */
	public final List<UnwarpedRect> unwarpedRectsLinesReadOnly;
	public final List<UnwarpedRect> unwarpedRectsWordsReadOnly;
	
	private BufferedImage  tileImage;
	private Rectangle2D  rDESKboundingBoxIfComputed;
	private GeneralPath gpDESKoutlineIfComputed;
	private Matrix[] mDESKcornersClockwiseIfCalculated;		
	private boolean visible;
	private Matrix mTILEtoDESK;
	private Matrix mDESKtoTILEifCalculated;
	private boolean destroyed;
	private LinkedList<UnwarpedRect> unwarpedRectsLines; 
	private LinkedList<UnwarpedRect> unwarpedRectsWords; 
	private double TILEtoDESKrotationClockwise, DESKcentreX, DESKcentreY, DESKwidth, DESKheight; // clockwise
	private Tile[][] splitComponentsPackedByCol;
	
	
	
	Tile(StateManager stateManager, int elId, int w, int h, int flags) {
		super(stateManager, elId);
		this.tileWidth = w;
		this.tileHeight = h;
		this.mTILEcornersClockwise = new Matrix[] {
				JOGLHelper.getMFromD(0, 0, 1),
				JOGLHelper.getMFromD(0, this.tileHeight, 1),
				JOGLHelper.getMFromD(this.tileWidth, this.tileHeight, 1),
				JOGLHelper.getMFromD(this.tileWidth, 0, 1)
			};
		this.destroyed = false;
		this.flags = flags;
		this.visible = false;
		this.unwarpedRectsLines = new LinkedList<UnwarpedRect>();
		this.unwarpedRectsWords = new LinkedList<UnwarpedRect>();
		this.unwarpedRectsLinesReadOnly = Collections.unmodifiableList(this.unwarpedRectsLines);
		this.unwarpedRectsWordsReadOnly = Collections.unmodifiableList(this.unwarpedRectsWords);
		
		this.tileType = 
			!stateManager.backTiles
			? TILETYPE_UNBACKED
			: (
				(stateManager.maxDimensionForTileImages!=0 && (w>stateManager.maxDimensionForTileImages ||h>stateManager.maxDimensionForTileImages))
				? TILETYPE_SPLIT
				: TILETYPE_BACKED
			);
		
		if(this.tileType==TILETYPE_UNBACKED) {
			
			this.imageW=0;
			this.imageH=0;
			this.tileImage = null;		
			
		} else if(this.tileType==TILETYPE_SPLIT) {	
			
			this.imageW=0;
			this.imageH=0;
			this.tileImage = null;
			int nx = (int)Math.ceil(((double)w)/((double)this.stateManager.splitDimensionForTileImages));
			int ny = (int)Math.ceil(((double)h)/((double)this.stateManager.splitDimensionForTileImages));
			this.splitComponentsPackedByCol = new Tile[nx][ny];
			for(int ix=0; ix<nx; ix++) {
				int TILEdividedTileWidth = 
					(ix==nx-1) 
					? (this.tileWidth % this.stateManager.splitDimensionForTileImages)
					: this.stateManager.splitDimensionForTileImages;
				this.splitComponentsPackedByCol[ix]=new Tile[ny];
				for(int iy=0; iy<ny; iy++){
					int TILEdividedTileHeight = 
						(iy==ny-1) 
						? (this.tileHeight % this.stateManager.splitDimensionForTileImages)
						: this.stateManager.splitDimensionForTileImages;
						this.splitComponentsPackedByCol[ix][iy]=stateManager.opCreateTile(nextDivTileId++, TILEdividedTileWidth, TILEdividedTileHeight, this.flags);
				}
			}	
			
		} else {
			assert this.tileType==TILETYPE_BACKED;
			this.imageW = stateManager.usePowersOfTwoForTileImages ? JOGLHelper.getNearestPowerOfTwoGE(this.tileWidth) : w;
			this.imageH = stateManager.usePowersOfTwoForTileImages ? JOGLHelper.getNearestPowerOfTwoGE(this.tileHeight) : h;
			this.tileImage = JOGLHelper.createNewBufferedImageForTexture(imageW,imageH,this.isFlagSet(FLAG_BI_USHORT565RGB_NOT_INTARGB));
			Graphics g = this.tileImage.getGraphics();
			g.setColor(Color.WHITE);
			g.fillRect(0,0,this.tileWidth,this.tileHeight);
			
		} 	
		
		this.setAff(0,0,100.0,100.0,0.0, true); // no optimisation for the first time round
	}
	
	
	public boolean isVisible() { return this.visible; }
	public boolean isFlagSet(int f) { return (this.flags & f)!=0; }
	public double getTILEtoDESKrotationClockwise() { return this.TILEtoDESKrotationClockwise; }
	public double getDESKcentreX() { return this.DESKcentreX; }
	public double getDESKcentreY() { return this.DESKcentreY; }
	public double getDESKwidth() { return this.DESKwidth; }
	public double getDESKheight() { return this.DESKheight; }
	public Matrix[] getmTILEcornersClockwise() { return this.mTILEcornersClockwise; }
	public Matrix[] getmDESKcornersClockwise() { 
	    if(this.mDESKcornersClockwiseIfCalculated==null) {
            this.mDESKcornersClockwiseIfCalculated = JOGLHelper.applyMatrixToSetOfMatrices(this.mTILEtoDESK, this.mTILEcornersClockwise);
        }
        return this.mDESKcornersClockwiseIfCalculated;        
    }
	public Rectangle2D getrDESKboundingBox() {
	    if(this.rDESKboundingBoxIfComputed==null) {
            this.rDESKboundingBoxIfComputed = JOGLHelper.getBoundingRectFrom2hmPoly(getmDESKcornersClockwise(),null);
        }
        return this.rDESKboundingBoxIfComputed; 
        
	}
	public Matrix getmTILEtoDESK() { return this.mTILEtoDESK; }
	public Matrix getmDESKtoTILE() { 
        if(this.mDESKtoTILEifCalculated==null) {
            this.mDESKtoTILEifCalculated = this.mTILEtoDESK.inverse();
        }
        return this.mDESKtoTILEifCalculated; 
    }
	public boolean isDestroyed() { return this.destroyed; }
    
	public GeneralPath getGpDESKoutline() { 
        if(this.gpDESKoutlineIfComputed==null) {
            this.gpDESKoutlineIfComputed = JOGLHelper.getGeneralPathFrom2hmPoly(this.getmDESKcornersClockwise());
        }
        return this.gpDESKoutlineIfComputed;
    }
	
	
	/**
	 * For a split tile, this returns its component backed Tiles, packed by column.
	 * @return
	 */
	public Tile[][] getSplitComponentsPackedByColForReadOnly() {
		if(this.tileType!=TILETYPE_SPLIT) { throw new IllegalStateException("Tile is not split."); } 
		return splitComponentsPackedByCol;
	}
	
	/**
	 * For a backed tile only.
	 * @return  The backed tile's BufferedImage that contains the tile's pixel data.
	 */
	public BufferedImage getImageForReadingOnlyNoPainting() { 
		if(this.tileType!=TILETYPE_BACKED) { throw new IllegalArgumentException("Tile is split or unbacked."); }
		return this.tileImage; 
	}	
	
	/**
	 * For a backed tile, this returns the backed tile's BufferedImage that contains the tile's pixel data.
	 * <p>
	 * For an unbacked tile, this creates a new empty BufferedImage, calls callback_repaintUnbackedTile
	 * to paint into the empty image and returns the result.
	 * <p>
	 * For a split tile this results in an IllegalStateException.
	 * @return
	 */
	public BufferedImage getImageForReadingOnlyPaintIfNecessary() {
		if(this.tileType==TILETYPE_SPLIT) {
			throw new IllegalStateException("Tile is split");
		} else if(this.tileType==TILETYPE_BACKED) {
			 return this.getImageForReadingOnlyNoPainting();
		} else {
			assert this.tileType==TILETYPE_UNBACKED;
			BufferedImage result = this.createCompatibleBufferedImage(this.tileWidth, this.tileHeight);
			Graphics2D g = result.createGraphics();
			g.setClip(0,0,result.getWidth(),result.getHeight());
			this.stateManager.stateListener.callback_repaintUnbackedTile(
				this,
				new Rectangle(0, 0, this.tileWidth,	this.tileHeight),
				result,
				g
			);
			return result;
		}
	}
	
	

	
	
	/**
	 * Sets the centre, width, height and rotation of the tile on the display surface.
	 * 
	 * @param DESKcentreX
	 * @param DESKcentreY
	 * @param DESKwidth
	 * @param DESKheight
	 * @param thetaClockwise
	 */
	public void opSetAff(double DESKcentreX, double DESKcentreY, double DESKwidth, double DESKheight, double thetaClockwise) {

		checkNotDestroyed();
		
		if(DESKwidth<=0.0 || DESKheight<=0.0) {
			throw new IllegalArgumentException("Cannot have negative or zero width or height");
		}		

		setAff(DESKcentreX, DESKcentreY, DESKwidth, DESKheight, thetaClockwise, false);
		this.stateManager.stateListener.callback_updatedTileAff(this,DESKcentreX, DESKcentreY, DESKwidth, DESKheight, thetaClockwise);
	}
	
	
	private void setAff(double DESKcentreX, double DESKcentreY, double DESKwidth, double DESKheight, double thetaClockwise, boolean forceNoOpt) {
		
		if(!forceNoOpt && this.DESKheight==DESKheight && this.DESKwidth==DESKwidth && this.TILEtoDESKrotationClockwise == thetaClockwise) {
			
			 // if just a translate then much simpler
			double dx = DESKcentreX-this.DESKcentreX;
			double dy = DESKcentreY-this.DESKcentreY;
			this.DESKcentreX = DESKcentreX;
			this.DESKcentreY = DESKcentreY;
			this.mTILEtoDESK = JOGLHelper.get2hmTranslation(dx,dy).times(this.mTILEtoDESK);
			this.mDESKtoTILEifCalculated = null;
			this.mDESKcornersClockwiseIfCalculated = null;
			AffineTransform at = new AffineTransform();
			at.translate(dx,dy);
            if(this.gpDESKoutlineIfComputed!=null) {
                this.gpDESKoutlineIfComputed.transform(at);
            } else {
                // not computed so ignore it.
            }
            if(this.rDESKboundingBoxIfComputed!=null) {
    			this.rDESKboundingBoxIfComputed.setRect(
    				this.rDESKboundingBoxIfComputed.getX()+dx,
    				this.rDESKboundingBoxIfComputed.getY()+dy,
    				this.rDESKboundingBoxIfComputed.getWidth(),
    				this.rDESKboundingBoxIfComputed.getHeight()
    			);
            }
			
			if(this.tileType==TILETYPE_SPLIT) {
				// apply transform to each split component
				for(Tile[] ts: this.splitComponentsPackedByCol) {
					for(Tile t: ts) {
						t.opSetAff(
							t.DESKcentreX+dx,
							t.DESKcentreY+dy,
							t.DESKwidth,
							t.DESKheight,
							t.TILEtoDESKrotationClockwise
						);
					}
				}
			} else {
				// no need to do anything as we have no divided tiles
			}
			
		} else {			
			
			this.DESKcentreX = DESKcentreX;
			this.DESKcentreY = DESKcentreY;
			this.DESKheight = DESKheight;
			this.DESKwidth = DESKwidth;
			this.TILEtoDESKrotationClockwise = thetaClockwise;
			
			double TILEtoDESKscaleFactorX = DESKwidth/(double)this.tileWidth;
			double TILEtoDESKscaleFactorY = DESKheight/(double)this.tileHeight;		
	
			this.mTILEtoDESK =
                JOGLHelper.get2hmScaRotTra(
                        TILEtoDESKflipsHoriz ? -TILEtoDESKscaleFactorX : TILEtoDESKscaleFactorX, 
                        TILEtoDESKflipsVert ? -TILEtoDESKscaleFactorY : TILEtoDESKscaleFactorY,
                        thetaClockwise,
                        DESKcentreX,DESKcentreY
                ).times(
                        JOGLHelper.get2hmTranslation(-this.tileWidth/2.0,-this.tileHeight/2.0)
                );
			this.mDESKtoTILEifCalculated = null;
			this.mDESKcornersClockwiseIfCalculated = null;
			this.gpDESKoutlineIfComputed = null;
			this.rDESKboundingBoxIfComputed= null;            
		
			if(this.tileType==TILETYPE_SPLIT) {
				// apply tranform to each divided tile
				for(int xt=0; xt<this.splitComponentsPackedByCol.length; xt++) {
					double THISTILEdivTileWidth = this.splitComponentsPackedByCol[xt][0].tileWidth;
					double THISTILEdivTileCentreX = (this.stateManager.splitDimensionForTileImages*xt) + THISTILEdivTileWidth/2.0;
					for(int yt=0; yt<this.splitComponentsPackedByCol[0].length; yt++) {
						double THISTILEdivTileHeight = this.splitComponentsPackedByCol[xt][yt].tileHeight;
						double THISTILEdivTileCentreY = (this.stateManager.splitDimensionForTileImages*yt) + THISTILEdivTileHeight/2.0; 
						Matrix mDESKDivTileCentre = 
							this.mTILEtoDESK.times(
								JOGLHelper.getMFromD(
									THISTILEdivTileCentreX, 
									THISTILEdivTileCentreY, 
									1.0
								)
							);
						this.splitComponentsPackedByCol[xt][yt].opSetAff(
							mDESKDivTileCentre.get(0,0)/mDESKDivTileCentre.get(2,0),
							mDESKDivTileCentre.get(1,0)/mDESKDivTileCentre.get(2,0),
							this.DESKwidth * (THISTILEdivTileWidth/this.tileWidth),
							this.DESKheight * (THISTILEdivTileHeight/this.tileHeight),
							this.TILEtoDESKrotationClockwise
						);
	 				}
				}
			} else {
				// no need to do anything
			}
		}		
	}
	
	
	
	/**
	 * Set tile visibility.
	 * @param v
	 */
	public void opSetVisibility(boolean v) {
		if( (this.visible && !v) || (!this.visible && v) ) {
			this.visible = v;
			if(this.tileType==TILETYPE_SPLIT) {
				for(Tile[] ts: this.splitComponentsPackedByCol) {
					for(Tile t: ts) {
						t.opSetVisibility(v);
					}
				}
			} else {
				// no need to do anything to unwarped rects
			}
			this.stateManager.stateListener.callback_updatedTileVisibility(this, v);
		} else {
			// do nothing
		}
		
	}
	
	/** 
	 * Creates a BufferedImage compatible with the tile's image. 
	 * @param w
	 * @param h
	 * @return
	 */
	public BufferedImage createCompatibleBufferedImage(int w, int h) {
		return JOGLHelper.createNewBufferedImageForTexture(w,h,this.isFlagSet(FLAG_BI_USHORT565RGB_NOT_INTARGB));
	}	
	
	
	/**
	 * Creates a Graphics2D from a Rectangle and a BufferedImage such that
	 * (r.x,r.y) in Graphics2D corresponds to (0,0) in the BufferedImage
	 * and the clip area of the Graphics2D is exactly the size of the BufferedImage.
	 *  
	 * @param r
	 * @param tileUpdateImage
	 * @return
	 */
	public Graphics2D createGraphics2DForTileUpdate(Rectangle r, BufferedImage tileUpdateImage) {
		Graphics2D g = tileUpdateImage.createGraphics();
		g.setClip(0,0,r.width,r.height);
		g.translate(-r.x,-r.y);
		return g;
	}	
	
	
	/**
	 * Updates the contents of this tile by copying data from another tile. 
	 * Neither tile may be split.
	 * @param sx	
	 * @param sy
	 * @param dx
	 * @param dy
	 * @param w
	 * @param h
	 * @param source
	 */
	public void opUpdateContentsByCopyingFromOtherTile(int sx, int sy, int dx, int dy, int w, int h, Tile source) {
		checkNotDestroyed();
		source.checkNotDestroyed();
		if(sx<0 || dx<0 || sy<0 || dy<0 || sx+w>source.tileWidth || dx+w>this.tileWidth || sy+h>source.tileHeight || dy+h>this.tileHeight) {
			throw new IllegalStateException("Coordinates are out of bounds in either source or destination tiles");			
		}
		if(this.tileType == TILETYPE_SPLIT) {
			throw new IllegalStateException("Cannot perform opUpdateContentsByCopyingFromOtherTile on a split tile");
		} else if(this.tileType == TILETYPE_BACKED) {
			source.checkNotSplit();
			JOGLHelper.pastePartOfImageAIntoPartOfImageBWithoutSharing(
				dx,dy,
				sx, sy, w, h, 
				source.getImageForReadingOnlyPaintIfNecessary(), 
				this.tileImage
			);
		} else {
			assert this.tileType==TILETYPE_UNBACKED;
			// no need to do anything.
		}
		this.stateManager.stateListener.callback_updatedTileContentsByCopyingFromOtherTile(this, sx, sy, dx, dy, w, h, source);
	}
	
	
	/**
	 * Pastes a supplied image into part of the tile's contents.
	 * This works for all types of tile.
	 * <p>
	 * Typical usage: 
	 * Say you want to update an area r of the tile.
	 * Create a new BufferedImage with same dimensions as r using createCompatibleBufferedImage. 
	 * Create a Graphics2D using createGraphics2DForTileUpdate.
	 * Now draw into the Graphics2D. (0,0) will correspond to the origin in TILE space, but only
	 * things that you draw into the area of r will appear in the update. Now call opUpdateContents.
	 * 
	 * 
	 * @param x 		x coordinate of update in TILE
	 * @param y 		y coordinate of update in TILE
	 * @param update 	contents of update
	 */
	public void opUpdateContents(int x, int y, BufferedImage update, int compressionHint) {
		checkNotDestroyed();
		if(this.tileType==TILETYPE_SPLIT) {
			int startIx = x/this.stateManager.splitDimensionForTileImages;
			int startIy = y/this.stateManager.splitDimensionForTileImages;
			int finishIx = (x+update.getWidth())/this.stateManager.splitDimensionForTileImages;
			int finishIy = (y+update.getHeight())/this.stateManager.splitDimensionForTileImages;
			for(int ix=startIx; ix<=finishIx; ix++) {
					int oxInUpdate = (ix==startIx) 
						? 0
						: ix*this.stateManager.splitDimensionForTileImages-x;
					int oxInDivTile = (ix==startIx)
						? x % this.stateManager.splitDimensionForTileImages
						: 0;
					// for h calcn,need to handle case where iy==finishIy==startIy
					int w = (ix==finishIx) 
						? (x+update.getWidth()) % this.stateManager.splitDimensionForTileImages - oxInDivTile
						: this.stateManager.splitDimensionForTileImages - oxInDivTile;
				for(int iy=startIy; iy<=finishIy; iy++) {
					int oyInUpdate = (iy==startIy) 
						? 0
						: iy*this.stateManager.splitDimensionForTileImages-y;
					int oyInDivTile = (iy==startIy)
						? y % this.stateManager.splitDimensionForTileImages
						: 0;
					// for h calcn,need to handle case where iy==finishIy==startIy
					int h = (iy==finishIy) 
						? (y+update.getHeight()) % this.stateManager.splitDimensionForTileImages - oyInDivTile
						: this.stateManager.splitDimensionForTileImages - oyInDivTile;
					if(w>0 && h>0) {
						// create update for divided tile - this will share the old image's data array
						BufferedImage updateForDivTile = update.getSubimage(oxInUpdate,oyInUpdate,w,h);
						this.splitComponentsPackedByCol[ix][iy].opUpdateContents(oxInDivTile,oyInDivTile,updateForDivTile, compressionHint);
					} else {
						// update is an exact number of tiles so we end up with this zero width/height scenario
						// just do nothing.
					}
				}
			}
		} else if(this.tileType==TILETYPE_BACKED) {
            if(
                    x==0 
                    && y==0 
                    && update.getWidth() == this.tileImage.getWidth() 
                    && update.getHeight() == this.tileImage.getHeight()
                    && update.getColorModel().equals(this.tileImage.getColorModel())
                    && update.getRaster().getDataBuffer().getSize()==this.tileImage.getRaster().getDataBuffer().getSize()
            ) {
                // optimise - don't copy the data!
                this.tileImage = update;
            } else {
    			JOGLHelper.pastePartOfImageAIntoPartOfImageBWithoutSharing(
    					x,y,
    					0, 0, update.getWidth(), update.getHeight(), 
    					update, tileImage
    				);
            }
		} else {
			assert this.tileType==TILETYPE_UNBACKED;
			// no need to do anything.
		}
		this.stateManager.stateListener.callback_updatedTileContents(this, x, y, update, compressionHint);
	}
	
	
	
	/**
	 * Creates a new unwarped rectangle. Not to be confused with split tiles.
	 */
	public void opCreateUnwarpedRect(UnwarpedRect r) {
		checkNotDestroyed();
		checkNotSplit();
		if(r.typeLineNotWord) {
			this.unwarpedRectsLines.add(r);
		} else {
			this.unwarpedRectsWords.add(r);			
		}
		this.stateManager.stateListener.callback_createdUnwarpedRect(this,r);
	}
	
	
	/**
	 * Destroys an unwarped rectangle.
	 * @param r
	 */
	/*public void opDestroyUnwarpedRect(Rectangle r) {
		checkNotDestroyed();
		checkNotSplit();
		if(!this.unwarpedRects.contains(r)) {
			throw new IllegalArgumentException("Unwarped rectangle doesn't exist");
		}
		this.unwarpedRects.remove(r);
		this.stateManager.stateListener.callback_destroyedUnwarpedRect(this,r);
	}*/
	
	/**
	 * Destroys all unwarped rectangles intersecting a specified rectangle.
	 * @param r
	 */
	public void opDestroyAllUnwarpedRectsIntersecting(Rectangle r) {
		// if r is null then destroy all UnwarpedRects
		checkNotDestroyed();
		checkNotSplit();
		if(r==null) {
			this.unwarpedRectsWords.clear();
			this.unwarpedRectsLines.clear();
		} else {
			ListIterator<UnwarpedRect> i = this.unwarpedRectsWords.listIterator();
			while(i.hasNext()) {
				UnwarpedRect ur = i.next();
				if(r.intersects(ur.r)) {
					i.remove();
				} else {
					// continue
				}
			}
			i = this.unwarpedRectsLines.listIterator();
			while(i.hasNext()) {
				UnwarpedRect ur = i.next();
				if(r.intersects(ur.r)) {
					i.remove();
				} else {
					// continue
				}
			}
		}
		this.stateManager.stateListener.callback_destroyedUnwarpedRectsIntersecting(this,r);
	}
	
	void destroyed() {
		if(this.tileType==TILETYPE_SPLIT) {
			for(Tile[] ts: this.splitComponentsPackedByCol) {
				for(Tile t: ts) {
					this.stateManager.opDestroyTile(t);
				}
			}
		}
		this.destroyed = true;
	}
	
	private void checkNotDestroyed() {
		if(this.destroyed) {
			throw new IllegalArgumentException("Tile is destroyed");
		}
	}
	
	private void checkNotSplit() {
		if(this.tileType==TILETYPE_SPLIT) {
			throw new IllegalArgumentException("Tile is split.");
		}
	}
	
	/**
	 * Translates DESK space coordinates into TILE space coordinates and then rounds to integers.
	 * @param intTileSpaceCoords	array of two ints, which are set to the x and y integer coordinates.
	 * @param dDESK 				array of three doubles, the DESK space coordinates.
	 * @return true iff the coordinates lie within the tile.
	 */
	public boolean getIntegerTileSpaceCoordsFromDESK(int[] intTileSpaceCoords, double[] dDESK) {
		if(this.getGpDESKoutline().contains(dDESK[0]/dDESK[2], dDESK[1]/dDESK[2])) {
			double[] dTILE = 
				JOGLHelper.getDFromM(
					this.getmDESKtoTILE().times( 
						JOGLHelper.getMFromD(dDESK) 
					)
				);				
			intTileSpaceCoords[0] = (int)Math.round(dTILE[0]/dTILE[2]);
			intTileSpaceCoords[1] = (int)Math.round(dTILE[1]/dTILE[2]);
			
			// compensate for maths errors
			if(intTileSpaceCoords[0]==-1) { intTileSpaceCoords[0]=0; }
			if(intTileSpaceCoords[1]==-1) { intTileSpaceCoords[1]=0; }
			if(intTileSpaceCoords[0]==this.tileWidth) {intTileSpaceCoords[0]=this.tileWidth-1; }
			if(intTileSpaceCoords[1]==this.tileHeight) {intTileSpaceCoords[1]=this.tileHeight-1; }
			
			assert 
				intTileSpaceCoords[0]>=0 
				&& intTileSpaceCoords[1]>=0
				&& intTileSpaceCoords[0]<this.tileWidth
				&&  intTileSpaceCoords[1]<this.tileHeight;
			return true;
		} else {
			return false;
		}
	}

	public String toString() {
		return super.toString() + " elId="+this.elementId;
	}
	
}
