/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;


import java.awt.Color;
import java.awt.geom.Area;
import java.awt.geom.Rectangle2D;
import java.util.List;
import java.util.logging.Logger;

import Jama.Matrix;

/**
 * A Link represents a link between two rectangles in DESK space.
 * 
 * Create new Links by calling the appropriate method on the a StateManager object. 
 * <p>
 * Threading notes: this class is not thread-safe. You must use some kind of locking scheme
 * if you use it in a multithreaded environment.
 * 
 * @author pjt40
 *
 */
public class Link extends OrderedElement {

	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	// Standard Rect has bottom left corner at (0,0) and width 1, height 1
	
	private boolean destroyed;
	public final Color color;
	public final int displayType;
	public static final int DISPLAYTYPE_NORMAL = 0;
	private ScaRotTraTransformImmutable tStandardRectToDESKrectA;
	private ScaRotTraTransformImmutable tStandardRectToDESKrectB;
	private boolean calculated;
	private Rectangle2D rDESKboundingBox;
	private List<List<Matrix>> mDESKpolyPoints;
	
	Link(StateManager stateManager, int elId, int displayType, Color color, ScaRotTraTransformImmutable tUniformRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		super(stateManager, elId);
		this.displayType = displayType;
		this.color = color;
		this.setAff(tUniformRectToDESKrectA, tStandardRectToDESKrectB);
	}
	
	/**
	 * Sets the positions, dimensions and orientations in DESK space of the two rectangles
	 * linked by the link. They are specified as scale, rotation and translations of the standard
	 * rectangle which has width 1, height 1 and bottom left corner at the origin.
	 * 
	 * @param tStandardRectToDESKrectA
	 * @param tStandardRectToDESKrectB
	 */
	public void opSetAff( ScaRotTraTransformImmutable tStandardRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		this.setAff(tStandardRectToDESKrectA, tStandardRectToDESKrectB);
		this.stateManager.stateListener.callback_updatedLinkAff(this,tStandardRectToDESKrectA, tStandardRectToDESKrectB);
	}
	
	private void setAff(  ScaRotTraTransformImmutable tUniformRectToDESKrectA, ScaRotTraTransformImmutable tStandardRectToDESKrectB) {
		this.tStandardRectToDESKrectA = tUniformRectToDESKrectA;
		this.tStandardRectToDESKrectB = tStandardRectToDESKrectB;
		this.calculated = false;
	}
	
	void destroyed() {
		this.destroyed = true;
	}
	
	private void checkNotDestroyed() {
		if(this.destroyed) {
			throw new IllegalArgumentException("Link is destroyed");
		}
	}

	public boolean isDestroyed() {
		return destroyed;
	}

	
	
	public ScaRotTraTransformImmutable getTstandardRectToDESKrectA() {
		return this.tStandardRectToDESKrectA;
	}
	
	public ScaRotTraTransformImmutable getTstandardRectToDESKrectB() {
		return this.tStandardRectToDESKrectB;
	}

	public Rectangle2D getrDESKboundingBox() {
		this.calculate();
		return this.rDESKboundingBox;
	}
	
	public List<List<Matrix>> getmDESKpolyPoints() {
		this.calculate();
		return this.mDESKpolyPoints;
	}
	
	
	private void calculate() {
		
		if(!this.calculated) {
			
			ScaRotTraRectangle DESKrectA = new ScaRotTraRectangle(this.tStandardRectToDESKrectA.getMatrix2dHomogReadOnly());
			ScaRotTraRectangle DESKrectB = new ScaRotTraRectangle(this.tStandardRectToDESKrectB.getMatrix2dHomogReadOnly());
			
			Matrix m2hDESKcentreRectA = DESKrectA.get2hmScaRotTraCentrePoint();
			Matrix m2hDESKcentreRectB = DESKrectB.get2hmScaRotTraCentrePoint();
			
			double dxAtoB = m2hDESKcentreRectB.get(0,0)/m2hDESKcentreRectB.get(2,0)
				-m2hDESKcentreRectA.get(0,0)/m2hDESKcentreRectA.get(2,0);
			double dyAtoB = m2hDESKcentreRectB.get(1,0)/m2hDESKcentreRectB.get(2,0)
				-m2hDESKcentreRectA.get(1,0)/m2hDESKcentreRectA.get(2,0);
			double h = 2.0;
			double w = Math.sqrt(dxAtoB*dxAtoB+dyAtoB*dyAtoB);
			double thetaClockwise = -Math.atan2(dyAtoB,dxAtoB);
			
			Matrix m2hStandardRectToDESKlinkRect = 
				JOGLHelper.get2hmTranslation(m2hDESKcentreRectA.get(0,0)/m2hDESKcentreRectA.get(2,0),m2hDESKcentreRectA.get(1,0)/m2hDESKcentreRectA.get(2,0)).times(
					JOGLHelper.get2hmRotation(thetaClockwise).times(
						JOGLHelper.get2hmScale(w,h).times(
							JOGLHelper.get2hmTranslation(0.0,-0.5)
						)
					)
				);
			
			ScaRotTraRectangle m2hDESKrectLink = new ScaRotTraRectangle(m2hStandardRectToDESKlinkRect);
			
			Area a = new Area(m2hDESKrectLink.getGpScaRotTraOutlineReadOnly());
			a.subtract(new Area(DESKrectA.getGpScaRotTraOutlineReadOnly()));
			a.subtract(new Area(DESKrectB.getGpScaRotTraOutlineReadOnly()));			
			
			this.rDESKboundingBox = a.getBounds2D();
			this.mDESKpolyPoints = JOGLHelper.get2hmPolysFromPolygonalArea(a);
			
			/*
			 * todo: arrowhead at centre of dest rect
			 */
			this.calculated = true;			
		}
	}	

	public String toString() {
		return super.toString() + " elId="+this.elementId;
	}
	
		
}
