/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.color.ColorSpace;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferInt;
import java.awt.image.DataBufferUShort;
import java.awt.image.DirectColorModel;
import java.awt.image.SampleModel;
import java.awt.image.WritableRaster;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.IntBuffer;
import java.nio.ShortBuffer;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUtessellator;
import javax.media.opengl.glu.GLUtessellatorCallback;
import javax.media.opengl.glu.GLUtessellatorCallbackAdapter;

import Jama.Matrix;
import Jama.SingularValueDecomposition;

public class JOGLHelper {
    

	public static String implode(List<?> l, String sep) {
		if (l.size() == 0) {
			return "";
		} else {
			ListIterator<?> i = l.listIterator();
			String r = i.next().toString();
			while (i.hasNext()) {
				r = r + "," + i.next();
			}
			return r;
		}
	}
	

	public static void logMatrix(Logger logger, Level level, String msg, Matrix m) {
		logger.log(
			level, 
			msg+"\n"+JOGLHelper.matrixToString(m, 3, 3)
			);
	}	

	public static void props_setList(List<?> l, Properties p, String s) {
		p.setProperty(s,JOGLHelper.implode(l,","));
	}
	
	public static List<String> props_getStringList(Properties p, String s)
			throws PropertiesException {
		String v = p.getProperty(s);
		if (v == null) {
			throw new PropertiesException("Property '" + s + "' not found");
		}
		return explode(v,", ");
	}
    
    public static List<String> explode(String v, String sep) {
        StringTokenizer st = new StringTokenizer(v, sep);
        List<String> l = new LinkedList<String>();
        while (st.hasMoreElements()) {
            l.add(st.nextToken());
        }
        return l;
   }
	
	
	public static List<Integer> props_getIntList(Properties p, String k) throws PropertiesException {
		List<String> ss = props_getStringList(p, k);
		List<Integer> is = new LinkedList<Integer>();
		for(String s: ss) {
			try {
				is.add(Integer.parseInt(s));
			} catch (NumberFormatException e) {
				throw new PropertiesException("Bad integer '" + s
						+ "' for property '" + k + "'");
			}
			
		}
		return is;
		}

	public static String props_getString(Properties p, String s)
			throws PropertiesException {
		String v = p.getProperty(s);
		if (v == null) {
			throw new PropertiesException("Property '" + s + "' not found");
		}
		return v;
	}

	public static int props_getInt(Properties p, String s) throws PropertiesException {
		String v = p.getProperty(s);
		if (v == null) {
			throw new PropertiesException("Property '" + s + "' not found");
		}
		try {
			return Integer.parseInt(v);
		} catch (NumberFormatException e) {
			throw new PropertiesException("Bad integer '" + v
					+ "' for property '" + s + "'");
		}
	}

	public static double props_getDouble(Properties p, String s)
			throws PropertiesException {
		String v = p.getProperty(s);
		if (v == null) {
			throw new PropertiesException("Property '" + s + "' not found");
		}
		try {
			return Double.parseDouble(v);
		} catch (NumberFormatException e) {
			throw new PropertiesException("Bad double '" + v
					+ "' for property '" + s + "'");
		}
	}

	public static boolean props_getBoolean(Properties p, String s)
			throws PropertiesException {
		String v = p.getProperty(s);
		if (v == null) {
			throw new PropertiesException("Property '" + s + "' not found");
		}
		try {
			return Boolean.parseBoolean(v);
		} catch (NumberFormatException e) {
			throw new PropertiesException("Bad boolean '" + v
					+ "' for property '" + s + "'");
		}
	}

	public static class PropertiesException extends Exception {
		public PropertiesException(String s) {
			super(s);
		}

		public PropertiesException(Throwable e) {
			super(e);
		}
	}
	

	/** **************** STATS STUFF *********************** */
	
	public static void sortAscending(List<Double> l) {
		Collections.sort(l);
	}
	
	public static double mean(List<Double> l) {
		int n=0;
		double t=0.0;
		for(double a: l) { n++; t+=a; }
		return t/n;
	}
	
	public static double sd(List<Double> l) {
		double mean = mean(l);
		int n=0;
		double t=0.0;
		for(double a: l) { n++; t+=a*a; }
		return Math.sqrt(t/n-mean*mean);
	}

	/** **************** IMAGE STUFF *********************** */
	
	
	private static int[] texresarr = new int[1];
	
	public static boolean oglCurrentBoundTextureIsResident(GL gl) {
		gl.glGetTexParameteriv(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_RESIDENT, texresarr, 0);
		return texresarr[0]==gl.GL_TRUE;
	}
	
	
	public static int oglCreateTexture(javax.media.opengl.GL gl, int w, int h, BufferedImage b
			) {
		/* 
		 * if BufferedImage is type TYPE_INT_ARGB: We then set
		 * gl.GL_UNPACK_SWAP_BYTES to true and send as gl.GL_BGRA_EXT format
		 * with type gl.GL_UNSIGNED_BYTE.
		 * 
		 * if BufferedImage is type TYPE_USHORT_565_RGB: We then set
		 * gl.GL_UNPACK_SWAP_BYTES to false and send as gl.GL_RGB with type
		 * gl.GL_UNSIGNED_SHORT_5_6_5
		 */
		
		assert b.getWidth() == w && b.getHeight() == h;
		assert b.getType() == BufferedImage.TYPE_INT_ARGB
				|| b.getType() == BufferedImage.TYPE_USHORT_565_RGB;

		int textureInternalFormat = gl.GL_RGBA;
		

		// get texture name
		int[] textureName = new int[1];
		gl.glGenTextures(1, textureName, 0);

		// bind and set options
		gl.glBindTexture(gl.GL_TEXTURE_2D, textureName[0]);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_S, gl.GL_CLAMP);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_WRAP_T, gl.GL_CLAMP);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER,
				gl.GL_LINEAR);
		gl.glTexParameteri(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER,
				gl.GL_LINEAR);

		if (b.getType() == BufferedImage.TYPE_INT_ARGB) {
			int[] ibuf = bufferedImageToIntArrayForTextureWithoutCopyIfPossible(b);
			gl.glPixelStorei(gl.GL_UNPACK_ALIGNMENT, 1);
			gl.glPixelStorei(gl.GL_UNPACK_SWAP_BYTES, gl.GL_FALSE);
			gl.glTexImage2D(gl.GL_TEXTURE_2D, 0, // target and level
					textureInternalFormat, // internal format of texture
					w, h, 0, // border
					gl.GL_BGRA, // format of data being transferred
					gl.GL_UNSIGNED_BYTE, // type of data being transferred
					IntBuffer.wrap(ibuf));
		} else {
			short[] sbuf = bufferedImageToShortArrayForTextureWithoutCopyIfPossible(b);
			gl.glPixelStorei(gl.GL_UNPACK_ALIGNMENT, 1);
			gl.glPixelStorei(gl.GL_UNPACK_SWAP_BYTES, gl.GL_FALSE);
			gl.glTexImage2D(gl.GL_TEXTURE_2D, 0, // target and level
					textureInternalFormat, // internal format of texture
					w, h, 0, // border
					gl.GL_RGB, // format of data being transferred
					gl.GL_UNSIGNED_SHORT_5_6_5, // type of data being transferred
					ShortBuffer.wrap(sbuf));
		}
		return textureName[0];
	}

	public static void oglDestroyTexture(GL gl, int textureName) {
		gl.glDeleteTextures(1, new int[] { textureName }, 0);
	}

	public static void oglUpdateTextureContents(GL gl, int tx, int ty, int iw,
			int ih, BufferedImage b) {
		// assumes you've already set the texture target
		// image must be the correct size

		assert b.getWidth() == iw && b.getHeight() == ih;
		assert b.getType() == BufferedImage.TYPE_INT_ARGB
				|| b.getType() == BufferedImage.TYPE_USHORT_565_RGB;

		if (b.getType() == BufferedImage.TYPE_INT_ARGB) {
			int[] ibuf = bufferedImageToIntArrayForTextureWithoutCopyIfPossible(b);
			gl.glPixelStorei(gl.GL_UNPACK_ALIGNMENT, 1);
			gl.glPixelStorei(gl.GL_UNPACK_SWAP_BYTES, gl.GL_TRUE);
			gl.glTexSubImage2D(gl.GL_TEXTURE_2D, 0, tx, ty, iw, ih,
					gl.GL_BGRA, // format of data being transferred
					gl.GL_UNSIGNED_BYTE, // type of data being transferred
					IntBuffer.wrap(ibuf)
				);
		} else {
			short[] sbuf = bufferedImageToShortArrayForTextureWithoutCopyIfPossible(b);
			gl.glPixelStorei(gl.GL_UNPACK_ALIGNMENT, 1);
			gl.glPixelStorei(gl.GL_UNPACK_SWAP_BYTES, gl.GL_FALSE);
			gl.glTexSubImage2D(gl.GL_TEXTURE_2D, 0, tx, ty, iw, ih, 
					gl.GL_RGB, // format of data being transferred
					gl.GL_UNSIGNED_SHORT_5_6_5, // type of data being  transferred
					ShortBuffer.wrap(sbuf)
				);
		}
	}

	public static BufferedImage createNewBufferedImageForTexture(int w, int h,
			boolean useUSHORT565RGBnotINTARGB) {
		/*
		 * if smallbi then we use bufferedimage TYPE_USHORT_565_RGB otherwise we
		 * use bufferedimage TYPE_INT_ARGB
		 */
		BufferedImage i = new BufferedImage(
				createNewColorModelForTexture(useUSHORT565RGBnotINTARGB),
				createNewRasterForTexture(w, h, useUSHORT565RGBnotINTARGB),
				false, null);
		assert (!useUSHORT565RGBnotINTARGB && i.getType() == BufferedImage.TYPE_INT_ARGB)
				|| (useUSHORT565RGBnotINTARGB && i.getType() == BufferedImage.TYPE_USHORT_565_RGB);
		return i;
	}

	public static WritableRaster createNewRasterForTexture(int w, int h,
			boolean useUSHORT565RGBnotINTARGB) {
		return java.awt.image.Raster
				.createWritableRaster(createNewSampleModelForTexture(w, h,
						useUSHORT565RGBnotINTARGB), null);
	}

	public static SampleModel createNewSampleModelForTexture(int w, int h,
			boolean useUSHORT565RGBnotINTARGB) {
		return createNewColorModelForTexture(useUSHORT565RGBnotINTARGB)
				.createCompatibleSampleModel(w, h);
	}

	public static ColorModel createNewColorModelForTexture(
			boolean useUSHORT565RGBnotINTARGB) {
		return useUSHORT565RGBnotINTARGB ? new DirectColorModel(ColorSpace
				.getInstance(ColorSpace.CS_sRGB), 16, 0xf800, 0x07E0, 0x001F,
				0, false, DataBuffer.TYPE_USHORT) : new DirectColorModel(
				ColorSpace.getInstance(ColorSpace.CS_sRGB), 32, 0x00ff0000,// Red
				0x0000ff00,// Green
				0x000000ff,// Blue
				0xff000000,// Alpha
				false, // Alpha not Premultiplied
				DataBuffer.TYPE_INT);
	}

	public static WritableRaster intArrayToWritableRasterForTextureWithoutCopy(
			int[] buff, int w, int h) {
		// creates a buffered image of TYPE_INT_ARGB
		assert buff.length == w * h;
		return java.awt.image.Raster.createWritableRaster(
				createNewSampleModelForTexture(w, h, false), new DataBufferInt(
						buff, buff.length), null);
	}

	public static WritableRaster shortArrayToWritableRasterForTextureWithoutCopy(
			short[] buff, int w, int h) {
		// creates a buffered image of TYPE_USHORT_565
		assert buff.length == w * h;
		return java.awt.image.Raster.createWritableRaster(
				createNewSampleModelForTexture(w, h, true),
				new DataBufferUShort(buff, buff.length), null);
	}

	public static int[] bufferedImageToIntArrayForTextureWithoutCopyIfPossible(
			BufferedImage b) {
		// only works for type TYPE_INT_ARGB
		assert b.getType() == BufferedImage.TYPE_INT_ARGB;
		BufferedImage bb = getBIthatUsesAllOfBuffer(b);
		// now return the data array - not a copy of it
		return ((DataBufferInt) (bb.getRaster().getDataBuffer())).getData();
	}

	public static short[] bufferedImageToShortArrayForTextureWithoutCopyIfPossible(
			BufferedImage b) {
		// only works for type TYPE_USHORT_565_RGB
		assert b.getType() == BufferedImage.TYPE_USHORT_565_RGB;
		BufferedImage bb = getBIthatUsesAllOfBuffer(b);
		// now return the data array - not a copy of it
		return ((DataBufferUShort) (bb.getRaster().getDataBuffer())).getData();
	}

	public static BufferedImage getBIthatUsesAllOfBuffer(BufferedImage b) {
		/*
		 * If b uses all of its buffer then just return b.
		 * 
		 * If not then b uses only some sub portion of its buffer. We don't want
		 * to get the entire buffer so we create a new bufferedimage with a copy
		 * of the relevent part of b's buffer and no sharing.
		 */
		if (b.getSampleModel().getWidth() == b.getWidth()
				&& b.getSampleModel().getHeight() == b.getHeight()) {
			return b;
		} else {
			BufferedImage bb = JOGLHelper
					.createNewImageFromPartOfImageAWithoutSharing(0, 0, b
							.getWidth(), b.getHeight(), b);
			assert bb.getSampleModel().getWidth() == bb.getWidth()
					&& bb.getSampleModel().getHeight() == bb.getHeight();
			return bb;
		}
	}

	public static BufferedImage createNewImageFromPartOfImageAWithoutSharing(
			int x, int y, int w, int h, BufferedImage a) {
		BufferedImage n = new BufferedImage(a.getColorModel(), a.getRaster()
				.createCompatibleWritableRaster(w, h),
				a.isAlphaPremultiplied(), null);
		pastePartOfImageAIntoPartOfImageBWithoutSharing(0, 0, x, y, w, h, a, n);
		return n;
	}

	public static BufferedImage createNewImageFromPartOfImageAWithSharing(
			int ax, int ay, int w, int h, int offsetXInNewImage,
			int offsetYInNewImage, BufferedImage a) {
		// (offsetXInNewImage, offsetYInNewImage) in new image correspondes to
		// (ax,ay) in image a
		// and new image is w wide and h high

		assert false;
		return null;
		// DO NOT USE THIS METHOD.
		// IF YOU WANT A BUFFERED IMAGE THAT SHARES A DATA BUFFER WITH ANOTHER
		// BUFFERED IMAGE THEN JUST USE THE SAME BUFFERED IMAGE

		/*
		 * OLD
		 * 
		 * assert a.getRaster() instanceof WritableRaster;
		 * 
		 * WritableRaster subRaster = (WritableRaster)
		 * a.getRaster().createChild( ax, // left in parent's coord space ay, //
		 * top in parent's coord space w, //width h, //height ax, //left in
		 * child's coord space, corresponds to same pixel as left in parent's
		 * coord space ay, // top in child's coord space null );
		 * 
		 * return new BufferedImage( a.getColorModel(), subRaster,
		 * a.isAlphaPremultiplied(), null );
		 */
	}

	public static void pastePartOfImageAIntoPartOfImageBWithoutSharing(int bx,
			int by, int ax, int ay, int w, int h, BufferedImage a,
			BufferedImage b) {
		assert a.getColorModel().equals(b.getColorModel());
		assert b.getRaster() instanceof WritableRaster;

		// create a raster from b to copy into
		WritableRaster bSubRaster = (WritableRaster) b.getRaster().createChild(
				bx, // left in parent's coord space
				by, // top in parent's coord space
				w, // width
				h, // height
				ax, // left in child's coord space, corresponds to same pixel as
					// left in parent's coord space
				ay, // top in child's coord space
				null);

		// now copy into it row by row, no sharing
		a.copyData(bSubRaster);
		return;

		/*
		 * THE OLD WAY: // createChild shares data Raster subRaster =
		 * a.getRaster().createChild( ax, // left in parent's coord space ay, //
		 * top in parent's coord space w, //width h, //height bx, //left in
		 * child's coord space, corresponds to same pixel as left in parent's
		 * coord space by, // top in child's coord space null ); // setData
		 * copies pixel for pixel but doesn't share b.setData(subRaster);
		 */
	}

	static public BufferedImage convertBufferedImageToNewTypeIfNecessary(
			BufferedImage o, int newType) {
		/*
		 * can be v time consuming! 700x700 in 188ms
		 */
		if (o.getType() != newType) {
			// long t= System.currentTimeMillis();
			BufferedImage n = new BufferedImage(o.getWidth(), o.getHeight(),
					newType);
			Graphics2D g = n.createGraphics();
			g.drawImage(o, 0, 0, o.getWidth(), o.getHeight(), null);
			g.dispose();
			// t = System.currentTimeMillis()-t;
			// System.out.println("Converted "+o.getWidth()+"x"+o.getHeight()+"
			// in "+t);
			assert n.getType() == newType;
			return n;
		} else {
			return o;
		}
	}
	
	public static Color getRandomColor(int alpha) {
		return new Color((int)(255.0*Math.random()),(int)(255.0*Math.random()),(int)(255.0*Math.random()), alpha);
	}
    

	public static void drawShapesInRandomColors(Graphics2D g, List<Shape> ss) {
        g.setComposite(AlphaComposite.SrcOver);
		for(Shape s: ss) {
			g.setColor(getRandomColor(128));
			g.draw(s);
		}
	}

	    
    public static void fillShapeInRandomColor(Graphics2D g, Shape s) {
        g.setComposite(AlphaComposite.SrcOver);
        g.setColor(getRandomColor(128));
        g.fill(s);        
    }
    
    public static void fillAreaPolysInRandomColors(Graphics2D g, Area a) {
        List<List<Matrix>> polys = get2hmPolysFromPolygonalArea(a);
        //System.out.println(polys.size());
        for(List<Matrix> poly: polys) {
            Area b = new Area(JOGLHelper.getGeneralPathFrom2hmPoly(poly));
            fillShapeInRandomColor(g, b);
            //System.out.println(b.getBounds2D());
        }
    }
    
    public static Area cleanUpPolygonalArea(Area messyArea) {
        // remove any zero area polygons from a, which can arise from 
        // crapness of computational geometry
        List<List<Matrix>> polys = get2hmPolysFromPolygonalArea(messyArea);
        Area result = new Area();
        for(List<Matrix> poly: polys) {
            Area a = new Area(JOGLHelper.getGeneralPathFrom2hmPoly(poly));
            Rectangle2D r = a.getBounds2D();
            if(r.getWidth()!=0.0 && r.getHeight()!=0.0) {
                result.add(a);
            }
        }
        return result;
    }
    
    public static void removeLimbsWidthLessThanD(List<Matrix> poly, double d) {
        int aaInd = 0;
        boolean otherWay = false;
        while(aaInd<poly.size() && poly.size()>=5) {
            int delta = otherWay ? -1 : +1;
            int aInd = (aaInd+poly.size()+delta) % poly.size();
            int bInd = (aInd+poly.size()+delta) % poly.size();
            int cInd = (bInd+poly.size()+delta) % poly.size();
            int dInd = (cInd+poly.size()+delta) % poly.size();
            double distfromAToCD = getDistOfPointFromLine(poly.get(aInd), poly.get(cInd), poly.get(dInd));
            double distfromBToCD = getDistOfPointFromLine(poly.get(bInd), poly.get(cInd), poly.get(dInd));
            if(distfromAToCD<d && distfromBToCD<d) {
                // should remove a, b and c and replace with point of intersection of aa->a with c->d
                // actually remove b and c.
                
                Matrix newPoint = getIntersectionOfTwoLines(poly.get(aaInd),poly.get(aInd),poly.get(cInd),poly.get(dInd));
                assert newPoint !=null;
                
                //remove aInd, bInd and cInd
                int minInd = Math.min(aInd, Math.min(bInd, cInd));
                int maxInd = Math.max(aInd, Math.max(bInd, cInd));
                int midInd = (minInd==aInd) ? (maxInd==bInd ? cInd : bInd) 
                        : (minInd==bInd ? (maxInd==aInd ? cInd : aInd) :  (maxInd==bInd ? aInd : bInd));
                
                poly.remove(maxInd);
                poly.remove(midInd);
                poly.remove(minInd);
                poly.add(minInd, newPoint);
                 
                // changed so start again
                aaInd=0;
                otherWay = false;
            } else {
                // next set of points
                if(aaInd==poly.size()-1 && !otherWay) {
                    aaInd = 0;
                    otherWay = true;
                } else {
                    aaInd++;
                }
            }
        }
        return;
    }
    
    public static void printOutPolygons( List<List<Matrix>> polys) {
        System.out.println("Polys");
        for(List<Matrix> poly: polys) {
            printOutPolygon(poly);
        }
    }
    
    public static void printOutPolygon( List<Matrix> poly) {
        System.out.println("  Poly");
        for(Matrix p: poly) {
            printOutPoint(p);
        }
    }  
    
    public static void printOutPoint( Matrix m2hPoint) {
        double x1 = m2hPoint.get(0,0)/m2hPoint.get(2,0);
        double y1 = m2hPoint.get(1,0)/m2hPoint.get(2,0);   
        System.out.println("    Point: "+x1+", "+y1);
    }      
    
	/** **************** GEOM STUFF *********************** */
	
    public static double getNormalisedAngleRads(double o) {
        while(o<0) { o+=Math.PI*2.0; }
        while(o>=2.0*Math.PI) { o-=Math.PI*2.0; }
        assert o>=0 && o<2.0*Math.PI;
        return o;
    }
    

	public static double moduloAndPositive(double n, double mod) {
		n = n % mod;
		if(n<0) {
			n+=mod;
		}
		return n;
	}
	
	
	public static void drawRectangleCorrectSize(Graphics2D g, Rectangle r) {
		// java usually draws rectangles so that right hand side is at (x+width).
		// this procedure draws them so that right hand side is at (x+width-1)
		// thus the number of pixels across including both lines is equal to the width.
		// when java fills rectangles it fills the correct width and height.
		g.draw(new Rectangle(r.x,r.y,r.width-1,r.height-1));
	}
	
    public static Matrix getIntersectionOfTwoLines(Matrix l1a, Matrix l1b, Matrix l2a, Matrix l2b) {
        double l1ax = l1a.get(0,0)/l1a.get(2,0);
        double l1ay = l1a.get(1,0)/l1a.get(2,0);   
        double l1bx = l1b.get(0,0)/l1b.get(2,0);
        double l1by = l1b.get(1,0)/l1b.get(2,0);   
        double l2ax = l2a.get(0,0)/l2a.get(2,0);
        double l2ay = l2a.get(1,0)/l2a.get(2,0);   
        double l2bx = l2b.get(0,0)/l2b.get(2,0);
        double l2by = l2b.get(1,0)/l2b.get(2,0);   
        Point2D p = getIntersectionOfTwoLines(l1ax, l1ay, l1bx, l1by,l2ax, l2ay, l2bx, l2by);
        if(p==null) {
            return null;
        } else {
            return JOGLHelper.getMFromD(p.getX(),p.getY(),1.0);
        }
    }
    
    
    /**
     * Computes the intersection between two lines. The calculated point is approximate, 
     * since integers are used. If you need a more precise result, use doubles
     * everywhere. 
     * (c) 2007 Alexander Hristov. Use Freely (LGPL license). http://www.ahristov.com
     *
     * @param x1 Point 1 of Line 1
     * @param y1 Point 1 of Line 1
     * @param x2 Point 2 of Line 1
     * @param y2 Point 2 of Line 1
     * @param x3 Point 1 of Line 2
     * @param y3 Point 1 of Line 2
     * @param x4 Point 2 of Line 2
     * @param y4 Point 2 of Line 2
     * @return Point where the segments intersect, or null if they don't
     */
    public static Point2D getIntersectionOfTwoLines(
      double x1,double y1,double x2,double y2, 
      double x3, double y3, double x4,double y4
    ) {
      double d = (x1-x2)*(y3-y4) - (y1-y2)*(x3-x4);
      if (d == 0) return null;
      
      double xi = ((x3-x4)*(x1*y2-y1*x2)-(x1-x2)*(x3*y4-y3*x4))/d;
      double yi = ((y3-y4)*(x1*y2-y1*x2)-(y1-y2)*(x3*y4-y3*x4))/d;
      
      return new Point2D.Double(xi,yi);
    }

    
	
	public static Area getCopyOfArea(Area a) {
		// I don't think the normal new Area(a) copies everything - some stuff is still shared.
		// using this certainly has an effect.
		return a.createTransformedArea(AffineTransform.getTranslateInstance(0.0,0.0));
	}
	
	public static boolean rectanglesAreEqualToPrecision(Rectangle2D i, Rectangle2D j, double precision) {
		return equalToPrecision(i.getMinX(),j.getMinX(), precision)
			&& equalToPrecision(i.getMinY(),j.getMinY(), precision)
			&& equalToPrecision(i.getMaxX(),j.getMaxX(), precision)
			&& equalToPrecision(i.getMaxY(),j.getMaxY(), precision);		
	}
	
	public static boolean equalToPrecision(double a, double b, double precision) {
		return (a>b-precision && a<b+precision);
	}
	
	public static boolean areasIntersect(Area i, Area j) {
		if(i.getBounds2D().intersects(j.getBounds2D())) {
			Area ijInt = JOGLHelper.getCopyOfArea(i); //new Area(i);
			ijInt.intersect(j);
			if(ijInt.isEmpty()) {
				return false;
			} else {
				//System.out.println("====================");
				//System.out.println(i.getBounds2D());
				//System.out.println(j.getBounds2D());
				//System.out.println(ijInt.getBounds2D());
				return true;
			}
		} else {
			return false;
		}		
	/*
		if(i.getBounds2D().intersects(j.getBounds2D())) {
			Area ijInt = JOGLHelper.getCopyOfArea(i);
			ijInt.intersect(j);
			if(ijInt.isEmpty()) {
				return false;
			} else {
				return true;
			}
		} else {
			return false;
		}*/
	}

	public static Rectangle2D createUnionButIgnoreZeroAreaRects(Rectangle2D a, Rectangle2D b) {
		if(a.getWidth()==0.0 || a.getHeight()==0.0) {
			return (Rectangle2D) (b.clone());
		} else if(b.getWidth()==0.0 || b.getHeight()==0.0) {
			return (Rectangle2D) (a.clone());
		} else {
			return a.createUnion(b);
		}
	}
	
	
	public static int getNearestPowerOfTwoGE(int i) {
		int j = 1;
		while (i > j) {
			j = j * 2;
		}
		return j;
	}

	public static Matrix getProjectionMatrixFromFrustum(double left,
			double right, double bottom, double top, double near, double far) {
		// from msdn
		// does the projection but we keep the z coordinates
		double a = (right + left) / (right - left);
		double b = (top + bottom) / (top - bottom);
		double c = -(far + near) / (far - near);
		double d = -(2.0 * far * near) / (far - near);
		double e = 2.0 * near / (right - left);
		double f = 2.0 * near / (top - bottom);
		return new Matrix(new double[][] { { e, 0, a, 0 }, { 0, f, b, 0 },
				{ 0, 0, c, d }, { 0, 0, -1, 0 } });
	}

	public static Matrix getViewportMatrix(double w, double h, double cx,
			double cy, double f, double n) {
		// cx and cy are centre x and y
		// w and h are width and height
		// n and f are set using depthrange and are normally 0 and 1
		Matrix scale = new Matrix(new double[][] { { w / 2, 0, 0, 0 },
				{ 0, h / 2, 0, 0 }, { 0, 0, (f - n) / 2, 0 }, { 0, 0, 0, 1 } });
		Matrix translate = get3hmTranslation(cx, cy, (n + f) / 2);
		return translate.times(scale);
	}

	public static Matrix getm2hTo3hSettingZToConst(double z) {
		return new Matrix(new double[][] { new double[] { 1, 0, 0 },
				new double[] { 0, 1, 0 }, new double[] { 0, 0, z },
				new double[] { 0, 0, 1 } });
	}

	public static Matrix getm3hTo2hDiscardZDoNotProject() {
		return new Matrix(new double[][] { new double[] { 1, 0, 0, 0 },
				new double[] { 0, 1, 0, 0 }, new double[] { 0, 0, 0, 1 } });
	}

	public static Matrix[] rotateArray(Matrix[] a, int n) {
		Matrix[] temp = (Matrix[]) new Matrix[a.length];
		for (int i = 0; i < a.length; i++) {
			int j = (i + n) % a.length;
			if (j < 0) {
				j += a.length;
			}
			temp[j] = a[i];
		}
		return temp;
	}

	public static Matrix get2hmScale(double sx, double sy) {
		return new Matrix(new double[][] { { sx, 0, 0 }, { 0, sy, 0 },
				{ 0, 0, 1 } });
	}

	public static Matrix get3hmScale(double sx, double sy, double sz) {
		return new Matrix(new double[][] { { sx, 0, 0, 0 }, { 0, sy, 0, 0 },
				{ 0, 0, sz, 0 }, { 0, 0, 0, 1 } });
	}

	public static Matrix get2hmIdentity() {
		return new Matrix(new double[][] { { 1, 0, 0 }, { 0, 1, 0 },
				{ 0, 0, 1 } });
	}

	public static Matrix get3hmIdentity() {
		return new Matrix(new double[][] { { 1, 0, 0, 0 }, { 0, 1, 0, 0 },
				{ 0, 0, 1, 0 }, { 0, 0, 0, 1 } });
	}

	public static Matrix get2hmTranslation(double dx, double dy) {
		return new Matrix(new double[][] { { 1, 0, dx }, { 0, 1, dy },
				{ 0, 0, 1 } });
	}
    
    public static Matrix get2hmScaRotTra(double sx, double sy, double thetaClockwise, double dx, double dy) {
        double cosTheta = Math.cos(thetaClockwise);
        double sinTheta = Math.sin(thetaClockwise);
        return new Matrix(new double[][] { { sx*cosTheta, sy*sinTheta, dx }, { -sx*sinTheta, sy*cosTheta, dy },
                { 0, 0, 1 } });
    }

	public static Matrix get3hmTranslation(double dx, double dy, double dz) {
		return new Matrix(new double[][] { { 1, 0, 0, dx }, { 0, 1, 0, dy },
				{ 0, 0, 1, dz }, { 0, 0, 0, 1 } });
	}

	public static Matrix get2hmRotation(double thetaClockwise) {
		// theta is clockwise
		double cosTheta = Math.cos(thetaClockwise);
		double sinTheta = Math.sin(thetaClockwise);
		return new Matrix(new double[][] { { cosTheta, sinTheta, 0 },
				{ -sinTheta, cosTheta, 0 }, { 0, 0, 1 } });
	}
	
	public static double getDistanceBetween2hmPoints(Matrix pt1, Matrix pt2) {
		double x1 = pt1.get(0,0)/pt1.get(2,0);
		double y1 = pt1.get(1,0)/pt1.get(2,0);
		double x2 = pt2.get(0,0)/pt2.get(2,0);
		double y2 = pt2.get(1,0)/pt2.get(2,0);
		double dx = x2-x1;
		double dy = y2-y1;
		return Math.sqrt(dx*dx+dy*dy);
	}
	
	public static double getPolarAngleOfLineBetween2hmPoints(Matrix pt1, Matrix pt2) {
		double x1 = pt1.get(0,0)/pt1.get(2,0);
		double y1 = pt1.get(1,0)/pt1.get(2,0);
		double x2 = pt2.get(0,0)/pt2.get(2,0);
		double y2 = pt2.get(1,0)/pt2.get(2,0);
		double dx = x2-x1;
		double dy = y2-y1;
		return Math.atan2(dy,dx);
	}

	public static Matrix[] applyMatrixToSetOfMatrices(Matrix m, Matrix[] ms) {
		// this one has to be fast.
		Matrix[] result = new Matrix[ms.length];
		for (int i = 0; i < ms.length; i++) {
			result[i] = m.times(ms[i]);
		}
		return result;
	}
	
	public static List<Matrix> applyMatrixToSetOfMatrices(Matrix m, List<Matrix> ms) {
		// this one is not fast.
		List<Matrix> result = new LinkedList<Matrix>();
		for(Matrix mm: ms) {
			result.add(m.times(mm));
		}
		return result;
	}

	public static double[] unhomogeniseD(double[] v) {
		double[] result = new double[v.length - 1];
		for (int i = 0; i < v.length - 1; i++) {
			result[i] = v[i] / v[v.length - 1];
		}
		return result;
	}

	public static Matrix homogeniseM(Matrix m) {
		return getMFromD(homogeniseD(getDFromM(m)));
	}
	
	public static void unHomogeniseMInPlaceLeavingThreeEls(Matrix m) {
		m.set(0,0,m.get(0,0)/m.get(2,0));
		m.set(1,0,m.get(1,0)/m.get(2,0));
		m.set(2,0,1.0);
	}

	public static Matrix unhomogeniseM(Matrix m) {
		return getMFromD(unhomogeniseD(getDFromM(m)));
	}

	public static double[] homogeniseD(double[] v) {
		double[] result = new double[v.length + 1];
		for (int i = 0; i < v.length; i++) {
			result[i] = v[i];
		}
		result[result.length] = 1.0;
		return result;
	}
	
	

	public static double[] getDFromM(Matrix m) {
		assert (m.getColumnDimension() == 1);
		return m.getColumnPackedCopy();
	}

	public static Matrix getMFromD(double[] d) {
		return new Matrix(d, d.length);
	}

	public static Matrix getMFromD(double d1, double d2, double d3) {
		return new Matrix(new double[] { d1, d2, d3 }, 3);
	}

	public static Matrix getMFromD(double d1, double d2, double d3, double d4) {
		return new Matrix(new double[] { d1, d2, d3, d4 }, 4);
	}
    
    
    /**
     * @param poly
     * @param transform can be null if desired
     * @return
     */
    public static Rectangle2D getBoundingRectFrom2hmPoly(List<Matrix> poly, Matrix transform) {
        double minx=0.0,miny=0.0,maxx=0.0,maxy=0.0;
        boolean first = true;
        for(Matrix p: poly) {
            Matrix q = (transform==null) ? p : transform.times(p);
            double h = q.get(2,0);
            double x = (h==1.0) ? q.get(0,0) : q.get(0,0)/h;
            double y = (h==1.0) ? q.get(1,0) : q.get(1,0)/h;
            if(first) {
                minx = x;
                miny = y;
                maxx = x;
                maxy = y;
                first = false;
            } else {
                minx = Math.min(minx, x);
                miny = Math.min(miny, y);
                maxx = Math.max(maxx, x);
                maxy = Math.max(maxy, y);
            }
        }
        return new Rectangle2D.Double(minx,miny,maxx-minx, maxy-miny);
    }
    
    
    /**
     * @param poly
     * @param transform can be null if desired
     * @return
     */
    public static Rectangle2D.Double getBoundingRectFrom2hmPoly(Matrix[] poly, Matrix transform) {
        double minx=0.0,miny=0.0,maxx=0.0,maxy=0.0;
        boolean first = true;
        for(Matrix p: poly) {
            Matrix q = (transform==null) ? p : transform.times(p);
            double h = q.get(2,0);
            double x = (h==1.0) ? q.get(0,0) : q.get(0,0)/h;
            double y = (h==1.0) ? q.get(1,0) : q.get(1,0)/h;
            if(first) {
                minx = x;
                miny = y;
                maxx = x;
                maxy = y;
                first = false;
            } else {
                minx = Math.min(minx, x);
                miny = Math.min(miny, y);
                maxx = Math.max(maxx, x);
                maxy = Math.max(maxy, y);
            }
        }
        return new Rectangle2D.Double(minx,miny,maxx-minx, maxy-miny);
    }    
    
    public static Matrix[] getMatrixArrayFromMatrixList(List<Matrix> poly) {
        Matrix[] ms = new Matrix[poly.size()];
        ms = poly.toArray(ms);
        return ms;
    }
		
	public static GeneralPath getGeneralPathFrom2hmPoly(List<Matrix> poly) {	
		GeneralPath p = new GeneralPath();
		boolean first=true;
		for (Matrix point2hm : poly) {
			double[] point2ud = unhomogeniseD(getDFromM(point2hm));
			if (first) {
				p.moveTo((float) point2ud[0], (float) point2ud[1]);
				first=false;
			} else {
				p.lineTo((float) point2ud[0], (float) point2ud[1]);
			}
		}
		p.closePath();
		return p;
	}

	public static GeneralPath getGeneralPathFrom2hmPoly(Matrix[] poly) {
		// should be fast
		GeneralPath p = new GeneralPath();
		boolean first=true;
		for (Matrix point2hm : poly) {
			double[] point2ud = unhomogeniseD(getDFromM(point2hm));
			if (first) {
				p.moveTo((float) point2ud[0], (float) point2ud[1]);
				first=false;
			} else {
				p.lineTo((float) point2ud[0], (float) point2ud[1]);
			}
		}
		p.closePath();
		return p;
	}
    
    public static double getDistOfPointFromPoly(double px, double py, Matrix[] poly) {
        double result = 0.0;
        boolean first = true;
        for(int i=0; i<poly.length; i++) {
            double thisresult = getDistOfPointFromLine(px, py, poly[i], poly[(i+1) % poly.length]);
            if((thisresult<result || first) && !Double.isNaN(thisresult)) {
                result = thisresult;
                first=false;
            } else {
                // do nothing
            }
        }
        assert !first;
        return result;
    }
    
    public static double getHarvilleDistOfPointFromPoly(double px, double py, Matrix[] poly) {
        double result = 1.0;
        int ncontrib = 0;
        for(int i=0; i<poly.length; i++) {
            double thisresult = getDistOfPointFromLine(px, py, poly[i], poly[(i+1) % poly.length]);
            if(Double.isNaN(thisresult)) {
                // ignore
            } else {
                result *= thisresult;
                ncontrib++;
            }
        }
        
        // need to normalise for number of sides!
        return Math.pow(result,4.0/ncontrib);
    }    
    
    // returns NAN if lp1 and lp2 are equal
    public static double getDistOfPointFromLine(double px, double py, Matrix lp1, Matrix lp2) {
        Line2D l = new Line2D.Double(
                lp1.get(0,0)/lp1.get(2,0), //x1
                lp1.get(1,0)/lp1.get(2,0), //y1
                lp2.get(0,0)/lp2.get(2,0), //x2
                lp2.get(1,0)/lp2.get(2,0)  //y2
        );
        double r = l.ptSegDist(px, py);
        return r;
    }
    
    // returns NAN if lp1 and lp2 are equal
    public static double getDistOfPointFromLine(Matrix p, Matrix lp1, Matrix lp2) {
        return getDistOfPointFromLine(p.get(0,0)/p.get(2,0), p.get(1,0)/p.get(2,0), lp1, lp2);
    }

	
	public static List<List<Matrix>> get2hmPolysFromPolygonalArea(java.awt.geom.Area a) {

		if(!a.isPolygonal()) {
			throw new IllegalArgumentException("Area is not polygonal");
		}
		List<List<Matrix>> r = new LinkedList<List<Matrix>>();
		PathIterator p = a.getPathIterator(null);
		double[] coords = new double[6];
		List<Matrix> currentPoly=null;
		while (!p.isDone()) {
			int segmentType = p.currentSegment(coords);
			p.next();
			if (segmentType == p.SEG_MOVETO && currentPoly == null && !p.isDone()) {
				currentPoly = new LinkedList<Matrix>();
				currentPoly.add(JOGLHelper.getMFromD(coords[0], coords[1], 1));
			} else if (segmentType == p.SEG_LINETO && !p.isDone() && currentPoly!=null) {
				currentPoly.add(JOGLHelper.getMFromD(coords[0], coords[1], 1));
			} else if (segmentType == p.SEG_CLOSE) {
				assert currentPoly.size()>=3;
				r.add(currentPoly);
				currentPoly = null;
			} else {
				// incompatible Area
				throw new IllegalArgumentException("Area does not contain 1 or more closed polygons and nothing else");
			}
		}
		return r;
	}	

	public static Matrix[] get2hmCornerPointsFrom2r(java.awt.geom.Rectangle2D r) {
		Matrix[] result = new Matrix[] {
				new Matrix(new double[][] { new double[] { r.getX(), r.getY(),
						1.0 } }),
				new Matrix(new double[][] { new double[] {
						r.getX() + r.getWidth(), r.getY(), 1.0 } }),
				new Matrix(
						new double[][] { new double[] {
								r.getX() + r.getWidth(),
								r.getY() + r.getHeight(), 1.0 } }),
				new Matrix(new double[][] { new double[] { r.getX(),
						r.getY() + r.getHeight(), 1.0 } }), };
		return result;
	}

	public static Matrix getPlaneToPlaneHomogMatrixFromFour2hmPointCorrespondances(
			Matrix[] pointsFrom2hm, Matrix[] pointsTo2hm) {

		// from ashdown's dissertation p59

		assert (pointsFrom2hm.length == 4);
		assert (pointsTo2hm.length == 4);

		// first unhomogenise the points
		double[][] pointsFrom2ud = new double[4][2];
		double[][] pointsTo2ud = new double[4][2];
		for (int i = 0; i < 4; i++) {
			pointsFrom2ud[i] = unhomogeniseD(getDFromM(pointsFrom2hm[i]));
			pointsTo2ud[i] = unhomogeniseD(getDFromM(pointsTo2hm[i]));
		}

		// now construct the matrix
		Matrix A = new Matrix(new double[][] {
				new double[] { pointsFrom2ud[0][0], pointsFrom2ud[0][1], 1, 0,
						0, 0, -pointsFrom2ud[0][0] * pointsTo2ud[0][0],
						-pointsFrom2ud[0][1] * pointsTo2ud[0][0] },
				new double[] { pointsFrom2ud[1][0], pointsFrom2ud[1][1], 1, 0,
						0, 0, -pointsFrom2ud[1][0] * pointsTo2ud[1][0],
						-pointsFrom2ud[1][1] * pointsTo2ud[1][0] },
				new double[] { pointsFrom2ud[2][0], pointsFrom2ud[2][1], 1, 0,
						0, 0, -pointsFrom2ud[2][0] * pointsTo2ud[2][0],
						-pointsFrom2ud[2][1] * pointsTo2ud[2][0] },
				new double[] { pointsFrom2ud[3][0], pointsFrom2ud[3][1], 1, 0,
						0, 0, -pointsFrom2ud[3][0] * pointsTo2ud[3][0],
						-pointsFrom2ud[3][1] * pointsTo2ud[3][0] },
				new double[] { 0, 0, 0, pointsFrom2ud[0][0],
						pointsFrom2ud[0][1], 1,
						-pointsFrom2ud[0][0] * pointsTo2ud[0][1],
						-pointsFrom2ud[0][1] * pointsTo2ud[0][1] },
				new double[] { 0, 0, 0, pointsFrom2ud[1][0],
						pointsFrom2ud[1][1], 1,
						-pointsFrom2ud[1][0] * pointsTo2ud[1][1],
						-pointsFrom2ud[1][1] * pointsTo2ud[1][1] },
				new double[] { 0, 0, 0, pointsFrom2ud[2][0],
						pointsFrom2ud[2][1], 1,
						-pointsFrom2ud[2][0] * pointsTo2ud[2][1],
						-pointsFrom2ud[2][1] * pointsTo2ud[2][1] },
				new double[] { 0, 0, 0, pointsFrom2ud[3][0],
						pointsFrom2ud[3][1], 1,
						-pointsFrom2ud[3][0] * pointsTo2ud[3][1],
						-pointsFrom2ud[3][1] * pointsTo2ud[3][1] } });
		Matrix otherSide = getMFromD(new double[] { pointsTo2ud[0][0],
				pointsTo2ud[1][0], pointsTo2ud[2][0], pointsTo2ud[3][0],
				pointsTo2ud[0][1], pointsTo2ud[1][1], pointsTo2ud[2][1],
				pointsTo2ud[3][1] });

		// now A.h = otherSide so let's solve to find hVector
		Matrix homogVectorM = A.solve(otherSide);

		// now get the homography matrix
		double[] homogVectorD = getDFromM(homogVectorM);

		Matrix homogMatrix = new Matrix(new double[][] {
				new double[] { homogVectorD[0], homogVectorD[1],
						homogVectorD[2] },
				new double[] { homogVectorD[3], homogVectorD[4],
						homogVectorD[5] },
				new double[] { homogVectorD[6], homogVectorD[7], 1 } });

		return homogMatrix;
	}

	
	private static Matrix getNormalisationMatrixForPlaneToPlaneAlg(Matrix[] ms) {

		// calculate centroid
		double cx = 0.0, cy = 0.0;
		for (Matrix m : ms) {
			cx += m.get(0, 0) / m.get(2, 0);
			cy += m.get(1, 0) / m.get(2, 0);
		}
		cx /= ms.length;
		cy /= ms.length;

		// calculate length
		double len = 0.0;
		for (Matrix m : ms) {
			len += Math.sqrt(m.get(0, 0) * m.get(0, 0) + m.get(1, 0)
					* m.get(1, 0))
					/ m.get(2, 0);
		}

		// Scale vectors so the average length is root 2.
		double s = Math.sqrt(2.0) / len;

		return new Matrix(new double[][] { new double[] { s, 0, s * -cx },
				new double[] { 0, s, s * -cy }, new double[] { 0, 0, 1 } });
	}
	
	

	public static Matrix getPlaneToPlaneHomogMatrixFromN2hmPointCorrespondances(
			Matrix[] pointsFrom2hm, Matrix[] pointsTo2hm, int n) {

		// from ashdown's dissertation p59
		// and from ashdown's PGFuncs.cpp

		assert (pointsFrom2hm.length == n);
		assert (pointsTo2hm.length == n);

		// first normalise the points. apparently this is a good idea.
		Matrix normFrom = getNormalisationMatrixForPlaneToPlaneAlg(pointsFrom2hm);
		Matrix normTo = getNormalisationMatrixForPlaneToPlaneAlg(pointsTo2hm);
		Matrix normFromInv = normFrom.inverse();
		Matrix normToInv = normTo.inverse();

		// first unhomogenise the points
		double[][] pointsFrom2ud = new double[n][2];
		double[][] pointsTo2ud = new double[n][2];
		for (int i = 0; i < n; i++) {
			pointsFrom2ud[i] = unhomogeniseD(getDFromM(normFrom
					.times(pointsFrom2hm[i])));
			pointsTo2ud[i] = unhomogeniseD(getDFromM(normTo
					.times(pointsTo2hm[i])));
		}

		// now construct the matrix
		Matrix A = new Matrix(n * 2, 9);
		for (int i = 0; i < n; i++) {
			A.set(i, 0, pointsFrom2ud[i][0]);
			A.set(i, 1, pointsFrom2ud[i][1]);
			A.set(i, 2, 1);
			A.set(i, 3, 0);
			A.set(i, 4, 0);
			A.set(i, 5, 0);
			A.set(i, 6, -pointsFrom2ud[i][0] * pointsTo2ud[i][0]);
			A.set(i, 7, -pointsFrom2ud[i][1] * pointsTo2ud[i][0]);
			A.set(i, 8, -pointsTo2ud[i][0]);
			A.set(n + i, 0, 0);
			A.set(n + i, 1, 0);
			A.set(n + i, 2, 0);
			A.set(n + i, 3, pointsFrom2ud[i][0]);
			A.set(n + i, 4, pointsFrom2ud[i][1]);
			A.set(n + i, 5, 1);
			A.set(n + i, 6, -pointsFrom2ud[i][0] * pointsTo2ud[i][1]);
			A.set(n + i, 7, -pointsFrom2ud[i][1] * pointsTo2ud[i][1]);
			A.set(n + i, 8, -pointsTo2ud[i][1]);
		}

		// do svd and h is the righthand column
		// can crash on this call unless we're careful about the contents of A
		SingularValueDecomposition svd = A.svd();
		Matrix v = svd.getV();

		Matrix homogMatrix = new Matrix(new double[][] {
				new double[] { v.get(0, 8), v.get(1, 8), v.get(2, 8) },
				new double[] { v.get(3, 8), v.get(4, 8), v.get(5, 8) },
				new double[] { v.get(6, 8), v.get(7, 8), v.get(8, 8) } });
		
		// compensate for our normalisation
		homogMatrix = normToInv.times(homogMatrix.times(normFrom));

		// now divide through so bottom right is a 1.0
		homogMatrix = homogMatrix.times(1.0/homogMatrix.get(2,2));

		return homogMatrix;
	}

	public static void oglVertex4dFromM(GL gl, Matrix m) {
		gl.glVertex4d(m.get(0, 0), m.get(1, 0), m.get(2, 0), m.get(3, 0));
	}
	
	public static void oglVertex4dFromMs(GL gl, List<Matrix> ms, Matrix transform) {
		for(Matrix m: ms) {
			oglVertex4dFromM(gl, transform.times(m) );
		}
	}

	public static void oglThrowExceptionIfError(GL gl, GLU glu)
			throws OGLException {
		int e = gl.glGetError();
		if (e != 0) {
			throw new OGLException("OpenGL error code " + e + ": "
					+ glu.gluErrorString(e));
		}
	}

	public static void oglReportIfError(GL gl, GLU glu) {
		try {
			oglThrowExceptionIfError(gl, glu);
		} catch (OGLException e) {
			e.printStackTrace();
		}
	}
	
	public static void oglColor(GL gl, Color c) {
		gl.glColor4d(c.getRed()/255.0,c.getGreen()/255.0,c.getBlue()/255.0, c.getAlpha()/255.0);
		
	}
	

	public static void oglColorFromARGBint(GL gl, int ARGB) {
		gl.glColor4d( 
			((ARGB & 0x00ff0000) >>> 16)/255.0,
			((ARGB & 0x0000ff00) >>> 8)/255.0, 
			((ARGB & 0x000000ff) >>> 0)/255.0, 
			((ARGB & 0xff000000) >>> 24)/255.0
		);
	}

	public static class OGLException extends RuntimeException {
		public OGLException(String s) {
			super(s);
		}
	}

	public static String matrixToString(Matrix m, int w, int d) {
		DecimalFormat format = new DecimalFormat();
		format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
		format.setMinimumIntegerDigits(1);
		format.setMaximumFractionDigits(d);
		format.setMinimumFractionDigits(d);
		format.setGroupingUsed(false);
		int width = w - 2;

		String s = "";
		double[][] a = m.getArray();
		for (int i = 0; i < m.getRowDimension(); i++) {
			if (i != 0) {
				s = s + '\n';
			}
			for (int j = 0; j < m.getColumnDimension(); j++) {
				int padding = Math.max(1, width - s.length()); // At _least_ 1
																// space
				for (int k = 0; k < padding; k++)
					s = s + ' ';
				s = s + format.format(a[i][j]);
			}
		}
		return s;
	}

	/*
	 * public static WritableRaster createNewRasterForTexture(int w, int h) {
	 * return java.awt.image.Raster.createWritableRaster(
	 * createNewSampleModelForTexture(w, h), null ); }
	 * 
	 * public static SampleModel createNewSampleModelForTexture(int w, int h) {
	 * return new PixelInterleavedSampleModel( DataBuffer.TYPE_BYTE, w,h, 4,
	 * 4*w, new int[] { 3, 2, 1, 0 } ); }
	 * 
	 * public static ComponentColorModel createNewColorModelForTexture() {
	 * return new ComponentColorModel(
	 * ColorSpace.getInstance(ColorSpace.CS_sRGB), new int[] {8,8,8,8}, true,
	 * false, Transparency.TRANSLUCENT, DataBuffer.TYPE_BYTE ); }
	 * 
	 * public static BufferedImage createNewBufferedImageForTexture(int w, int
	 * h) { BufferedImage i = new BufferedImage(
	 * createNewColorModelForTexture(), createNewRasterForTexture(w,h), false,
	 * null); assert i.getType() == BufferedImage.TYPE_4BYTE_ABGR; return i; }
	 * 
	 * 
	 * 
	 * public static WritableRaster byteArrayToWritableRasterForTexture(byte[]
	 * buff, int w, int h) {
	 * 
	 * return java.awt.image.Raster.createWritableRaster(
	 * createNewSampleModelForTexture(w, h), new DataBufferByte(buff,
	 * buff.length), null ); }
	 * 
	 * public static byte[] bufferedImageToByteArray(BufferedImage b) { Raster
	 * r; if( b.getSampleModel().getWidth()== b.getWidth() &&
	 * b.getSampleModel().getHeight() == b.getHeight() ) { // the image uses all
	 * of its byte buffer // we'll just grab the byte buffer, no need to make a //
	 * seperate copy r = b.getRaster(); } else { // the image uses some sub
	 * portion of a byte buffer // we don't want to get the entire byte buffer //
	 * so copy the relevent bit! r = b.getData(new Rectangle(0,0,b.getWidth(),
	 * b.getHeight())); } // now return the data array - not a copy of it return (
	 * (DataBufferByte)(r.getDataBuffer()) ).getData(); }
	 */
	
	/******************* writing out a properties file ***********/
	
	public static void storePropertiesSorted(Properties p, OutputStream out, String comments)
    throws IOException
    {
        BufferedWriter awriter;
        awriter = new BufferedWriter(new OutputStreamWriter(out, "8859_1"));
        if (comments != null) {
        	awriter.write("#" + comments);
            awriter.newLine();
        }
        awriter.write("#" + new Date().toString());
        awriter.newLine();
        List l = new LinkedList(p.keySet());
        Collections.sort(l);
        
        for (Object k: l) {
        	String key = (String)k;
            String val = (String)p.get(key);
            key = saveprops_saveConvert(key, true);

	    /* No need to escape embedded and trailing spaces for value, hence
	     * pass false to flag.
	     */
            val = saveprops_saveConvert(val, false);
            awriter.write(key + "=" + val);
            awriter.newLine();
        }
        awriter.flush();
    }
	
	
	
	private static String saveprops_saveConvert(String theString, boolean escapeSpace) {
        int len = theString.length();
        int bufLen = len * 2;
        if (bufLen < 0) {
            bufLen = Integer.MAX_VALUE;
        }
        StringBuffer outBuffer = new StringBuffer(bufLen);

        for(int x=0; x<len; x++) {
            char aChar = theString.charAt(x);
            // Handle common case first, selecting largest block that
            // avoids the specials below
            if ((aChar > 61) && (aChar < 127)) {
                if (aChar == '\\') {
                    outBuffer.append('\\'); outBuffer.append('\\');
                    continue;
                }
                outBuffer.append(aChar);
                continue;
            }
            switch(aChar) {
		case ' ':
		    if (x == 0 || escapeSpace) 
			outBuffer.append('\\');
		    outBuffer.append(' ');
		    break;
                case '\t':outBuffer.append('\\'); outBuffer.append('t');
                          break;
                case '\n':outBuffer.append('\\'); outBuffer.append('n');
                          break;
                case '\r':outBuffer.append('\\'); outBuffer.append('r');
                          break;
                case '\f':outBuffer.append('\\'); outBuffer.append('f');
                          break;
                case '=': // Fall through
                case ':': // Fall through
                case '#': // Fall through
                case '!':
                    outBuffer.append('\\'); outBuffer.append(aChar);
                    break;
                default:
                    if ((aChar < 0x0020) || (aChar > 0x007e)) {
                        outBuffer.append('\\');
                        outBuffer.append('u');
                        outBuffer.append(saveprops_toHex((aChar >> 12) & 0xF));
                        outBuffer.append(saveprops_toHex((aChar >>  8) & 0xF));
                        outBuffer.append(saveprops_toHex((aChar >>  4) & 0xF));
                        outBuffer.append(saveprops_toHex( aChar        & 0xF));
                    } else {
                        outBuffer.append(aChar);
                    }
            }
        }
        return outBuffer.toString();
    }
	
    private static char saveprops_toHex(int nibble) {
    	return saveprops_hexDigit[(nibble & 0xF)];
        }

    /** A table of hex digits */
    private static final char[] saveprops_hexDigit = {
	'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
    };

    
    
    
    
    
    /************* tesselation! ****************/
    
    // http://fly.cc.fer.hr/~unreal/theredbook/appendixc.html
    // http://download.java.net/media/jogl/builds/nightly/javadoc_public/
    
    
    public static class TesselatedPolygonComponent {
    	public final int fanOrStripOrTriangles;
    	public final List<double[]> points = new LinkedList<double[]>();
		public TesselatedPolygonComponent(int fanOrStripOrTriangles) {
			this.fanOrStripOrTriangles = fanOrStripOrTriangles;
		}
    	
    }
    
    
    public static List<TesselatedPolygonComponent> tesselateJavaAreaInto2hmPolys(Area a, final Matrix mTransform) {
    	// mTransform can be 2hto3h or 2hto2h
    	
    	PathIterator pit = a.getPathIterator(null);
    	List<TesselatedPolygonComponent> result = new LinkedList<TesselatedPolygonComponent>();
    	
    	GLU glu = new GLU();
    	GLUtessellator tobj = glu.gluNewTess();
    	GLUtessellatorCallback callback = new GLUtessellatorCallbackAdapter() {
    		public void beginData(int type, Object polygonData)
    		{
    			List<TesselatedPolygonComponent> result = (List<TesselatedPolygonComponent>)polygonData;
    			result.add(new TesselatedPolygonComponent(type));
    		}

    		public void vertexData(Object vertexData, Object polygonData)
    		{
    			List<TesselatedPolygonComponent> result = (List<TesselatedPolygonComponent>)polygonData;
    			Matrix mPoint = getMFromD((double[])vertexData);
    			result.get(result.size()-1).points.add(JOGLHelper.getDFromM(mTransform.times(mPoint)));
    		}

    		public void endData(Object polygonData)
    		{
    		}

    		public void combineData(double[] coords, Object[] data, float[] weight, Object[] outData,
    			Object polygonData)
    		{
    			outData[0] = coords;

    		}

    		public void errorData(int errnum, Object polygonData)
    		{
    			throw new IllegalArgumentException("OpenGL error while tesselating: "+errnum);
    		}
    	};    	

		glu.gluTessCallback(tobj, GLU.GLU_TESS_BEGIN_DATA, callback);
		glu.gluTessCallback(tobj, GLU.GLU_TESS_VERTEX_DATA, callback);
		glu.gluTessCallback(tobj, GLU.GLU_TESS_COMBINE_DATA, callback);
		glu.gluTessCallback(tobj, GLU.GLU_TESS_END_DATA, callback);
		glu.gluTessCallback(tobj, GLU.GLU_TESS_ERROR_DATA, callback);
		
		if (pit.getWindingRule()== PathIterator.WIND_EVEN_ODD) {
            glu.gluTessProperty(tobj, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_ODD);
		} else {
			assert pit.getWindingRule()== PathIterator.WIND_NON_ZERO;
            glu.gluTessProperty(tobj, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_NONZERO);
		}
		
		glu.gluTessProperty(tobj, GLU.GLU_TESS_TOLERANCE, 0);
		//glu.gluTessNormal(tobj, 0.0f, 0.0f, zNormal);
		glu.gluTessBeginPolygon(tobj,result);

		double[] coords = new double[6];
		boolean inContour = false;
				
		while(!pit.isDone()) {
			List<Matrix> currentPoly=null;
			int segmentType = pit.currentSegment(coords);
			if (segmentType == pit.SEG_MOVETO) {
				assert !inContour;
				glu.gluTessBeginContour(tobj);
				inContour = true;
				double[] d2hPoint = getDFromM(getMFromD(coords[0], coords[1], 1.0));
				glu.gluTessVertex(tobj, d2hPoint, 0, d2hPoint);
			} else if (segmentType == pit.SEG_LINETO) {
				assert inContour;
				double[] d2hPoint = getDFromM(getMFromD(coords[0], coords[1], 1.0));
				glu.gluTessVertex(tobj, d2hPoint, 0, d2hPoint);
			} else if (segmentType == pit.SEG_CLOSE) {
				assert inContour;
				glu.gluTessEndContour(tobj);
				inContour = false;
				// no need to do anything
			} else {
				// incompatible Area
				throw new IllegalArgumentException("Area does not contain 1 or more closed polygons and nothing else");
			}
			pit.next();
		}
		
		glu.gluTessEndPolygon(tobj);
		glu.gluDeleteTess(tobj);
		
    	return result;
    }

    public static void oglDrawTesselatedPolygon(GL gl, List<TesselatedPolygonComponent> l) {
    	//int n=0;
    	for(TesselatedPolygonComponent t: l) {
    		gl.glBegin(t.fanOrStripOrTriangles);
    		for(double[] vertex: t.points) {
    			gl.glVertex4dv(vertex,0);
    			//n++;
    		}
    		gl.glEnd();
    	}
    	//System.out.println(n);
    }


    
    
    
    
    /************* tesselation - direct drawing! ****************/
    
    // http://fly.cc.fer.hr/~unreal/theredbook/appendixc.html
    // http://download.java.net/media/jogl/builds/nightly/javadoc_public/
    private static boolean oglTesselateAndDraw2hmPoly_configured = false;
    private static GL oglTesselateAndDraw2hmPoly_gl = null;
    private static Matrix oglTesselateAndDraw2hmPoly_trans = null;
    private static GLU oglTesselateAndDraw2hmPoly_glu = new GLU();
    private static GLUtessellator oglTesselateAndDraw2hmPoly_tobj = oglTesselateAndDraw2hmPoly_glu.gluNewTess();
    private static GLUtessellatorCallback oglTesselateAndDraw2hmPoly_callback = new GLUtessellatorCallbackAdapter() {
        public void beginData(int type, Object polygonData)
        {
            oglTesselateAndDraw2hmPoly_gl.glBegin(type);
        }

        public void vertexData(Object vertexData, Object polygonData)
        {
            Matrix mPoint = oglTesselateAndDraw2hmPoly_trans.times(getMFromD((double[])vertexData));
            JOGLHelper.oglVertex4dFromM(oglTesselateAndDraw2hmPoly_gl, mPoint); 
        }

        public void endData(Object polygonData)
        {
            oglTesselateAndDraw2hmPoly_gl.glEnd();
        }

        public void combineData(double[] coords, Object[] data, float[] weight, Object[] outData,
            Object polygonData)
        {
            double[] vertex = new double[3];
            vertex[0] = coords[0];
            vertex[1] = coords[1];
            vertex[2] = coords[2];
            outData[0] = vertex;
        }

        public void errorData(int errnum, Object polygonData)
        {
            throw new IllegalArgumentException("OpenGL error while tesselating: "+errnum);
        }
    };
    
    
    
    public static void oglTesselateAndDraw2hmPoly(GL gl, Iterable<Matrix> m2hPoints, final Matrix m2hTo3hTransform) {
        if(!oglTesselateAndDraw2hmPoly_configured) {
            oglTesselateAndDraw2hmPoly_glu.gluTessCallback(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_BEGIN_DATA, oglTesselateAndDraw2hmPoly_callback);
            oglTesselateAndDraw2hmPoly_glu.gluTessCallback(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_VERTEX_DATA, oglTesselateAndDraw2hmPoly_callback);
            oglTesselateAndDraw2hmPoly_glu.gluTessCallback(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_COMBINE_DATA, oglTesselateAndDraw2hmPoly_callback);
            oglTesselateAndDraw2hmPoly_glu.gluTessCallback(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_END_DATA, oglTesselateAndDraw2hmPoly_callback);
            oglTesselateAndDraw2hmPoly_glu.gluTessCallback(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_ERROR_DATA, oglTesselateAndDraw2hmPoly_callback);
            oglTesselateAndDraw2hmPoly_glu.gluTessProperty(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_ODD);
            oglTesselateAndDraw2hmPoly_glu.gluTessProperty(oglTesselateAndDraw2hmPoly_tobj, GLU.GLU_TESS_TOLERANCE, 0);
            oglTesselateAndDraw2hmPoly_configured=true;
            
        }
        oglTesselateAndDraw2hmPoly_gl = gl;
        oglTesselateAndDraw2hmPoly_trans = m2hTo3hTransform;
        oglTesselateAndDraw2hmPoly_glu.gluTessBeginPolygon(oglTesselateAndDraw2hmPoly_tobj,new Object());
        oglTesselateAndDraw2hmPoly_glu.gluTessBeginContour(oglTesselateAndDraw2hmPoly_tobj);
        for(Matrix m: m2hPoints) {
            // we do need to make a new double array for every point otherwise it doesn't work!
            double[] d2hPoint = JOGLHelper.getDFromM(m);
            assert d2hPoint.length==3;
            oglTesselateAndDraw2hmPoly_glu.gluTessVertex(oglTesselateAndDraw2hmPoly_tobj, d2hPoint, 0, d2hPoint);
        }
        oglTesselateAndDraw2hmPoly_glu.gluTessEndContour(oglTesselateAndDraw2hmPoly_tobj);
        oglTesselateAndDraw2hmPoly_glu.gluTessEndPolygon(oglTesselateAndDraw2hmPoly_tobj);
    }

 
    
}