/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cassert>
#include <stdexcept>
#include "halfedge.h"
#include "knotinterval.h"
#include "patchtreewalker.h"
#include "patchtree.h"
#include "patchtreeleaf.h"
#include "patchtreeroot.h"

namespace snurbs {

using namespace std;

PatchTreeWalker::PatchTreeWalker(const PatchTree *start,
                                 unsigned char rotateSteps) :
                                     currentPatchTree(start),
                                     rotation        (rotateSteps)
{
    assert(rotateSteps < 4);
}

void PatchTreeWalker::followPath(const Path &path)
{
    const Path *currentPath = &path;

    while (!currentPatchTree->isLeaf())
    {
        currentPatchTree =
            currentPatchTree->getChild(rotate(currentPath->getDir()));
        currentPath = currentPath->getNext();

        if (currentPath == NULL)
        {
            break;
        }
    }
}

KnotInterval *PatchTreeWalker::getKnotInterval
              (PatchTree::Orientation orientation) const
{
    if (currentPatchTree->isLeaf())
    {
        const PatchTreeLeaf *leaf =
            static_cast<const PatchTreeLeaf *>(currentPatchTree);

        assert(orientation == PatchTree::VERTICAL ||
               orientation == PatchTree::HORIZONTAL);

        if ((orientation == PatchTree::VERTICAL   && rotation % 2 == 0) ||
            (orientation == PatchTree::HORIZONTAL && rotation % 2 == 1))
        {
            return leaf->getKnotInterval(PatchTree::VERTICAL);
        }
        else
        {
            return leaf->getKnotInterval(PatchTree::HORIZONTAL);
        }
    }
    else
    {
        PatchTreeWalker w(currentPatchTree->getChild(PatchTree::ANY),
                          rotation);
        KnotInterval *childKnot = w.getKnotInterval(orientation);

        if (splitsOrientation(orientation))
        {
            childKnot = childKnot->getParent();
        }
        return childKnot;
    }
}

bool PatchTreeWalker::knotIsNew(PatchTree::Direction dir) const
{
    PatchTree::Orientation orientation = getOrientationForDir(dir);
    KnotInterval *knot = getKnotInterval(orientation);

    if (knot->getParent() == NULL)
    {
        return false;
    }

    bool splits = PatchTreeWalker(currentPatchTree->getParent(),
                                  rotation).splitsOrientation(orientation);

    if (splits)
    {
        bool isLeftChild = knot->getParent()->getLeftChild() == knot;

        return isLeftChild ^
               intervalIsFlipped(orientation) ^
               (dir == PatchTree::NORTH || dir == PatchTree::WEST);
    }
    else
    {
        return false;
    }
}

bool PatchTreeWalker::intervalIsFlipped(PatchTree::Orientation orientation)
const
{
    // By looking at each of the cases, we find that this function must
    // satisfy the following truth table:
    //
    //                      rotation
    //                   | 0  1  2  3
    //                ---+------------
    //  orientation    H | F  F  T  T
    //                 V | F  T  T  F

    // Therefore...

    switch (orientation)
    {
        case PatchTree::VERTICAL:
            return (rotation == 1) || (rotation == 2);
        case PatchTree::HORIZONTAL:
            return (rotation > 1);
    }

    throw runtime_error("Undefined orientation in intervalIsFlipped");
}

void PatchTreeWalker::findNeighbour(PatchTree::Direction dir, const Path *path)
{
    if (dir != PatchTree::NORTH &&
        dir != PatchTree::EAST  &&
        dir != PatchTree::SOUTH &&
        dir != PatchTree::WEST)
    {
        throw logic_error("findNeighbour called with invalid direction");
    }

    if (currentPatchTree->isRoot())
    {
        // Cross to neighbouring PatchTree using HalfEdge objects. May involve
        // a change to rotation
        const PatchTreeRoot *root =
            static_cast<const PatchTreeRoot *>(currentPatchTree);

        unsigned char numNext = rotation;
        unsigned char dirCopy = dir >> 1;
        while (dirCopy)
        {
            dirCopy >>= 1;
            ++numNext;
        }
        numNext %= 4;

        HalfEdge::PatchIterator iter(root->getEdge());
        while (numNext)
        {
            ++iter;
            --numNext;
        }

        HalfEdge *pair = (*iter)->getPair();
        PatchTreeRoot *neighbour = pair->getPatch();
        if (neighbour == NULL)
        {
            throw hit_boundary();
        }

        currentPatchTree = neighbour;

        switch (dir)
        {
            case PatchTree::NORTH:
                pair = pair->getNext()->getNext();
                break;
            case PatchTree::EAST:
                pair = pair->getNext();
                break;
            case PatchTree::WEST:
                pair = pair->prev();
                break;
            default:
                // Nothing
                break;
        }

        HalfEdge::PatchIterator pseudoNorth(neighbour->getEdge());
        unsigned char rotCopy = rotation;
        while (rotCopy)
        {
            ++pseudoNorth;
            --rotCopy;
        }
        rotation += (*pseudoNorth)->stepsTo(pair);
        rotation %= 4;

        if (path)
        {
            followPath(*path);
        }
    }
    else
    {
        PatchTree::Direction fromP = rotate(currentPatchTree->dirFromParent(),
                                            (4 - rotation) % 4);

        if (fromP & rotate(dir, 2))
        {
            // The neighbour is directly accessible from the parent

            // Calculate which child we require (remove any component in
            // direction opposite to `dir', then add in dir). For example, if
            // dir is S and fromP is NW, this calculates child as SW.
            PatchTree::Direction child =
                static_cast<PatchTree::Direction>((fromP & (~rotate(dir, 2)))
                                                  | dir);

            // Set the currentPatchTree to the required child
            currentPatchTree = currentPatchTree->getParent();

            followPath(Path(child, path));
        }
        else
        {
            // Not a root, but not directly accessible either.
            currentPatchTree = currentPatchTree->getParent();

            // Calculate which child we require in the neighbour (remove any
            // component in direction the same as `dir', then add in the
            // direction opposite to dir). For example, if dir is N and fromP
            // is W or NW, this calculates child as SW.
            PatchTree::Direction child =
                static_cast<PatchTree::Direction>((fromP & ~dir) |
                                                  rotate(dir, 2));

            Path neighbourPath(child, path);
            findNeighbour(dir, &neighbourPath);
        }
    }
}

bool PatchTreeWalker::splitsOrientation(PatchTree::Orientation orientation)
const
{
    PatchTree::Orientation otherOrient =
                static_cast<PatchTree::Orientation>(1 - orientation);

    return ((currentPatchTree->splits(orientation) && rotation % 2 == 0) ||
            (currentPatchTree->splits(otherOrient) && rotation % 2 == 1));
}

KnotPrecision PatchTreeWalker::knotSum(PatchTree::Direction dir,
                                       unsigned char steps)
{
    PatchTree::Orientation orientation = getOrientationForDir(dir);
    KnotPrecision knotSum = 0;

    while(steps)
    {
        findNeighbour(dir);
        knotSum += getKnotInterval(orientation)->getInterval();
        --steps;
    }

    return knotSum;
}

PatchTree::Direction PatchTreeWalker::rotate(PatchTree::Direction dir,
                                             unsigned char steps)
{
    assert(steps < 4);

    // Because of the definition of directions in PatchTree::Direction,
    // rotation is equivalent to a rolling bitwise shift left on the
    // lower four bits.
    return static_cast<PatchTree::Direction>(((dir << steps) & 0x0F) |
                                             (dir >> (4 - steps)));
}

PatchTree::Orientation PatchTreeWalker::getOrientationForDir
                           (PatchTree::Direction d)
{
    switch (d)
    {
        case PatchTree::NORTH:
        case PatchTree::SOUTH:
            return PatchTree::VERTICAL;
        case PatchTree::WEST:
        case PatchTree::EAST:
            return PatchTree::HORIZONTAL;
        default:
            throw logic_error("getOrientationForDir called with "
                              "invalid direction");
    }
}

PatchTree::Direction PatchTreeWalker::rotate(PatchTree::Direction dir)
{
    return rotate(dir, rotation);
}

}
