/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cassert>
#include <limits>
#include "vertex.h"

namespace snurbs {

using namespace std;

Vertex::Vertex(VertexPrecision x,
               VertexPrecision y,
               VertexPrecision z,
               VertexPrecision w) : replacement(NULL)
{
    coords[0] = x;
    coords[1] = y;
    coords[2] = z;
    coords[3] = w;
}

Vertex::Vertex(void)
{
    coords[0] = 0;
    coords[1] = 0;
    coords[2] = 0;
    coords[3] = 1;
    initReplacement();
}

Vertex::~Vertex()
{
    if (hasReplacement())
    {
        delete replacement;
        replacement = NULL;
    }
}

bool Vertex::isReal(void) const
{
    return !(is_nan(coords[0]) ||
             is_nan(coords[1]) ||
             is_nan(coords[2]) ||
             is_nan(coords[3]));
}

const Vertex Vertex::operator-(const Vertex &v) const
{
    return Vertex(coords[0] - v.coords[0],
                  coords[1] - v.coords[1],
                  coords[2] - v.coords[2],
                  coords[3] - v.coords[3]);
}

const Vertex Vertex::operator+(const Vertex &v) const
{
    return Vertex(coords[0] + v.coords[0],
                  coords[1] + v.coords[1],
                  coords[2] + v.coords[2],
                  coords[3] + v.coords[3]);
}

const Vertex Vertex::operator*(const Vertex &v) const
{
    assert(coords[3] == 0);
    assert(v.coords[3] == 0);

    return Vertex(coords[1] * v.coords[2] - coords[2] * v.coords[1],
                  coords[2] * v.coords[0] - coords[0] * v.coords[2],
                  coords[0] * v.coords[1] - coords[1] * v.coords[0],
                  0);
}

void Vertex::discardReplacement(void)
{
    replacement->coords[0] = 0;
    replacement->coords[1] = 0;
    replacement->coords[2] = 0;
    replacement->coords[3] = 0;
    replacement->denominator = 0;
}

void Vertex::receiveContribution(const Vertex &v, VertexPrecision weight)
{
    #pragma omp critical
    {
        if (isReal() && weight != 0)
        {
            if (!hasReplacement())
            {
                initReplacement();
            }

            replacement->coords[0] += v.coords[0] * weight;
            replacement->coords[1] += v.coords[1] * weight;
            replacement->coords[2] += v.coords[2] * weight;
            replacement->coords[3] += v.coords[3] * weight;
            replacement->denominator += weight;
        }
    }
}

void Vertex::replace(bool freeReplacement)
{
    if (hasReplacement())
    {
        if (replacement->denominator != 0)
        {
            coords[0] = replacement->coords[0] / replacement->denominator;
            coords[1] = replacement->coords[1] / replacement->denominator;
            coords[2] = replacement->coords[2] / replacement->denominator;
            coords[3] = replacement->coords[3] / replacement->denominator;
        }
        else if (is_nan(replacement->denominator))
        {
            setNotReal();
        }

        if (freeReplacement)
        {
            delete replacement;
            replacement = NULL;
        }
        else
        {
            discardReplacement();
        }
    }
}

const Vertex Vertex::project(void) const
{
    return Vertex(coords[0] / coords[3],
                  coords[1] / coords[3],
                  coords[2] / coords[3],
                  1);
}

void Vertex::initReplacement(void)
{
    replacement = new Replacement;
    replacement->coords[0] = 0;
    replacement->coords[1] = 0;
    replacement->coords[2] = 0;
    replacement->coords[3] = 0;
    replacement->denominator = 0;
}

void Vertex::setNotReal(void)
{
    coords[0] = numeric_limits<VertexPrecision>::quiet_NaN();
    coords[1] = numeric_limits<VertexPrecision>::quiet_NaN();
    coords[2] = numeric_limits<VertexPrecision>::quiet_NaN();
    coords[3] = numeric_limits<VertexPrecision>::quiet_NaN();
}

ostream &operator<<(ostream &os, const Vertex &vertex)
{
    os << &vertex << " :";
    os << " " << vertex.getX();
    os << " " << vertex.getY();
    os << " " << vertex.getZ();
    os << " " << vertex.getW();

    if (vertex.hasReplacement())
    {
        os << " : replacement ";
        os << vertex.replacement->coords[0] << " ";
        os << vertex.replacement->coords[1] << " ";
        os << vertex.replacement->coords[2] << " ";
        os << vertex.replacement->coords[3] << " , ";
        os << vertex.replacement->denominator;
    }

    os << endl;
    return os;
}

}
