/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef QUADPATCHTREE_H
#define QUADPATCHTREE_H

#include <tr1/array>
#include "patchtree.h"

namespace snurbs {

/**
 * Splits a PatchTree into four subtrees: NorthEast, SouthEast, SouthWest and
 * NorthWest.
 *
 * @ingroup Mesh
 */
class QuadPatchTree : public PatchTree
{
public:
    /** QuadPatchTree constructor. */
    QuadPatchTree(PatchTree *child1,
                  PatchTree *child2,
                  PatchTree *child3,
                  PatchTree *child4);

    /** QuadPatchTree destructor. */
    virtual ~QuadPatchTree(void);

    bool isLeaf(void) const;

    bool isRoot(void) const;

    bool splits(Orientation orientation) const;

    PatchTree *&getChild(Direction dir);

    PatchTree *&getParent(void);

    void setParent(PatchTree *newParent);

    Direction dirFromParent(void) const;

    Direction childDir(const PatchTree *child) const;

    /**
     * Used to implement
     * operator<<(std::ostream &os, const PatchTreeRoot &patch);
     */
    void stream(std::ostream &os, unsigned char level) const;

private:
    /** The parent PatchTree. */
    PatchTree *parent;
    /**
     * The four child PatchTree objects.
     *  - children[0] is the North-East child
     *  - children[1] is the South-East child
     *  - children[2] is the South-West child
     *  - children[3] is the North-West child
     */
    std::tr1::array<PatchTree *, 4> children;
};

inline bool QuadPatchTree::isLeaf(void) const
{
    return false;
}

inline bool QuadPatchTree::isRoot(void) const
{
    return false;
}

inline bool QuadPatchTree::splits(Orientation /* orientation */) const
{
    return true;
}

inline PatchTree *&QuadPatchTree::getParent(void)
{
    return parent;
}

inline void QuadPatchTree::setParent(PatchTree *newParent)
{
    parent = newParent;
}

}

#endif
