/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cmath>
#include <limits>
#include "bccoeffs.h"
#include "binarypatchtree.h"
#include "knotinterval.h"
#include "patchtreewalker.h"
#include "primalpatchtreeleaf.h"
#include "quadpatchtree.h"
#include "unarypatchtree.h"
#include "vertex.h"

namespace snurbs {

using namespace std;

PrimalPatchTreeLeaf::PrimalPatchTreeLeaf(void)
 : PatchTreeLeaf()
{
    vertices[0] = NULL;
    vertices[1] = NULL;
    vertices[2] = NULL;
    vertices[3] = NULL;
}

void PrimalPatchTreeLeaf::getNormal(std::tr1::array<VertexPrecision, 3>
                                    &normal) const
{
    Vertex projected[4] = {vertices[0]->project(),
                           vertices[1]->project(),
                           vertices[2]->project(),
                           vertices[3]->project()};

    const Vertex splitDiff = projected[2] - projected[0];
    const Vertex result    = ((projected[1] - projected[0]) * splitDiff)
                            + (splitDiff * (projected[3] - projected[0]));

    normal[0] = result.getX();
    normal[1] = result.getY();
    normal[2] = result.getZ();
    assert(result.getW() == 0);
}

void PrimalPatchTreeLeaf::makeInsertionRequests(void)
{
    try
    {
        PatchTreeWalker walker(this);
        walker.findNeighbour(PatchTree::WEST);
        KnotPrecision left =
            walker.getKnotInterval(PatchTree::HORIZONTAL)->getInterval();

        walker.reset(this);
        walker.findNeighbour(PatchTree::EAST);
        KnotPrecision right =
            walker.getKnotInterval(PatchTree::HORIZONTAL)->getInterval();

        KnotPrecision center      = horizontal->getInterval();
        KnotPrecision insertRatio = sqrt((center + left) / (center + right));

        horizontal->insertionRequest(center * insertRatio / (1 + insertRatio));
    }
    catch (hit_boundary&)
    {
        horizontal->insertionRequest(horizontal->getInterval() / 2);
    }

    try
    {
        PatchTreeWalker walker(this);
        walker.findNeighbour(PatchTree::NORTH);
        KnotPrecision top =
            walker.getKnotInterval(PatchTree::VERTICAL)->getInterval();

        walker.reset(this);
        walker.findNeighbour(PatchTree::SOUTH);
        KnotPrecision bottom =
            walker.getKnotInterval(PatchTree::VERTICAL)->getInterval();

        KnotPrecision center      = vertical->getInterval();
        KnotPrecision insertRatio = sqrt((center + top) / (center + bottom));

        vertical->insertionRequest(center * insertRatio / (1 + insertRatio));
    }
    catch (hit_boundary&)
    {
        vertical->insertionRequest(vertical->getInterval() / 2);
    }
}

PatchTree *PrimalPatchTreeLeaf::refineTree()
{
    bool splitVertical   =   vertical->toBeSubdivided();
    bool splitHorizontal = horizontal->toBeSubdivided();
    PatchTree *newChild  = NULL;
    PatchTree *oldParent = parent;

    if (splitVertical && splitHorizontal)
    {
        PrimalPatchTreeLeaf *se = new PrimalPatchTreeLeaf();
        PrimalPatchTreeLeaf *sw = new PrimalPatchTreeLeaf();
        PrimalPatchTreeLeaf *nw = new PrimalPatchTreeLeaf();

        newChild = new QuadPatchTree(this, se, sw, nw);

        se->vertices[1] = vertices[1];
        sw->vertices[2] = vertices[2];
        nw->vertices[3] = vertices[3];

        vertices[1] = NULL;
        vertices[2] = NULL;
        vertices[3] = NULL;

        se->horizontal = horizontal->getRightChild();
        se->vertical   = vertical  ->getRightChild();

        sw->horizontal = horizontal->getLeftChild();
        sw->vertical   = vertical  ->getRightChild();

        nw->horizontal = horizontal->getLeftChild();
        nw->vertical   = vertical  ->getLeftChild();

            horizontal = horizontal->getRightChild();
            vertical   = vertical  ->getLeftChild();
    }
    else if (splitVertical && !splitHorizontal)
    {
        PrimalPatchTreeLeaf *s = new PrimalPatchTreeLeaf();

        newChild = new BinaryPatchTree(VERTICAL, this, s);

        s->vertices[1] = vertices[1];
        s->vertices[2] = vertices[2];

        vertices[1] = NULL;
        vertices[2] = NULL;

        s->horizontal = horizontal;
        s->vertical   = vertical->getRightChild();
           vertical   = vertical->getLeftChild();
    }
    else if (!splitVertical && splitHorizontal)
    {
        PrimalPatchTreeLeaf *e = new PrimalPatchTreeLeaf();

        newChild = new BinaryPatchTree(HORIZONTAL, this, e);

        e->vertices[0] = vertices[0];
        e->vertices[1] = vertices[1];

        vertices[0] = NULL;
        vertices[1] = NULL;

        e->vertical   = vertical;
        e->horizontal = horizontal->getRightChild();
           horizontal = horizontal->getLeftChild();
    }
    else if (!splitVertical && !splitHorizontal)
    {
        newChild = new UnaryPatchTree(this);
    }

    newChild->setParent(oldParent);
    return newChild;
}

void PrimalPatchTreeLeaf::refineVertices(unsigned char degree)
{
    VertexPrecision vertComb = 0, horizComb = 0;

    if (parent->splits(VERTICAL))
    {
        // Parent is either a QuadPatchTree or a BinaryPatchTree with
        // VERTICAL splitType
        vertComb = getRefineComb(VERTICAL, degree);
    }

    if (parent->splits(HORIZONTAL))
    {
        // Parent is either a QuadPatchTree or a BinaryPatchTree with
        // HORIZONTAL splitType
        horizComb = getRefineComb(HORIZONTAL, degree);
    }

    for (unsigned char vert = 0; vert < 4; ++vert)
    {
        if (vertices[vert] == NULL)
        {
            // This vertex needs to be allocated in memory. Calling new
            // Vertex() also allocates a Vertex::Replacement, which we need
            // to cope with any extraordinary weights.
            vertices[vert] = new Vertex();
            PatchTreeWalker w(this, vert);
            PrimalPatchTreeLeaf *leaf = NULL;

            bool reset = false;

            // Once vertices[vert] has been allocated, we need to set the
            // remaining 3 pointers (note: it is always 3 -- we only ever
            // introduce regular vertices). This series of steps with
            // PatchTreeWalker finds the three surrounding PrimalPatchTreeLeaf
            // objects and sets the appropriate vertex pointers.
            try
            {
                w.findNeighbour(NORTH);
                leaf = static_cast<PrimalPatchTreeLeaf *>(w.getCurrent());
                leaf->vertices[(w.getRotation() + 1) % 4] = vertices[vert];
            }
            catch (hit_boundary&)
            {
                reset = true;
            }

            try
            {
                if (reset)
                {
                    w.reset(this, vert);
                    w.findNeighbour(EAST);
                    w.findNeighbour(NORTH);
                }
                else
                {
                    w.findNeighbour(EAST);
                }
                leaf = static_cast<PrimalPatchTreeLeaf *>(w.getCurrent());
                leaf->vertices[(w.getRotation() + 2) % 4] = vertices[vert];
            } catch (hit_boundary&) {}

            try
            {
                w.reset(this, vert);
                w.findNeighbour(EAST);
                leaf = static_cast<PrimalPatchTreeLeaf *>(w.getCurrent());
                leaf->vertices[(w.getRotation() + 3) % 4] = vertices[vert];
            } catch (hit_boundary&) {}
        }

        // vertices[vert]->hasReplacement() if either
        //  - we just allocated a vertex in the above if-block, or
        //  - the vertex was allocated while processing another
        //    PrimalPatchTreeLeaf.
        if (vertices[vert]->hasReplacement())
        {
            // In either case, we now need to call receiveContribution on the
            // new vertex with any contributions from within this leaf

            // We need to deal with three cases:
            //  - QuadPatchTree parent with vertices[vert] the new face vertex
            //  - QuadPatchTree parent with vertices[vert] the new edge vertex
            //  - BinaryPatchTree parent with vertices[vert] the new edge
            //    vertex

            // A new vertex is either a 'face vertex', or an 'edge vertex'.
            // This bool keeps track of which.
            bool faceVertex = false;

            // The weight we'll use to call receiveContribution
            VertexPrecision weight = 0;

            if (parent->splits(HORIZONTAL) && parent->splits(VERTICAL))
            {
                // Parent is a QuadPatchTree

                // fVertNum is the value vert needs to be in order for this
                // to be a new face vertex
                unsigned char fVertNum = toIndex(dirFromParent());
                fVertNum += 2;
                fVertNum %= 4;

                if (vert == fVertNum)
                {
                    // vertices[vert] is a new face vertex
                    faceVertex = true;
                }
                // If not a face vertex, it must be an edge vertex. In which
                // case the contribution is either from the vertical knot
                // interval or the horizontal. The table we need to satisfy
                // is:
                //              |     vert
                //      weight  |  0  1  2  3
                //  ------------+----------------
                //            0 |  F  H     V
                //  fVertNum  1 |  H  F  V
                //            2 |     V  F  H
                //            3 |  V     H  F
                //
                // ( where F means face vertex, H means weight=horizComb and
                //   V means weight=vertComb).
                //
                // We can therefore just check if fVertNum + vert == 3
                else if (fVertNum + vert == 3)
                {
                    weight = vertComb;
                }
                else
                {
                    assert((fVertNum + vert) % 4 == 1);
                    weight = horizComb;
                }
            }

            // We need to know where the contribution is coming from as well
            // as what the weight is.
            unsigned char sourceVert = 0;

            if (faceVertex)
            {
                // For a face vertex, that's now easy - the source vertex is
                // directly opposite the current vertex
                sourceVert  = (vert + 2) % 4;

                // And the weight is the product of the two univariate weights
                weight = horizComb * vertComb;
            }
            else
            {
                // If it's an edge vertex, then the source vertex lies on one
                // side of the vertex or the other. So just try one, and change
                // it if we've got the wrong one.
                sourceVert = (vert + 1) % 4;

                // The source vertex is not NULL and doesn't have a replacement
                // (yet -- this will change during smoothing).
                if (vertices[sourceVert] == NULL ||
                    vertices[sourceVert]->hasReplacement())
                {
                    // We got the wrong side -- take the other one.
                    sourceVert = (vert + 3) % 4;
                }

                // All that remains is to find the weight if the parent is a
                // BinaryPatchTree instead of a QuadPatchTree. That just
                // depends on whether the splitType is HORIZONTAL or VERTICAL.
                // We can use PatchTree::splits() to find out.
                if (parent->splits(HORIZONTAL) && !parent->splits(VERTICAL))
                {
                    weight = horizComb;
                }
                else if (!parent->splits(HORIZONTAL) &&
                          parent->splits(VERTICAL))
                {
                    weight = vertComb;
                }

                // Edge vertices will receive a contribution from two leaves
                // on each side of the edge. So we need to halve the weight.
                weight = weight / 2;
            }

            // Finally, we've got the sourceVert and the weight, so we can
            // add the contribution from vertices[sourceVert] to
            // vertices[vert].
            vertices[vert]->receiveContribution(*vertices[sourceVert], weight);
        }
    }
}

void PrimalPatchTreeLeaf::extraordinaryRefine(unsigned char degree,
                                              Direction extNode,
                                              unsigned char valency)
{
    if (isImag())
    {
        return;
    }

    unsigned char extVert = toIndex(extNode);
    VertexPrecision beta  = primalCoeffs[valency - 3][(degree / 2) - 1][BETA];
    VertexPrecision gamma = primalCoeffs[valency - 3][(degree / 2) - 1][GAMMA];

    KnotPrecision vertComb = 0, horizComb = 0;

    if (parent->splits(HORIZONTAL))
    {
        horizComb = getRefineComb(HORIZONTAL, degree);

        vertices[3 - extVert]->receiveContribution(*vertices[extVert],
                                                   horizComb * (beta - 1) / 2);
    }

    if (parent->splits(VERTICAL))
    {
        vertComb  = getRefineComb(VERTICAL, degree);

        unsigned char target = (5 - extVert) % 4;
        vertices[target]->receiveContribution(*vertices[extVert],
                                              vertComb * (beta - 1) / 2);
    }

    if (parent->splits(HORIZONTAL) && parent->splits(VERTICAL))
    {
        vertices[(extVert + 2) % 4]->
            receiveContribution(*vertices[extVert], (gamma - 1) *
                                                    horizComb * vertComb);
    }
}

void PrimalPatchTreeLeaf::smooth(unsigned char degree,
                                 unsigned char stage)
{
    if (isImag())
    {
        return;
    }

    VertexPrecision smoothers[4][4];
    VertexPrecision weight;

    getSmoothCombs(degree, stage, smoothers[0],
                                  smoothers[1],
                                  smoothers[2],
                                  smoothers[3]);

    for (unsigned char source = 0; source < 4; ++source)
    {
        for (unsigned char target = 0; target < 4; ++target)
        {
            weight = smoothers[source][target];

            if (source == target)
            {
                weight /= 4;
            }
            else if ((source + 1) % 4 == target || (target + 1) % 4 == source)
            {
                weight /= 2;
            }

            vertices[target]->receiveContribution(*vertices[source], weight);
        }
    }
}

VertexPrecision PrimalPatchTreeLeaf::extraordinarySmooth(unsigned char degree,
                                                         unsigned char stage,
                                                         Direction extNode,
                                                         unsigned char valency)
{
    if (isImag())
    {
        return 0;
    }

    VertexPrecision alpha = primalCoeffs[valency - 3][(degree / 2) - 1][ALPHA];
    VertexPrecision beta  = primalCoeffs[valency - 3][(degree / 2) - 1][BETA];
    VertexPrecision gamma = primalCoeffs[valency - 3][(degree / 2) - 1][GAMMA];

    if (valency == 3)
    {
        alpha = 1.0;
    }

    VertexPrecision evWeight;
    VertexPrecision smoothers[4];
    unsigned char   eVert = toIndex(extNode);
    unsigned char tgtVert = eVert;

    getSmoothCombs(degree, stage, eVert == 0 ? smoothers : NULL,
                                  eVert == 1 ? smoothers : NULL,
                                  eVert == 2 ? smoothers : NULL,
                                  eVert == 3 ? smoothers : NULL);

    evWeight = alpha * smoothers[eVert] / valency;
    alpha    = (alpha - (valency / 4.0)) / valency;
    beta     = (beta  - 1) / 2;
    gamma    = (gamma - 1);

    vertices[tgtVert]->receiveContribution(*vertices[eVert],
                                           alpha * smoothers[tgtVert]);

    tgtVert += 1;
    tgtVert %= 4;

    vertices[tgtVert]->receiveContribution(*vertices[eVert],
                                           beta  * smoothers[tgtVert]);

    tgtVert += 1;
    tgtVert %= 4;

    vertices[tgtVert]->receiveContribution(*vertices[eVert],
                                           gamma * smoothers[tgtVert]);

    tgtVert += 1;
    tgtVert %= 4;

    vertices[tgtVert]->receiveContribution(*vertices[eVert],
                                           beta  * smoothers[tgtVert]);

    return evWeight;
}

void PrimalPatchTreeLeaf::extraVal3(unsigned char degree,
                                    Direction extNode,
                                    double weightsProd)
{
    if (isImag())
    {
        return;
    }

    VertexPrecision alpha = primalCoeffs[0][(degree / 2) - 1][ALPHA];
    unsigned char   eVert = toIndex(extNode);
    Direction         opp = PatchTreeWalker::rotate(extNode, 2);

    Vertex       *farVert =
        static_cast<PrimalPatchTreeLeaf *>(parent->getChild(opp))
                                                 ->getVertex(toIndex(opp));

    vertices[eVert]->receiveContribution(*farVert,
                                         (1 - weightsProd) * alpha / 3);

    vertices[eVert]->receiveContribution(*vertices[(eVert + 1) % 4],
                                         (1 - weightsProd) * (1 - alpha) / 6);

    vertices[eVert]->receiveContribution(*vertices[(eVert + 3) % 4],
                                         (1 - weightsProd) * (1 - alpha) / 6);

    vertices[eVert]->receiveContribution(*vertices[eVert],
                                         weightsProd / 3);
}

void PrimalPatchTreeLeaf::stream(std::ostream &os, unsigned char level) const
{
    streamDepthMarkers(os, level);
    os << " " << this << " : knotH " << horizontal << ", knotV " << vertical;
    os << ", vertices " << vertices[0] << " " << vertices[1] << " ";
    os << vertices[2] << " " << vertices[3] << endl;
}

KnotPrecision PrimalPatchTreeLeaf::getRefineComb(Orientation orientation,
                                                 unsigned char degree) const
{
    KnotPrecision here, opp, comb = 0;
    PatchTreeWalker walker(parent);

    // Calculate whether this child is on the NORTH or SOUTH side of the
    // parent, for orientation VERTICAL, or whether this child is on the
    // WEST or EAST side of the parent, for orientation HORIZONTAL.
    Direction dir = dirFromParent();

    switch (orientation)
    {
        case VERTICAL:
            dir  = static_cast<Direction>(dir & (NORTH | SOUTH));
            comb = vertical->getInterval();
            break;
        case HORIZONTAL:
            dir  = static_cast<Direction>(dir & (WEST  | EAST ));
            comb = horizontal->getInterval();
            break;
    }

    // Calculate a sum of knot intervals for dir and the opposite direction
    try
    {
        here = walker.knotSum(dir, degree / 2);
        walker.reset(parent);
        opp  = walker.knotSum(PatchTreeWalker::rotate(dir, 2), degree / 2);
        walker.reset(parent);

        // Calculate the corresponding affine combination
        comb += here;
        comb /= here + opp +
                walker.getKnotInterval(orientation)->getInterval();
        comb  = 1 - comb;
    }
    catch (hit_boundary&)
    {
        comb = numeric_limits<KnotPrecision>::quiet_NaN();
    }

    return comb;
}

void PrimalPatchTreeLeaf::calcKnotSums(Direction direction,
                                       unsigned char degree,
                                       unsigned char stage,
                                       KnotPrecision &nearSum,
                                       KnotPrecision &farSum,
                                       KnotPrecision &nearEndSum,
                                       KnotPrecision &farEndSum,
                                       KnotPrecision &furthestEndSum,
                                       bool &near,
                                       bool &far) const
{
    PatchTreeWalker walker(this);
    bool doneFarSum = false;

    try
    {
        nearSum = walker.knotSum(direction, stage - 1);
           near = walker.knotIsNew(direction);

        nearEndSum = nearSum;

        if (near)
        {
            nearEndSum += walker.knotSum(direction, 1);
            farSum = nearEndSum;
            far = walker.knotIsNew(direction);
            doneFarSum = true;
            walker.stepToParent();
            nearEndSum += walker.knotSum(direction, ((degree - 1) / 2)
                                                     - stage);
        }
        else if (((degree - 1) / 2) - stage > 0)
        {
            nearEndSum += walker.knotSum(direction, 1);
            farSum = nearEndSum;
            far = walker.knotIsNew(direction);
            if (far)
            {
                nearEndSum += walker.knotSum(direction, 1);
            }
            doneFarSum = true;
            walker.stepToParent();
            nearEndSum += walker.knotSum(direction, ((degree - 3) / 2)
                                                     - stage);
        }
    }
    catch (hit_boundary&)
    {
               nearSum = numeric_limits<KnotPrecision>::quiet_NaN();
            nearEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
                farSum = numeric_limits<KnotPrecision>::quiet_NaN();
             farEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
        furthestEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
        return;
    }

    try
    {
        if (doneFarSum)
        {
            farEndSum = nearEndSum;

            if (!near)
            {
                farEndSum += walker.knotSum(direction, 1);
            }
        }
        else
        {
            farSum = nearSum + walker.knotSum(direction, 1);
            far = walker.knotIsNew(direction);
            farEndSum = farSum;
            if (far)
            {
                farEndSum += walker.knotSum(direction, 1);
            }
            walker.stepToParent();
            farEndSum += walker.knotSum(direction, ((degree - 1) / 2) - stage);
        }
    }
    catch (hit_boundary&)
    {
                farSum = numeric_limits<KnotPrecision>::quiet_NaN();
             farEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
        furthestEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
        return;
    }

    try
    {
        furthestEndSum = farEndSum + walker.knotSum(direction, 1);
    }
    catch (hit_boundary&)
    {
        furthestEndSum = numeric_limits<KnotPrecision>::quiet_NaN();
    }
}

void PrimalPatchTreeLeaf::getCombsForOrientation(Orientation orientation,
                                                 unsigned char degree,
                                                 unsigned char stage,
                                                 VertexPrecision &LL,
                                                 VertexPrecision &RL,
                                                 VertexPrecision &RR,
                                                 VertexPrecision &LR) const
{
    KnotPrecision thisInterval = getKnotInterval(orientation)->getInterval();
    Direction leftDir = ANY, rightDir = ANY;

    switch (orientation)
    {
        case HORIZONTAL:
            leftDir  = WEST;
            rightDir = EAST;
            break;
        case VERTICAL:
            leftDir  = NORTH;
            rightDir = SOUTH;
            break;
    }

    KnotPrecision leftNearSum, leftFarSum;
    KnotPrecision leftNearEndSum, leftFarEndSum, leftFurthestEndSum;
    bool leftFar, leftNear;

    calcKnotSums(leftDir, degree, stage,
                 leftNearSum,    leftFarSum,
                 leftNearEndSum, leftFarEndSum, leftFurthestEndSum,
                 leftNear,       leftFar);

    KnotPrecision rightNearSum, rightFarSum;
    KnotPrecision rightNearEndSum, rightFarEndSum, rightFurthestEndSum;
    bool rightFar, rightNear;

    calcKnotSums(rightDir, degree, stage,
                 rightNearSum,    rightFarSum,
                 rightNearEndSum, rightFarEndSum, rightFurthestEndSum,
                 rightNear,       rightFar);

    if (leftFar && rightNear)
    {
        LL = (leftFarSum + thisInterval + rightNearSum) /
             (leftFarEndSum + thisInterval + rightNearEndSum);

        RL = (leftFarEndSum - leftFarSum) /
             (leftFarEndSum + thisInterval + rightNearEndSum);
    }
    else if (leftFar)
    {
        RL = (leftFarEndSum - leftFarSum) /
             (leftFarEndSum + thisInterval + rightFarEndSum);

        LL = 1 - RL;
    }
    else if (rightNear)
    {
        LL = (leftFurthestEndSum + thisInterval + rightNearSum) /
             (leftFurthestEndSum + thisInterval + rightNearEndSum);
        RL = 0;
    }
    else
    {
        LL = 1;
        RL = 0;
    }

    if (leftNear && rightFar)
    {
        RR = (leftNearSum + thisInterval + rightFarSum) /
             (leftNearEndSum + thisInterval + rightFarEndSum);

        LR = (rightFarEndSum - rightFarSum) /
             (leftNearEndSum + thisInterval + rightFarEndSum);
    }
    else if (leftNear)
    {
        RR = (leftNearSum + thisInterval + rightFurthestEndSum) /
             (leftNearEndSum + thisInterval + rightFurthestEndSum);
        LR = 0;
    }
    else if (rightFar)
    {
        LR = (rightFarEndSum - rightFarSum) /
             (leftFarEndSum + thisInterval + rightFarEndSum);
        RR = 1 - LR;
    }
    else
    {
        RR = 1;
        LR = 0;
    }
}

void PrimalPatchTreeLeaf::getSmoothCombs(unsigned char degree,
                                         unsigned char stage,
                                         VertexPrecision *fromNE,
                                         VertexPrecision *fromSE,
                                         VertexPrecision *fromSW,
                                         VertexPrecision *fromNW) const
{
    VertexPrecision NN, SN, SS, NS;
    getCombsForOrientation(  VERTICAL, degree, stage, NN, SN, SS, NS);

    VertexPrecision WW, EW, EE, WE;
    getCombsForOrientation(HORIZONTAL, degree, stage, WW, EW, EE, WE);

    if (fromNE)
    {
        fromNE[0] = NN * EE;
        if (NN == 0 || EE == 0) fromNE[0] = 0;
        fromNE[1] = NS * EE;
        if (NS == 0 || EE == 0) fromNE[1] = 0;
        fromNE[2] = NS * EW;
        if (NS == 0 || EW == 0) fromNE[2] = 0;
        fromNE[3] = NN * EW;
        if (NN == 0 || EW == 0) fromNE[3] = 0;
    }

    if (fromSE)
    {
        fromSE[0] = SN * EE;
        if (SN == 0 || EE == 0) fromSE[0] = 0;
        fromSE[1] = SS * EE;
        if (SS == 0 || EE == 0) fromSE[1] = 0;
        fromSE[2] = SS * EW;
        if (SS == 0 || EW == 0) fromSE[2] = 0;
        fromSE[3] = SN * EW;
        if (SN == 0 || EW == 0) fromSE[3] = 0;
    }

    if (fromSW)
    {
        fromSW[0] = SN * WE;
        if (SN == 0 || WE == 0) fromSW[0] = 0;
        fromSW[1] = SS * WE;
        if (SS == 0 || WE == 0) fromSW[1] = 0;
        fromSW[2] = SS * WW;
        if (SS == 0 || WW == 0) fromSW[2] = 0;
        fromSW[3] = SN * WW;
        if (SN == 0 || WW == 0) fromSW[3] = 0;
    }

    if (fromNW)
    {
        fromNW[0] = NN * WE;
        if (NN == 0 || WE == 0) fromNW[0] = 0;
        fromNW[1] = NS * WE;
        if (NS == 0 || WE == 0) fromNW[1] = 0;
        fromNW[2] = NS * WW;
        if (NS == 0 || WW == 0) fromNW[2] = 0;
        fromNW[3] = NN * WW;
        if (NN == 0 || WW == 0) fromNW[3] = 0;
    }
}

}
