/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <cassert>
#include "flatmeshhandler.h"
#include "knotinterval.h"
#include "mesh.h"
#include "patchtreeroot.h"
#include "patchtreewalker.h"
#include "primalmeshflattener.h"
#include "primalpatchtreeleaf.h"
#include "vertex.h"

namespace snurbs {

using namespace std;
using namespace tr1::placeholders;

PrimalMeshFlattener::PrimalMeshFlattener(const Mesh &mesh) :
        MeshFlattener(mesh)
{
}

KnotInterval *PrimalMeshFlattener::getKnotIntervalForEdge(unsigned int face,
                                                          unsigned int edge)
{
    assert(edge < 4);
    Mesh::PatchIterator end(mesh.endPatches());

    for (Mesh::PatchIterator iter = mesh.beginPatches(); iter != end; ++iter)
    {
        PrimalPatchTreeLeaf *leaf = static_cast<PrimalPatchTreeLeaf *>(*iter);

        if (leaf->isReal())
        {
            if (!face)
            {
                if (edge % 2)
                {
                    return leaf->getKnotInterval(PatchTree::HORIZONTAL);
                }
                else
                {
                    return leaf->getKnotInterval(PatchTree::VERTICAL);
                }
            }

            --face;
        }
    }

    return NULL;
}

void PrimalMeshFlattener::flattenFaces(void)
{
    tr1::array<VertexPrecision, 3>        normal = {{0}};
    tr1::array<VertexPrecision, 3> averageNormal = {{0}};
    Mesh::PatchIterator end(mesh.endPatches());

    for (Mesh::PatchIterator iter = mesh.beginPatches(); iter != end; ++iter)
    {
        PrimalPatchTreeLeaf *leaf = static_cast<PrimalPatchTreeLeaf *>(*iter);
        PatchTreeWalker walker(*iter);

        if (!compatible || leaf->isReal())
        {
            KnotPrecision left = 0, top = 0;

            if (flatMeshHandler != NULL)
            {
                flatMeshHandler->startFace(4);
            }

            KnotInterval *horizKnot, *vertKnot;

            horizKnot = leaf->getKnotInterval(PatchTree::HORIZONTAL);
            vertKnot  = leaf->getKnotInterval(PatchTree::VERTICAL);

            if (horizKnot == notifyKnot || vertKnot == notifyKnot)
            {
                startNotify();
            }

            // Vertices 3 and 0 would work just as well here.
            knotIntervalMap.insert(
                make_pair(horizKnot,
                    make_pair(leaf->getVertex(1), leaf->getVertex(2))));

            // Vertices 2 and 3 would work just as well here.
            knotIntervalMap.insert(
                make_pair(vertKnot,
                    make_pair(leaf->getVertex(0), leaf->getVertex(1))));

            if (flatMeshHandler != NULL || coordCallback != NULL)
            {
                KnotInterval *horiz = horizKnot, *vert = vertKnot;

                while (horiz->getParent() != NULL)
                {
                    KnotInterval *newParent = horiz->getParent();

                    if (newParent->getRightChild() == horiz)
                    {
                        left += newParent->getLeftChild()->getInterval();
                    }

                    horiz = newParent;
                }

                while (vert->getParent() != NULL)
                {
                    KnotInterval *newParent = vert->getParent();

                    if (newParent->getRightChild() == vert)
                    {
                        top += newParent->getLeftChild()->getInterval();
                    }

                    vert = newParent;
                }
            }

            if (flatNormalCallback != NULL)
            {
                leaf->getNormal(normal);
                flatNormalCallback(normal[0], normal[1], normal[2]);
            }

            for (unsigned char vertex = 0; vertex < 4; ++vertex)
            {
                KnotPrecision curLeft = 0, curTop = 0;

                switch (vertex)
                {
                    case 0:
                        curLeft = left + horizKnot->getInterval();
                        curTop  = top;
                        break;
                    case 1:
                        curLeft = left + horizKnot->getInterval();
                        curTop  = top  +  vertKnot->getInterval();
                        break;
                    case 2:
                        curLeft = left;
                        curTop  = top  + vertKnot->getInterval();
                        break;
                    case 3:
                        curLeft = left;
                        curTop  = top;
                        break;
                }

                if (flatMeshHandler != NULL)
                {
                    flatMeshHandler->addToFace(
                        vertexIndex(leaf->getVertex(vertex)));
                    flatMeshHandler->addTexCoords(curLeft, curTop);
                }

                walker.reset(*iter, vertex);
                averageNormal[0] = 0;
                averageNormal[1] = 0;
                averageNormal[2] = 0;

                if (smoothNormalCallback != NULL)
                {
                    do
                    {
                        walker.findNeighbour(PatchTree::NORTH);
                        walker.reset(walker.getCurrent(),
                                    (walker.getRotation() + 1) % 4);

                        const PrimalPatchTreeLeaf *current =
                            static_cast<const PrimalPatchTreeLeaf*>
                                (walker.getCurrent());
                        if (!current->isReal())
                        {
                            continue;
                        }
                        current->getNormal(normal);

                        averageNormal[0] += normal[0];
                        averageNormal[1] += normal[1];
                        averageNormal[2] += normal[2];
                    } while (walker.getCurrent() != *iter);

                    smoothNormalCallback(averageNormal[0],
                                         averageNormal[1],
                                         averageNormal[2]);
                }

                if (coordCallback != NULL)
                {
                    coordCallback(curLeft, curTop);
                }

                if (vertexCallback != NULL)
                {
                    vertexCallback(leaf->getVertex(vertex)->getX(),
                                   leaf->getVertex(vertex)->getY(),
                                   leaf->getVertex(vertex)->getZ(),
                                   leaf->getVertex(vertex)->getW());
                }
            }

            if (flatMeshHandler != NULL)
            {
                flatMeshHandler->closeFace();
            }

            if (horizKnot == notifyKnot || vertKnot == notifyKnot)
            {
                endNotify();
            }
        }
    }
}

}
