/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <boost/lambda/lambda.hpp>
#include <cassert>
#include <stdexcept>
#include <tr1/functional>
#include "primalmeshbuilder.h"
#include "mesh.h"
#include "patchtreeroot.h"
#include "patchnode.h"
#include "primalpatchtreeleaf.h"
#include "vertex.h"
#include "halfedge.h"
#include "knotinterval.h"

namespace snurbs {

using namespace boost;
using namespace std;
using namespace tr1::placeholders;

void PrimalMeshBuilder::addVertex(VertexPrecision x,
                                  VertexPrecision y,
                                  VertexPrecision z,
                                  VertexPrecision w)
{
    vertices.push_back(new Vertex(x, y, z, w));
    mesh.patchNodes.push_back(new PatchNode());
}

void PrimalMeshBuilder::startFace(unsigned int /* numEdges */)
{
    currentFace      = new PrimalPatchTreeLeaf();
    currentPatch     = NULL;
    currentVertIndex = 0;
}

void PrimalMeshBuilder::addToFace(unsigned int vertNum)
{
    if (currentVertIndex > 3)
    {
        throw runtime_error("Too many edges: for primal subdivision, "
                            "every face must have four edges");
    }

    if (vertNum >= vertices.size())
    {
        unsigned int to_insert = vertNum + 1 - vertices.size();

        while (to_insert)
        {
            vertices.push_back(new Vertex(0, 0, 0, 0));
            vertices.back()->setNotReal();
            mesh.patchNodes.push_back(new PatchNode());
            --to_insert;
        }
    }

    currentFace->vertices[currentVertIndex] = vertices[vertNum];
    currentVertIndex++;

    mesh.halfEdges.push_back(new HalfEdge(mesh.patchNodes[vertNum],
                                          currentPatch));

    if (currentVertIndex == 1)
    {
        // To begin with, we assign the northern HalfEdge for currentPatch
        // to just be the first edge specified. Later, we'll adjust this
        // assignment in order to match the orientation of KnotInterval
        // objects (see finishKnotIntervals(), below).
        currentPatch = new PatchTreeRoot(currentFace, mesh.halfEdges.back());
        mesh.patches.push_back(currentPatch);
        knotFlipMap.insert(make_pair(currentPatch, make_pair(false, false)));
        currentFace->parent = currentPatch;
        mesh.halfEdges.back()->patch = currentPatch;
    }
    else
    {
        assert(currentVertIndex > 1);
        // If we've added two HalfEdge objects for this face, we can set the
        // 'next' pointer of the HalfEdge just before the one we just added.
        (*(mesh.halfEdges.end() - 2))->next = mesh.halfEdges.back();
    }
}

void PrimalMeshBuilder::closeFace(void)
{
    if (currentVertIndex < 4)
    {
        throw runtime_error("Too few edges: for primal subdivision, "
                            "every face must have four edges");
    }

    // Close the loop of next pointers.
    mesh.halfEdges.back()->next = (*(mesh.halfEdges.end() - 4));
}

void PrimalMeshBuilder::finishFaces(void)
{
    // Make sure every HalfEdge has a pair.
    pairMap.clear();

    // Here we call findPair(edge) for every edge in mesh.halfEdges
    for_each(mesh.halfEdges.begin(), mesh.halfEdges.end(),
             tr1::bind(&PrimalMeshBuilder::findPair, this, _1));

    // All the remaining items in the map represent boundary edges
    for_each(pairMap.begin(), pairMap.end(),
             tr1::bind(&PrimalMeshBuilder::addBoundary, this, _1));

    // Set next pointers for boundary edges
    for_each(pairMap.begin(), pairMap.end(), &PrimalMeshBuilder::linkBoundary);

    // Make sure every PatchNode in mesh.patchNodes has an assigned HalfEdge.
    // We just assign HalfEdge pointers indiscriminantly; it doesn't matter if
    // we overwrite an existing assignment.
    for_each(mesh.halfEdges.begin(),
             mesh.halfEdges.end(),
             ((lambda::_1 ->* &HalfEdge::node) ->* &PatchNode::edge) =
               lambda::_1 ->* &HalfEdge::pair);

    // Any vertices which are too close to the boundary should be deleted and
    // replaced with NULL pointers.
    for_each(pairMap.begin(), pairMap.end(),
             tr1::bind(&PrimalMeshBuilder::cullBoundary, this, _1));
}

void PrimalMeshBuilder::addKnotInterval(unsigned int vertex1,
                                        unsigned int vertex2,
                                        KnotPrecision value)
{
    HalfEdge *startEdge = mesh.patchNodes[vertex1]->
                              getEdgeTo(mesh.patchNodes[vertex2]);

    addKnotInterval(startEdge, value);
}

void PrimalMeshBuilder::finishKnotIntervals(void)
{
    // Make sure every PatchTreeRoot has two KnotIntervals.
    for_each(mesh.patches.begin(), mesh.patches.end(),
             tr1::bind(&PrimalMeshBuilder::addMissingIntervals,
                       this, _1));

    // Use knotFlipMap to get all the orientations correct for every
    // PatchTreeRoot
    KnotFlipMap::iterator iter;

    while(!knotFlipMap.empty())
    {
        iter = knotFlipMap.begin();
        PrimalPatchTreeLeaf *leaf =
            dynamic_cast<PrimalPatchTreeLeaf *>(iter->first->child);

        if (iter->second.first && !iter->second.second)
        {
            // horizontal is flipped
            iter->first->edge = iter->first->edge->next;

            Vertex *tempVert  = leaf->vertices[0];
            leaf->vertices[0] = leaf->vertices[1];
            leaf->vertices[1] = leaf->vertices[2];
            leaf->vertices[2] = leaf->vertices[3];
            leaf->vertices[3] = tempVert;

            KnotInterval *tempKnot = leaf->vertical;
            leaf->vertical         = leaf->horizontal;
            leaf->horizontal       = tempKnot;
        }
        else if (iter->second.first && iter->second.second)
        {
            // vertical and horizontal both flipped
            iter->first->edge = iter->first->edge->next->next;

            Vertex *tempVert  = leaf->vertices[0];
            leaf->vertices[0] = leaf->vertices[2];
            leaf->vertices[2] = tempVert;

                    tempVert  = leaf->vertices[1];
            leaf->vertices[1] = leaf->vertices[3];
            leaf->vertices[3] = tempVert;
        }
        else if (!iter->second.first && iter->second.second)
        {
            // vertical is flipped
            iter->first->edge = iter->first->edge->next->next->next;

            Vertex *tempVert  = leaf->vertices[3];
            leaf->vertices[3] = leaf->vertices[2];
            leaf->vertices[2] = leaf->vertices[1];
            leaf->vertices[1] = leaf->vertices[0];
            leaf->vertices[0] = tempVert;

            KnotInterval *tempKnot = leaf->vertical;
            leaf->vertical         = leaf->horizontal;
            leaf->horizontal       = tempKnot;
        }

        knotFlipMap.erase(iter);
    }

    for_each(mesh.patchNodes.begin(), mesh.patchNodes.end(),
             tr1::bind(&PatchNode::setUpSpokeSets, _1));
}

void PrimalMeshBuilder::findPair(HalfEdge *edge)
{
    // Identify the PatchNode objects this HalfEdge lies between
    PatchNode *end   = edge->node;
    PatchNode *start = edge->prev()->node;

    // Try and find a pair for this edge.
    PairMap::iterator iter = pairMap.find(PairMapKey(end, start));

    if (iter == pairMap.end())
    {
        // We haven't encountered the pair yet, so just insert this edge into
        // pairMap
        pairMap.insert(PairMap::value_type(PairMapKey(start, end), edge));
    }
    else
    {
        // We've found the pair for this edge.
        edge->pair = iter->second;
        iter->second->pair = edge;

        // Remove this item from the map, so that when it finishes, it
        // contains only the pairs of boundary HalfEdge objects.
        pairMap.erase(iter);
    }
}

void PrimalMeshBuilder::addBoundary(const PairMap::value_type &unmatched)
{
    mesh.halfEdges.push_back(new HalfEdge(unmatched.first.first, NULL));
    mesh.halfEdges.back()->pair = unmatched.second;
    unmatched.second->pair = mesh.halfEdges.back();
}

void PrimalMeshBuilder::linkBoundary(const PairMap::value_type &unmatched)
{
    // We assume that each vertex is a part of at most one boundary. Every
    // manifold surface should satisfy this condition. Then we can assign next
    // pointers on the boundary using:
    HalfEdge *next = unmatched.second;
    while (next->patch != NULL)
    {
        next = next->prev()->pair;
    }
    unmatched.second->pair->next = next;
}

void PrimalMeshBuilder::cullVertex(const PatchTreeWalker &walker)
{
    PrimalPatchTreeLeaf *leaf =
        static_cast<PrimalPatchTreeLeaf *>(walker.getCurrent());

    leaf->vertices[(3 + walker.getRotation()) % 4]->setNotReal();
}

void PrimalMeshBuilder::cullBoundary(const PairMap::value_type &unmatched)
{
    HalfEdge::PatchStripIterator iter(unmatched.second->pair);
    HalfEdge::PatchStripIterator  end(unmatched.second);

    for (unsigned i = 0; i < mesh.degree / 2; ++i)
    {
        PatchNode::SurroundingLeafIterator leafIter((*iter)->pair->node->edge);
        for_each_circular(leafIter,
                          tr1::bind(&PrimalMeshBuilder::cullVertex, this, _1));

        leafIter = PatchNode::SurroundingLeafIterator((*iter)->node->edge);
        for_each_circular(leafIter,
                          tr1::bind(&PrimalMeshBuilder::cullVertex, this, _1));

        ++iter;
        // Check if we've hit another boundary (the only circumstance that
        // iter == end) or looped back to the beginning.
        if (iter == end || *iter == unmatched.second->pair)
        {
            break;
        }
    }
}

void PrimalMeshBuilder::linkKnotStrip(HalfEdge*     const edge,
                                      KnotInterval* const knotInterval,
                                      bool                defaultOrientation)
{
    HalfEdge *patchEdge = edge->patch->edge;
    unsigned char steps = patchEdge->stepsTo(edge);

    assert(steps < 4);

    PrimalPatchTreeLeaf *leaf =
        static_cast<PrimalPatchTreeLeaf *>(edge->patch->child);

    const string doubleError = "Doubly-defined knot interval";

    KnotFlipMap::iterator iter = knotFlipMap.find(edge->patch);

    if (steps % 2 == 0)
    {
        if (steps == 2)
        {
            iter->second.first = !defaultOrientation;
        }
        else
        {
            iter->second.first =  defaultOrientation;
        }
        if (leaf->horizontal != NULL)
        {
            throw runtime_error(doubleError);
        }
        leaf->horizontal = knotInterval;
    }
    else
    {
        if (steps == 3)
        {
            iter->second.second = !defaultOrientation;
        }
        else
        {
            iter->second.second =  defaultOrientation;
        }
        if (leaf->vertical != NULL)
        {
            throw runtime_error(doubleError);
        }
        leaf->vertical   = knotInterval;
    }
}

void PrimalMeshBuilder::addMissingIntervals(const PatchTreeRoot* const patch)
{
    PrimalPatchTreeLeaf *leaf =
            static_cast<PrimalPatchTreeLeaf *>(patch->child);

    if (leaf->horizontal == NULL)
    {
        addKnotInterval(patch->edge, defaultKnotSpacing);
    }

    if (leaf->vertical == NULL)
    {
        addKnotInterval(patch->edge->next, defaultKnotSpacing);
    }
}

void PrimalMeshBuilder::addKnotInterval(HalfEdge* startEdge,
                                        KnotPrecision value)
{
    mesh.knots.push_back(new KnotInterval(value));

    if (startEdge->patch == NULL)
    {
        startEdge = startEdge->pair->next->next;
    }
    assert(startEdge->patch != NULL);

    HalfEdge::PatchStripIterator iter(startEdge);

    iter = for_each_circular(iter, tr1::bind(&PrimalMeshBuilder::linkKnotStrip,
                                             this,
                                             _1, mesh.knots.back(), false));

    // If iter == startEdge, then the existing call to forEach already looped
    // the iterator right back to the beginning, and so there's no need to
    // consider the other side. If not, we might need another iterator and
    // another call to forEach.
    if (*iter != startEdge)
    {
        HalfEdge *oppositeDir = startEdge->next->next->pair->next->next;

        // If startEdge was originally part of a boundary, then oppositeDir is
        // also part of a boundary now. In which case we don't need to
        // consider the opposite direction. Otherwise, we do.
        if (oppositeDir->patch != NULL)
        {
            for_each_circular(HalfEdge::PatchStripIterator(oppositeDir),
                              tr1::bind(&PrimalMeshBuilder::linkKnotStrip,
                                        this, _1, mesh.knots.back(), true));
        }
    }
}

}
