/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PATCHTREEWALKER_H
#define PATCHTREEWALKER_H

#include "patchtree.h"
#include "types.h"

namespace snurbs {

class KnotInterval;
class PatchTreeRoot;

/**
 * Used to navigate a `forest' of PatchTree objects, presenting a interface
 * to a PatchTree object which takes into account rotation where necessary.
 *
 * @ingroup Support
 */
class PatchTreeWalker
{
public:
    /** PatchTreeWalker constructor. */
    PatchTreeWalker(const PatchTree *start, unsigned char rotateSteps = 0);

    /**
     * Represents a linked list of PatchTree::Direction enumerators. We have
     * our own implementation instead of using @c std::list , for example, so
     * that we can use pointer loops to succintly express a path with a
     * repeated PatchTree::Direction (or even a repeated pattern).
     *
     * @ingroup Support
     */
    class Path
    {
    public:
        /** Path constructor. */
        Path(PatchTree::Direction dir, const Path *next) :  dir(dir),
                                                           next(next) {};

        /**
         * @name Getter functions
         * @{
         */
        const Path          *getNext(void) const;
        PatchTree::Direction getDir (void) const;
        /** @} */

    private:
        /** The head of a linked list of directions for this Path. */
        PatchTree::Direction dir;
        /** The tail of the linked list: the remaining directions. */
        const Path          *next;
    };

    /**
     * Starting from #currentPatchTree, follow a Path. Stop if either
     * - the Path stops (the Path::next pointer is NULL), or
     * - we hit a leaf node.
     * @param path The Path to follow.
     */
    void followPath(const Path &path);

    /**
     * Returns the knot interval (horizontal or vertical, depending on the
     * parameter @c orientation ) associated with #currentPatchTree when
     * rotated by #rotation.
     */
    KnotInterval *getKnotInterval(PatchTree::Orientation orientation) const;

    /**
     * Given one of the compass directions PatchTree::NORTH, PatchTree::SOUTH,
     * PatchTree::WEST or PatchTree::EAST, returns @c true if the knot on the
     * given edge was inserted in the current subdivision step.
     */
    bool knotIsNew(PatchTree::Direction dir) const;

    /**
     * Returns whether the knot interval (horizontal or vertical, depending on
     * the parameter @c orientation ) associated with #currentPatchTree, when
     * rotated by #rotation, should be treated as flipped.
     */
    bool intervalIsFlipped(PatchTree::Orientation orientation) const;

    /**
     * Given one of the compass directions PatchTree::NORTH, PatchTree::SOUTH,
     * PatchTree::WEST or PatchTree::EAST, attempts to walk over the forest of
     * PatchTree objects to set #currentPatchTree to the neighbour in the given
     * direction.
     */
    void findNeighbour(PatchTree::Direction dir, const Path *path = NULL);

    /**
     * Attempts to call findNeighbour(dir) @c steps times, summing up the
     * KnotInterval objects parallel to @c dir on the way. Does not include
     * the #currentPatchTree in the sum.
     * @param dir One of the compass directions PatchTree::NORTH,
     * PatchTree::SOUTH, PatchTree::WEST or PatchTree::EAST
     * @param steps The number of steps to take in PatchTree::Direction @c dir
     */
    KnotPrecision knotSum(PatchTree::Direction dir, unsigned char steps);

    /** Getter for #currentPatchTree. */
    PatchTree *getCurrent(void) const;

    /** Getter for #rotation. */
    unsigned char getRotation(void) const;

    /** Resets a PatchTreeWalker to the given PatchTree*, and zero rotation. */
    void reset(const PatchTree *resetTree, unsigned char resetRot = 0);

    /**
     * Move this PatchTreeWalker to point to the parent of the
     * #currentPatchTree
     */
    void stepToParent(void);

    /**
     * @return @c true if the #currentPatchTree splits @c orientation when
     * rotated by #rotation
     */
    bool splitsOrientation(PatchTree::Orientation orientation) const;

    /**
     * @return the equivalent direction to @c dir when rotated @c steps steps
     */
    static PatchTree::Direction rotate(PatchTree::Direction dir,
                                       unsigned char steps);

    /**
     * @return the orientation corresponding to @c d
     */
    static PatchTree::Orientation getOrientationForDir(PatchTree::Direction d);

private:
    /** The current position of the PatchTreeWalker. */
    const PatchTree *currentPatchTree;

    /**
     * The number of steps (0 @f$\le@f$ @c rotation @f$\le@f$ 3) that the
     * #currentPatchTree is considered rotated from its actual orientation.
     */
    unsigned char rotation;

    /**
     * @return the equivalent direction to @c dir when rotated #rotation steps
     */
    PatchTree::Direction rotate(PatchTree::Direction dir);
};

inline const PatchTreeWalker::Path *PatchTreeWalker::Path::getNext(void) const
{
    return next;
}

inline PatchTree::Direction PatchTreeWalker::Path::getDir (void) const
{
    return dir;
}

inline PatchTree *PatchTreeWalker::getCurrent(void) const
{
    return const_cast<PatchTree *>(currentPatchTree);
}

inline unsigned char PatchTreeWalker::getRotation(void) const
{
    return rotation;
}

inline void PatchTreeWalker::reset(const PatchTree *resetTree,
                                   unsigned char resetRot)
{
    currentPatchTree = resetTree;
    rotation = resetRot;
}

inline void PatchTreeWalker::stepToParent(void)
{
    currentPatchTree = currentPatchTree->getParent();
}

}

#endif
