/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <cassert>
#include <limits>
#include <tr1/functional>
#include "patchnode.h"
#include "patchtreeroot.h"
#include "patchtreewalker.h"
#include "primalpatchtreeleaf.h"
#include "halfedge.h"
#include "knotinterval.h"

namespace snurbs {

using namespace boost;
using namespace std;
using namespace tr1::placeholders;

PatchNode::PatchNode(void) : edge(NULL),
                             spokeSetHead(NULL),
                             spokeSetNext(NULL),
                             minInterval(-1)
{
}

HalfEdge *PatchNode::getEdgeTo(const PatchNode* const node) const
{
    HalfEdge::NodeIterator iter =
            find_circular(HalfEdge::NodeIterator(edge),
                          node == lambda::bind(&HalfEdge::getNode,
                                               lambda::_1));

    return *iter;
}

unsigned char PatchNode::valency(void) const
{
    return count_if_circular(HalfEdge::NodeIterator(edge),
                             lambda::constant(true));
}

bool PatchNode::onBoundary(void) const
{
    PatchTreeRoot *nullPatch = NULL;

    try
    {
        find_circular(HalfEdge::NodeIterator(edge),
                      nullPatch ==
                      lambda::bind(&HalfEdge::getPatch, lambda::_1));
        return true;
    }
    catch (runtime_error&)
    {
        return false;
    }
}

bool PatchNode::extraordinary(void) const
{
    unsigned char v = valency();

    // A node is extraordinary
    // - if valency > 4
    // - or if valency == 4 and it's on the boundary
    // - or if valency < 4 and it's not on the boundary
    // The XOR in the second part of this expression takes care of the second
    // and third cases
    return (v > 4 || (onBoundary() ^ (v < 4)));
}

void PatchNode::requestNewKnots(void)
{
    if (extraordinary())
    {
        SurroundingLeafIterator iter(edge);

        // Request new knots to be inserted at half spokeSetHead->minInterval
        // away (to create a locally uniform knot distribution)
        if (spokeSetHead->minInterval > 0)
        {
            SurroundingLeafIterator end(iter);
            do
            {
                KnotInterval *knot =
                    (*iter).getKnotInterval(PatchTree::HORIZONTAL);
                KnotPrecision subdivPos = spokeSetHead->minInterval / 2.0;

                if (subdivPos < knot->getInterval())
                {
                    if ((*iter).intervalIsFlipped(PatchTree::HORIZONTAL))
                    {
                        knot->insertionRequest(knot->getInterval() -
                                               subdivPos);
                    }
                    else
                    {
                        knot->insertionRequest(subdivPos);
                    }
                }

                ++iter;
            } while (iter != end);
        }
    }
}

void PatchNode::findSpokeSetMin(void)
{
    if (extraordinary())
    {
        SurroundingLeafIterator iter(edge);

        KnotPrecision minInt = numeric_limits<KnotPrecision>::infinity();

        SurroundingLeafIterator end(iter);
        do
        {
            KnotInterval *knot =
                (*iter).getKnotInterval(PatchTree::HORIZONTAL);

            if (knot->getInterval() < minInt)
            {
                minInt = knot->getInterval();
            }
            ++iter;
        } while (iter != end);

        if (minInt > 0 && (spokeSetHead->minInterval < 0 ||
                           spokeSetHead->minInterval > minInt))
        {
            spokeSetHead->minInterval = minInt;
        }
    }
}

void PatchNode::removeRequests(void)
{
    if (extraordinary() && spokeSetHead->minInterval > 0)
    {
        SurroundingLeafIterator iter(edge);

        for_each_circular(iter,
                          tr1::bind(&KnotInterval::clearRequests,
                              tr1::bind(&PatchTreeWalker::getKnotInterval,
                                        _1, PatchTree::HORIZONTAL)
                                   )
                         );
    }
}

void PatchNode::extraordinaryRefine(unsigned char degree)
{
    cumProdWeights = 1;

    if (extraordinary())
    {
        SurroundingLeafIterator iter(edge);
        SurroundingLeafIterator end(iter);

        do
        {
            assert((*iter).getCurrent()->isLeaf());
            PatchTreeLeaf *leaf =
                static_cast<PatchTreeLeaf *>((*iter).getCurrent());
            leaf->extraordinaryRefine(
                    degree,
                    PatchTree::toDir((3 + (*iter).getRotation()) % 4),
                    valency());
            ++iter;
        } while (iter != end);
    }
}

void PatchNode::extraordinarySmooth(unsigned char degree, unsigned char stage)
{
    if (extraordinary())
    {
        evStageWeight = 0;
        SurroundingLeafIterator iter(edge);
        SurroundingLeafIterator end(iter);

        do
        {
            assert((*iter).getCurrent()->isLeaf());
            PatchTreeLeaf *leaf =
                static_cast<PatchTreeLeaf *>((*iter).getCurrent());
            evStageWeight += leaf->extraordinarySmooth(
                degree,
                stage,
                PatchTree::toDir((3 + (*iter).getRotation()) % 4),
                valency());
            ++iter;
        } while (iter != end);

        PrimalPatchTreeLeaf *leaf =
                static_cast<PrimalPatchTreeLeaf *>((*iter).getCurrent());
        evStageWeight /= leaf->getVertex((3 + (*iter).getRotation()) %
                                         4)->getDenominator();
        cumProdWeights *= evStageWeight;
    }
}

void PatchNode::valency3Adjust(unsigned char degree)
{
    if (valency() == 3 && !onBoundary())
    {
        SurroundingLeafIterator iter(edge);
        SurroundingLeafIterator end(iter);

        do
        {
            assert((*iter).getCurrent()->isLeaf());
            PrimalPatchTreeLeaf *leaf =
                static_cast<PrimalPatchTreeLeaf *>((*iter).getCurrent());
            leaf->extraVal3(degree,
                            PatchTree::toDir((3 + (*iter).getRotation()) % 4),
                            cumProdWeights);
            ++iter;
        } while (iter != end);
    }
}

void PatchNode::setUpSpokeSets(void)
{
    if (extraordinary())
    {
        spokeSetHead = this;
        SurroundingLeafIterator iter(edge);
        SurroundingLeafIterator end(iter);

        do
        {
            const PatchTreeWalker &walker = *iter;
            KnotInterval *knot = walker.getKnotInterval(PatchTree::HORIZONTAL);
            PatchNode *extNode = NULL;

            if (walker.intervalIsFlipped(PatchTree::HORIZONTAL))
            {
                extNode = knot->getRightExtNode();
                if (extNode == NULL)
                {
                    knot->setRightExtNode(this);
                }
            }
            else
            {
                extNode = knot->getLeftExtNode();
                if (extNode == NULL)
                {
                    knot->setLeftExtNode(this);
                }
            }

            if (extNode != NULL && extNode->spokeSetHead != spokeSetHead)
            {
                PatchNode *spokeSetCurr = extNode;
                while (spokeSetCurr->spokeSetNext != NULL)
                {
                    spokeSetCurr = spokeSetCurr->spokeSetNext;
                }
                spokeSetCurr->spokeSetNext = spokeSetHead;
                while (spokeSetCurr->spokeSetNext != NULL)
                {
                    spokeSetCurr = spokeSetCurr->spokeSetNext;
                    spokeSetCurr->spokeSetHead = extNode->spokeSetHead;
                }
            }
            ++iter;
        } while (iter != end);
    }
}

PatchNode::SurroundingLeafIterator::SurroundingLeafIterator(HalfEdge *edge) :
    nodeIter(edge),
    walker(edge->getPatch())
{
}

void PatchNode::SurroundingLeafIterator::operator++(void)
{
    do
    {
        ++nodeIter;
    }
    while ((*nodeIter)->getPatch() == NULL);
}

const PatchTreeWalker &PatchNode::SurroundingLeafIterator::operator*(void)
{
    walker.reset((*nodeIter)->getPatch(),
                 (*nodeIter)->getPatch()->getEdge()->stepsTo(*nodeIter));

    // This creates a PatchTreeWalker::Path that always selects the North-West
    // child
    PatchTreeWalker::Path nwPath(PatchTree::NORTH_WEST, &nwPath);
    walker.followPath(PatchTreeWalker::Path(PatchTree::CHILD, &nwPath));

    return walker;
}

bool PatchNode::SurroundingLeafIterator::operator==(
                                         const SurroundingLeafIterator &iter)
{
    return (nodeIter == iter.nodeIter);
}

bool PatchNode::SurroundingLeafIterator::operator!=(
                                         const SurroundingLeafIterator &iter)
{
    return (nodeIter != iter.nodeIter);
}

ostream &operator<<(ostream &os, const PatchNode &node)
{
    os << &node << " : edge " << node.getEdge();
    os << ", spoke set " << node.getSpokeSetHead() << endl;
    return os;
}

}
