/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef MESHFLATTENER_H
#define MESHFLATTENER_H

#include <map>
#include <tr1/functional>
#include <vector>
#include "types.h"

namespace snurbs {

class FlatMeshHandler;
class KnotInterval;
class Mesh;
class Vertex;

/**
 * Extracts objects from a Mesh and passes them to a FlatMeshHandler or to
 * callback functions for further processing. The onward processing might
 * write the flattened mesh to a file, or pass it to OpenGL for display.
 *
 * @ingroup Support
 */
class MeshFlattener
{
public:
    /**
     * Create a MeshFlattener. The caller should use dispose() to release the
     * MeshFlattener.
     * @param mesh The mesh to flatten. The @link Mesh::degree degree @endlink
     * of this mesh object determines the type of MeshFlattener that is
     * returned.
     * @see MeshBuilder::create
     */
    static MeshFlattener *create(const Mesh &mesh);

    /**
    * A very simple disposal method. For the rationale, see "Six of the best"
    * by Kevlin Henney: "Make acquisition and release symmetric"
    * @param meshFlattener The MeshFlattener to delete.
    * @see MeshBuilder::dispose
    */
    static void dispose(MeshFlattener *meshFlattener);

    /**
     * MeshFlattener destructor.
     */
    virtual ~MeshFlattener(void) {};

    /**
     * Call flatten() to extract objects from #mesh and send them to @c handler
     * for onward processing.
     * @param handler The FlatMeshHandler used to process the flat Mesh.
     */
    void flatten(FlatMeshHandler &handler);

    /**
     * Function type for the patch position callback requested using
     * receivePatchCoords()
     */
    typedef std::tr1::function<void (VertexPrecision,
                                     VertexPrecision)> CoordCallbackType;

    /**
     * Function type for the normal callback when receiving geometry. Called
     * once per face for receiveFlatNormals(), once per vertex for
     * receiveSmoothNormals().
     */
    typedef std::tr1::function<void (VertexPrecision,
                                     VertexPrecision,
                                     VertexPrecision)> NormalCallbackType;

    /**
     * Function type for the vertex callback when receiving geometry.
     */
    typedef std::tr1::function<void (VertexPrecision,
                                     VertexPrecision,
                                     VertexPrecision,
                                     VertexPrecision)> VertexCallbackType;

    /**
     * Function type for the notify callback used in notifyOnKnot()
     */
    typedef std::tr1::function<void (void)> NotifyCallbackType;

    /**
     * Returns the KnotInterval corresponding to the edge number @c edge in
     * face @c face . The indices must correspond to the order in which faces
     * and edges appear from flattenFaces().
     * @param face The zero-indexed face number
     * @param edge The zero-indexed edge number
     */
    virtual KnotInterval *getKnotIntervalForEdge(unsigned int face,
                                                 unsigned int edge) = 0;

    /**
     * Returns the KnotInterval which is flattened at a specific position in
     * the sequence of addKnotInterval() calls to the #flatMeshHandler. This
     * method calls flattenFaces() internally to fill the #knotIntervalMap, so
     * this should only be used on a newly-instantiated MeshFlattener.
     * @param index The zero-indexed number of the required knot in the list
     * of flattened KnotInterval objects.
     * @return The KnotInterval which is flattened as number @c index
     */
    KnotInterval *getFlattenedKnotNumber(unsigned int index);

    /**
     * @name Setters
     * @{
     */
    /** Setter for #compatible */
    void setCompatible(bool enabled);

    /**
     * Used to specify a KnotInterval object, where the FlatMeshHandler wants
     * to be notified every time the KnotInterval is encountered in the process
     * of flattening the mesh.
     * @param knot The KnotInterval of interest
     * @param startSelected The callback which receives notification before the
     * specified knot interval is encountered.
     * @param endSelected The callback which receives notification after the
     * specified knot interval is encountered.
     */
    void notifyOnKnot(KnotInterval *knot,
                      const NotifyCallbackType &startSelected,
                      const NotifyCallbackType &endSelected);

    /**
     * Used to request that patch coordinates (i.e.\ distance in parameter
     * space from the two edges of the patch) are sent to @c coordCallback
     * before the relevant vertex and normal callbacks are made.
     */
    void receivePatchCoords  (const CoordCallbackType &callback);

    /**
     * Used to request that face normals are sent to @c flatNormalCallback
     * before the face is flattened.
     */
    void receiveFlatNormals  (const NormalCallbackType &callback);

    /**
     * Used to request that vertex normals are sent to @c smoothNormalCallback
     * before the vertex is processed as part of a flattened face.
     */
    void receiveSmoothNormals(const NormalCallbackType &callback);

    /**
     * Used to request that vertex positions are sent to @c vertexCallback
     * as the vertex is processed as part of a flattened face.
     */
    void receiveVertices     (const VertexCallbackType &callback);
    /** @} */

    /**
     * Face flatten stage. This is implemented in the derived classes
     * PrimalMeshFlattener and DualMeshFlattener to account for the different
     * mesh structure in these two cases. This method fills the
     * #knotIntervalMap for use by flattenKnots() or getFlattenedKnotNumber(),
     * and is typically used as part of a call to flatten() with a
     * #flatMeshHandler. However, flattenFaces() can also be useful when called
     * independently of flatten() (e.g.\ when extracting geometry for OpenGL
     * calls), in which case the #flatMeshHandler may be NULL, and the caller
     * can use callback functions to receive the mesh instead.
     */
    virtual void flattenFaces(void) = 0;

protected:
    /** The Mesh which is being flattened. */
    const Mesh &mesh;
    /** The FlatMeshHandler used for output. */
    FlatMeshHandler *flatMeshHandler;
    /**
     * If @c true , the mesh is flattened in a way that is compatible with
     * other applications: no virtual faces are sent to the #flatMeshHandler
     * and vertices are projected into 3D space (with @f$ w=1 @f$) before being
     * output.
     */
    bool compatible;

    /**
     * @name Callback parameters
     * @{
     */
    /** The KnotInterval used in notifyOnKnot()     */
    KnotInterval        *notifyKnot;
    /** The start callback used in notifyOnKnot()   */
    NotifyCallbackType   startNotify;
    /** The end callback used in notifyOnKnot()     */
    NotifyCallbackType   endNotify;
    /** The callback used in receivePatchCoords()   */
    CoordCallbackType    coordCallback;
    /** The callback used in receiveVertices()      */
    VertexCallbackType   vertexCallback;
    /** The callback used in receiveFlatNormals()   */
    NormalCallbackType   flatNormalCallback;
    /** The callback used in receiveSmoothNormals() */
    NormalCallbackType   smoothNormalCallback;
    /** @} */

    /**
     * MeshFlattener constructor is protected. Use MeshFlattener::create to get
     * a MeshFlattener instance.
     * @param mesh The Mesh object to flatten
     */
    MeshFlattener(const Mesh &mesh)
            : mesh(mesh),
              flatMeshHandler(NULL),
              compatible(false),
              notifyKnot(NULL),
              coordCallback(NULL),
              vertexCallback(NULL),
              flatNormalCallback(NULL),
              smoothNormalCallback(NULL) {};

    /**
     * Given a vertex appearing in #realVertexList or #imagVertexList, returns
     * the index @c i such that <tt>realVertexList[i] == vertex</tt> or
     * <tt>imagVertexList[i - realVertexList.size()] == vertex</tt>.
     * Complexity is logarithmic.
     * @param vertex A pointer to a vertex in #realVertexList or
     * #imagVertexList.
     * @return The index of that vertex.
     */
    unsigned int vertexIndex(const Vertex *vertex) const;

    /**
     * @name Vertex Lists
     * The order of vertices in these lists determines the order in which
     * vertices are flattened. flattenVertices() uses forEachVertexUnique() on
     * vertexFlattener() to fill the lists. (Note: for primal meshes, there is
     * additional space complexity as a result of building the set contained in
     * Mesh::UniquePrimalVertexLister (through forEachVertexUnique()) and also
     * building these vectors containing the same values. The benefit is that
     * the iterator distance function used in vertexIndex() can then run in
     * constant time. Otherwise we would need linear time for every index
     * lookup.)
     * @{
     */
    /** For real vertices */
    std::vector<const Vertex *> realVertexList;
    /** For `imaginary' vertices */
    std::vector<const Vertex *> imagVertexList;
    /** @} */

    /**
     * Typedef for #knotIntervalMap
     */
    typedef std::map<const KnotInterval *,
                     std::pair<const Vertex*, const Vertex *> >
                         KnotIntervalMapType;

    /**
     * Map linking KnotInterval objects to an edge on which the knot spacing
     * is defined. The link between the knot interval and the specified
     * edge depends on whether the mesh is primal or dual.
     * @see PrimalMeshBuilder::addKnotInterval
     * @see DualMeshBuilder::addKnotInterval
     */
    KnotIntervalMapType knotIntervalMap;

    /** Vertex flatten stage. */
    virtual void flattenVertices(void);
    /** Knot interval flatten stage. */
    virtual void flattenKnots(void);

private:
    /**
     * Helper function used to call FileWriter::writeVertex on every vertex
     * referenced by #mesh.
     */
    void vertexFlattener(const Vertex* const vertex);
};

inline void MeshFlattener::setCompatible(bool enabled)
{
    compatible = enabled;
}

inline void MeshFlattener::notifyOnKnot(
                                 KnotInterval *knot,
                                 const NotifyCallbackType &startSelected,
                                 const NotifyCallbackType &endSelected)
{
     notifyKnot = knot;
    startNotify = startSelected;
      endNotify = endSelected;
}

inline void MeshFlattener::receivePatchCoords(
                                 const CoordCallbackType &callback)
{
    coordCallback = callback;
}

inline void MeshFlattener::receiveFlatNormals(
                                 const NormalCallbackType &callback)
{
    flatNormalCallback = callback;
}

inline void MeshFlattener::receiveSmoothNormals(
                                 const NormalCallbackType &callback)
{
    smoothNormalCallback = callback;
}

inline void MeshFlattener::receiveVertices(
                                 const VertexCallbackType &callback)
{
    vertexCallback = callback;
}

}

#endif
