/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <algorithm>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include "meshflattener.h"
#include "primalmeshflattener.h"
#include "dualmeshflattener.h"
#include "mesh.h"
#include "flatmeshhandler.h"
#include "vertex.h"
#include "knotinterval.h"

namespace snurbs {

using namespace boost;
using namespace std;
using namespace tr1::placeholders;

MeshFlattener *MeshFlattener::create(const Mesh &mesh)
{
    MeshFlattener *newFlattener;

    if (mesh.isPrimal())
    {
        newFlattener = new PrimalMeshFlattener(mesh);
    }
    else
    {
        newFlattener = new DualMeshFlattener(mesh);
    }

    return newFlattener;
}

void MeshFlattener::dispose(MeshFlattener *meshFlattener)
{
    delete meshFlattener;
}

void MeshFlattener::flatten(FlatMeshHandler &handler)
{
    flatMeshHandler = &handler;
    handler.startMesh();
    flattenVertices();
    handler.finishVertices();
    flattenFaces();
    handler.finishFaces();
    flattenKnots();
    realVertexList.clear();
    imagVertexList.clear();
    handler.finishKnotIntervals();
    flatMeshHandler = NULL;
}

KnotInterval *MeshFlattener::getFlattenedKnotNumber(unsigned int index)
{
    flatMeshHandler = NULL;
    flattenFaces();
    KnotIntervalMapType::iterator iter = knotIntervalMap.begin();
    while (index)
    {
        --index;
        ++iter;
    }
    KnotInterval *returnKnot = const_cast<KnotInterval *>(iter->first);
    knotIntervalMap.clear();
    return returnKnot;
}

unsigned int MeshFlattener::vertexIndex(const Vertex *vertex) const
{
    unsigned int intIndex;

    if (vertex->isReal())
    {
        vector<const Vertex *>::const_iterator index =
            lower_bound(realVertexList.begin(), realVertexList.end(), vertex);
        intIndex = distance(realVertexList.begin(), index);
    }
    else
    {
        intIndex = realVertexList.size();
        vector<const Vertex *>::const_iterator index =
            lower_bound(imagVertexList.begin(), imagVertexList.end(), vertex);
        intIndex += distance(imagVertexList.begin(), index);
    }

    return intIndex;
}

void MeshFlattener::flattenVertices(void)
{
    forEachVertexUnique(tr1::bind(&MeshFlattener::vertexFlattener, this, _1),
                        &mesh);
}

void MeshFlattener::flattenKnots(void)
{
    KnotIntervalMapType::iterator iter;

    while(!knotIntervalMap.empty())
    {
        iter = knotIntervalMap.begin();
        if (flatMeshHandler != NULL)
        {
            flatMeshHandler->addKnotInterval(vertexIndex(iter->second.first),
                                             vertexIndex(iter->second.second),
                                             iter->first->getInterval());
        }
        knotIntervalMap.erase(iter);
    }
}

void MeshFlattener::vertexFlattener(const Vertex* const vertex)
{
    if (vertex->isReal())
    {
        if (flatMeshHandler != NULL)
        {
            if (compatible)
            {
                flatMeshHandler->addVertex(vertex->getX() / vertex->getW(),
                                           vertex->getY() / vertex->getW(),
                                           vertex->getZ() / vertex->getW(),
                                           1.0);
            }
            else
            {
                flatMeshHandler->addVertex(vertex->getX(),
                                           vertex->getY(),
                                           vertex->getZ(),
                                           vertex->getW());
            }
        }
        realVertexList.push_back(vertex);
    }
    else
    {
        imagVertexList.push_back(vertex);
    }
}

}
