/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef MESH_H
#define MESH_H

#include <ostream>
#include <set>
#include <stack>
#include <vector>
#include "dualpatchtreeleaf.h"
#include "patchtree.h"
#include "patchtreewalker.h"
#include "primalpatchtreeleaf.h"

/** @file */

namespace snurbs {

class HalfEdge;
class KnotInterval;
class PatchNode;
class PatchTreeRoot;
class Vertex;

/**
 * Represents a manifold mesh. Use a MeshBuilder to build a Mesh object.
 *
 * @ingroup Mesh
 */
class Mesh
{
    friend class PrimalMeshBuilder;
    friend class DualMeshBuilder;

    friend std::ostream &operator<<(std::ostream &os, const Mesh &mesh);

    template <typename Fn, typename M>
    friend void forEachVertexUnique(Fn function, M mesh);

public:
    /**
     * Mesh constructor.
     * @param degree The degree of the B-spline patches that make up the limit
     * surface. If the degree is odd, then the scheme is `primal' and all faces
     * must be quadrilateral. If the degree is even, then the scheme is `dual'
     * and all vertices must be four-valent. The degree determines whether
     * primal classes (e.g. PrimalMeshBuilder) or dual classes (e.g.
     * DualMeshBuilder) are used to construct the mesh and make up its parts.
     */
    Mesh(unsigned char degree);

    /**
     * Mesh copy constructor
     * @param mesh The source for the copy
     */
    Mesh(const Mesh &mesh);

    /**
     * Mesh destructor. Uses the #patchNodes, #halfEdges, #patches and #knots
     * lists to delete all child objects.
     */
    ~Mesh(void);

    /**
     * Check whether the Mesh is primal or dual.
     * @return @c true if this Mesh object is primal (has odd degree), @c false
     * if it is dual (has even degree)
     */
    bool isPrimal(void) const;

    /**
     * Returns @c true if the Mesh contains a boundary HalfEdge (with NULL
     * HalfEdge::patch)
     */
    bool hasBoundary(void) const;

    /**
     * Delete all objects and reset the Mesh.
     */
    void clear(void);

    /**
     * Carry out a subdivision step.
     */
    void subdivide(void);

    /**
     * @return The number of Vertex objects accessible from this Mesh.
     */
    int getNumVerts(void) const;

    /**
     * @return The number of faces represented by this Mesh.
     */
    int getNumFaces(void) const;
    /**
     * @return The number of leaf KnotInterval objects accessible from this
     * Mesh.
     */
    int getNumKnots(void) const;

    /** Returns the subdivision degree for this Mesh. */
    unsigned int getDegree(void) const;

    /**
     * Iterates over the KnotInterval leaves in the list of binary trees with
     * roots in Mesh::knots
     *
     * @ingroup Iterators
     */
    class KnotIterator : public std::iterator<std::input_iterator_tag,
                                              KnotInterval *>
    {
    public:
        /** KnotIterator constructor */
        KnotIterator(std::vector<KnotInterval *>::const_iterator begin);

        /**
         * @name Operators
         * @{
         */
        bool operator==(const KnotIterator &iter) const;
        bool operator!=(const KnotIterator &iter) const;
        void operator++(void);
        KnotInterval *&operator*(void);
        /** @} */

    private:
        /**
         * Enum used to keep track of the current position of the iterator in
         * the binary tree of KnotInterval objects. The KnotIterator could be
         * implemented without this enum or the #childPath stack, but there
         * would be more comparisons to find whether a KnotInterval is the left
         * or right child of its parent. This stack should have minimal cost,
         * so seems the best implementation.
         */
        enum Child { LEFT, RIGHT };

        /** The underlying iterator into Mesh::knots */
        std::vector<KnotInterval *>::const_iterator  vectorIter;
        /**
         * The path from the KnotInterval pointed to by #vectorIter, to
         * #current.
         * @see Child
         */
        std::stack <Child, std::vector<Child> >       childPath;
        /** The KnotInterval pointed to by this KnotIterator. */
        KnotInterval *                                  current;
    };

    /**
     * Iterates over the PatchTreeLeaf objects in the list of patch trees with
     * roots in Mesh::patches
     *
     * @ingroup Iterators
     */
    class PatchIterator : public std::iterator<std::forward_iterator_tag,
                                               PatchTree *>
    {
    public:
        /** PatchIterator constructor. */
        PatchIterator(std::vector<PatchTreeRoot *>::const_iterator begin);

        /**
         * @name Operators
         * @{
         */
        bool operator==(const PatchIterator &iter) const;
        bool operator!=(const PatchIterator &iter) const;
        void operator++(void);
        PatchTree*& operator*(void);
        /** @} */

    protected:
        /** The underlying iterator into Mesh::patches */
        std::vector<PatchTreeRoot *>::const_iterator  patchIter;
        /**
         * The PatchTree pointed to by this PatchIterator. Note that if
         * @c current is not a PatchTreeLeaf, the dereference operation selects
         * the first leaf accessible from this tree. It is necessary to make
         * this implicit definition so that operator++() doesn't need to
         * dereference #patchIter before operator*() is called.
         */
        PatchTree *                                   current;
    };

    /**
     * Iterates over the same leaves as PatchIterator, but with two
     * differences.
     * - The deference operator does some extra work to make it suitable as an
     * l-value that we can assign to during Mesh::subdivide()
     * - The preincrement operator accounts for the fact that after calling
     * PatchTreeLeaf::refineTree, the #current PatchTree is one level lower
     * than before. So the preincrement operator steps to the PatchTree parent
     * before continuing.
     *
     * @ingroup Iterators
     */
    class RefineTreeIterator : public PatchIterator
    {
    public:
        /** RefineTreeIterator constructor */
        RefineTreeIterator(std::vector<PatchTreeRoot *>::iterator begin);

        /**
         * @name Operators
         * @{
         */
        void operator++(void);
        PatchTree*& operator*(void);
        /** @} */
    };

    /**
     * Base class for PrimalVertexIterator and DualVertexIterator. This class
     * can actually be instantiated (it has no virtual members, for efficiency)
     * but is useless as an iterator, as it provides no preincrement or
     * dereference operator implementations.
     *
     * @ingroup Iterators
     */
    class VertexIterator : public std::iterator<std::input_iterator_tag,
                                                Vertex *>
    {
    public:
        /** VertexIterator constructor */
        VertexIterator(std::vector<PatchTreeRoot *>::const_iterator begin);

        /**
         * @name Operators
         * @{
         */
        bool operator==(const VertexIterator &iter) const;
        bool operator!=(const VertexIterator &iter) const;
        /** @} */

    protected:
        /** The underlying PatchIterator */
        PatchIterator patchIter;
    };

    /**
     * Specialisation of VertexIterator for primal meshes. This iterator breaks
     * some of the STL iterator rules, by revisiting an object a multiple
     * number of times. Nevertheless, this is the most efficient way to iterate
     * over all the vertices in a primal mesh, if the caller is ambivalent
     * about uniqueness (i.e. the action performed must be idempotent).
     *
     * @ingroup Iterators
     */
    class PrimalVertexIterator : public VertexIterator
    {
    public:
        /** PrimalVertexIterator constructor */
        PrimalVertexIterator(
                std::vector<PatchTreeRoot *>::const_iterator begin);

        /**
         * @name Operators
         * @{
         */
        void operator++(void);
        Vertex* const &operator*(void);
        /** @} */

    private:
        /**
         * The index of the current vertex, as referenced from the underlying
         * PatchIterator
         */
        unsigned char vertexNum;
    };

    /**
     * Helper class which copies the output of a PrimalVertexIterator into a
     * std::set in order to create a unique list. Used when a caller wants to
     * apply a function to all the vertices in a Mesh exactly once.
     *
     * @ingroup Iterators
     */
    class UniquePrimalVertexLister
    {
    public:
        /** UniquePrimalVertexLister constructor */
        UniquePrimalVertexLister(
                std::vector<PatchTreeRoot *>::const_iterator begin,
                std::vector<PatchTreeRoot *>::const_iterator end);

        /** @return an iterator to the @c begin iterator of #vertices */
        std::set<Vertex *>::iterator  begin(void);
        /** @return an iterator to the @c end iterator of #vertices */
        std::set<Vertex *>::iterator  end(void);
        /** @return the result of calling @c size on #vertices */
        std::set<Vertex *>::size_type size(void) const;
    private:
        /**
         * The @c set used to construct a list of unique Vertex pointers.
         */
        std::set<Vertex *> vertices;
    };

    /**
     * Specialisation of VertexIterator for dual meshes.
     *
     * @ingroup Iterators
     */
    class DualVertexIterator : public VertexIterator
    {
    public:
        /** DualVertexIterator constructor. */
        DualVertexIterator(std::vector<PatchTreeRoot *>::const_iterator begin);
        /**
         * @name Operators
         * @{
         */
        void operator++(void);
        Vertex* const &operator*(void);
        /** @} */
    };

    /**
     * @return A PatchIterator initialised with the begin() iterator of
     * #patches
     */
    PatchIterator beginPatches(void) const;

    /**
     * @return A PatchIterator initialised with the end() iterator of #patches
     */
    PatchIterator endPatches(void) const;

    /**
     * @param knot The KnotInterval to find in #knots
     * @return The index @c i such that @code knots[i] == knot @endcode
     */
    unsigned int getKnotIndex(KnotInterval *knot) const;

    /**
     * @param index The index into #knots
     * @return @code knots[index] @endcode
     */
    KnotInterval *getKnotNumber(unsigned int index) const;

private:
    /**
     * Degree is const for now. It could, in fact, be allowed to vary as long
     * as it didn't switch primal to dual or vice-versa. And as long as no
     * subdivision functions have yet been called (unless subdividing at
     * different degrees at different steps is actually what's intended). And
     * as long as boundary culling is correctly adjusted. So it's probably
     * better to keep it const!
     */
    const unsigned char          degree;
    /** List of PatchNode objects for this Mesh.     */
    std::vector<PatchNode     *> patchNodes;
    /** List of HalfEdge objects for this Mesh.      */
    std::vector<HalfEdge      *> halfEdges;
    /** List of PatchTreeRoot objects for this Mesh. */
    std::vector<PatchTreeRoot *> patches;
    /** List of KnotInterval objects for this Mesh.  */
    std::vector<KnotInterval  *> knots;
};

/**
 * Preincrement operator for PatchTree::Direction enumerators; used in
 * Mesh::PatchIterator. Rotates PatchTree::NORTH_EAST to PatchTree::SOUTH_EAST,
 * and so on in the expected way. PatchTree::NORTH becomes PatchTree::SOUTH,
 * and PatchTree::WEST becomes PatchTree::EAST.
 * @param d The PatchTree::Direction to increment.
 */
PatchTree::Direction &operator++(PatchTree::Direction &d);

/** Stream << operator for Mesh objects. */
std::ostream &operator<<(std::ostream &os, const Mesh &mesh);

/**
 * Template function for the common scenario where we have a @c function which
 * we want to apply to every vertex in @c mesh , but only once per vertex. For
 * dual meshes, this is easy. Primal meshes have to use a
 * Mesh::UniquePrimalVertexLister to generate a list of unique Vertex pointers.
 * @param function The function to apply to each Vertex
 * @param mesh The mesh object to apply to. Note that @c mesh is a parameter
 * instead of making this a template member, because the template parameter
 * @c M can then adapt to Mesh and @c const Mesh types.
 */
template <typename Fn, typename M>
inline void forEachVertexUnique(Fn function, M mesh)
{
    if (mesh->isPrimal())
    {
        Mesh::UniquePrimalVertexLister list(mesh->patches.begin(),
                                            mesh->patches.end());
        for_each(list.begin(), list.end(), function);
    }
    else
    {
        for_each(Mesh::DualVertexIterator(mesh->patches.begin()),
                 Mesh::DualVertexIterator(mesh->patches.end()),
                 function);
    }
}

inline bool Mesh::isPrimal(void) const
{
    return degree % 2;
}

inline unsigned int Mesh::getDegree(void) const
{
    return degree;
}

inline bool Mesh::KnotIterator::operator==(const KnotIterator &iter) const
{
    return (iter.vectorIter == vectorIter);
}

inline bool Mesh::KnotIterator::operator!=(const KnotIterator &iter) const
{
    return !(*this == iter);
}

inline PatchTree::Direction &operator++(PatchTree::Direction &d)
{
    d = PatchTreeWalker::rotate(d, 1);
    if (d == PatchTree::NORTH || d == PatchTree::SOUTH ||
        d == PatchTree::WEST  || d == PatchTree::EAST)
    {
        d = PatchTreeWalker::rotate(d, 1);
    }
    return d;
}

inline bool Mesh::PatchIterator::operator==(const PatchIterator &iter) const
{
    return (iter.patchIter == patchIter);
}

inline bool Mesh::PatchIterator::operator!=(const PatchIterator &iter) const
{
    return !(*this == iter);
}

inline void Mesh::RefineTreeIterator::operator++(void)
{
    current = current->getParent();
    PatchIterator::operator++();
}

inline PatchTree*& Mesh::RefineTreeIterator::operator*(void)
{
    PatchTree *result = PatchIterator::operator*();
    return result->getParent()->getChild(result->dirFromParent());
}

inline bool Mesh::VertexIterator::operator==(const VertexIterator &iter) const
{
    return (patchIter == iter.patchIter);
}

inline bool Mesh::VertexIterator::operator!=(const VertexIterator &iter) const
{
    return !(*this == iter);
}

inline void Mesh::PrimalVertexIterator::operator++(void)
{
    vertexNum++;
    if (vertexNum == 4)
    {
        ++patchIter;
        vertexNum = 0;
    }
}

inline Vertex* const &Mesh::PrimalVertexIterator::operator*(void)
{
    return static_cast<PrimalPatchTreeLeaf *>(*patchIter)->
                                             getVertex(vertexNum);
}

inline std::set<Vertex *>::iterator Mesh::UniquePrimalVertexLister::begin(void)
{
    return vertices.begin();
}

inline std::set<Vertex *>::iterator Mesh::UniquePrimalVertexLister::end(void)
{
    return vertices.end();
}

inline std::set<Vertex *>::size_type Mesh::UniquePrimalVertexLister::size(void)
    const
{
    return vertices.size();
}

inline void Mesh::DualVertexIterator::operator++(void)
{
    ++patchIter;
}

inline Vertex* const &Mesh::DualVertexIterator::operator*(void)
{
    return static_cast<DualPatchTreeLeaf *>(*patchIter)->getVertex();
}

inline Mesh::PatchIterator Mesh::beginPatches(void) const
{
    return PatchIterator(patches.begin());
}

inline Mesh::PatchIterator Mesh::endPatches(void) const
{
    return PatchIterator(patches.end());
}

}

#endif
