/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef KNOTINTERVAL_H
#define KNOTINTERVAL_H

#include <ostream>
#include "types.h"

/** @file */
// This file appears in the documentation because of the global function
// std::ostream &operator<<(std::ostream &os, const KnotInterval &knot)
// defined here

namespace snurbs {

class PatchNode;

/**
 * KnotInterval objects form a binary tree of knot spacings, which are
 * subdivided in two whenever a particular strip of quadrilateral faces is
 * subdivided.
 *
 * @ingroup Mesh
 */
class KnotInterval
{
public:
    /**
     * KnotInterval constructor.
     * @param value The knot spacing represented by this KnotInterval object.
     */
    KnotInterval(KnotPrecision value);
    /** KnotInterval destructor. */
    ~KnotInterval(void);

    /**
     * @name Getter functions
     * @{
     */
    /** Getter for #interval. */
    KnotPrecision getInterval    (void) const;
    /** Getter for #leftChild. */
    KnotInterval *getLeftChild   (void) const;
    /** Getter for #rightChild. */
    KnotInterval *getRightChild  (void) const;
    /** Getter for #parent. */
    KnotInterval *getParent      (void) const;
    /** Getter for #leftExtNode. */
    PatchNode    *getLeftExtNode (void) const;
    /** Getter for #rightExtNode. */
    PatchNode    *getRightExtNode(void) const;
    /** @} */

    /**
     * @name Setter functions
     * @{
     */
    /** Setter for #interval. */
    void setInterval    (KnotPrecision newInterval);
    /** Setter for #leftExtNode. */
    void setLeftExtNode (PatchNode *newLeftExtNode);
    /** Setter for #rightExtNode. */
    void setRightExtNode(PatchNode *newRightExtNode);
    /** @} */

    /**
     * Make a request for a knot to be inserted at @c position , where 0
     * @f$\le@f$ @c position @f$\le@f$ #interval . If only one such request is
     * made of this KnotInterval, a new knot is inserted at the specified
     * position. If more than one request is made, the new knot is inserted at
     * the mean of the requested positions.
     * @param position The requested position for the new knot
     */
    void insertionRequest(KnotPrecision position);

    /**
     * Clears #numRequests and #requestSum so that toBeSubdivided() returns
     * @c false.
     */
    void clearRequests(void);

    /**
     * Subdivides (allocating new KnotInterval objects) if toBeSubdivided()
     * returns @c true.
     */
    void subdivide(void);

    /**
     * Returns @c true if this knot interval should be subdivided in the next
     * subdivision step: there must have been at least one insertionRequest(),
     * and #interval must be non-zero.
     */
    bool toBeSubdivided(void) const;

private:
    /**
     * KnotInterval objects are split on subdivision, and leftChild holds the
     * KnotInterval that gives the spacing of one half of the split. The
     * orientation is `left' with respect to the PatchTreeRoot::edge HalfEdge
     * for each PatchTreeRoot referencing this KnotInterval.
     */
    KnotInterval *leftChild;
    /**
     * KnotInterval objects are split on subdivision, and rightChild holds the
     * KnotInterval that gives the spacing of one half of the split. The
     * orientation is `right' with respect to the PatchTreeRoot::edge HalfEdge
     * for each PatchTreeRoot referencing this KnotInterval.
     */
    KnotInterval *rightChild;
    /**
     * The parent KnotInterval. NULL for KnotInterval root nodes.
     */
    KnotInterval *parent;
    /** The knot spacing represented by this KnotInterval object. */
    KnotPrecision interval;

    /**
     * Pointer used to construct "spoke-set" equivalence classes for
     * extraordinary PatchNodes.
     */
    PatchNode    *leftExtNode;
    /**
     * Pointer used to construct "spoke-set" equivalence classes for
     * extraordinary PatchNodes.
     */
    PatchNode    *rightExtNode;

    /** The sum of positions passed to insertionRequest(). */
    KnotPrecision requestSum;
    /** The number of calls to insertionRequest(). */
    unsigned char numRequests;

    /**
     * Returns the position a new knot should be inserted, taking into account
     * any requests made to insertionRequest().
     */
    KnotPrecision resolveRequests(void) const;
};

inline KnotPrecision KnotInterval::getInterval    (void) const
{
    return interval;
}

inline KnotInterval *KnotInterval::getLeftChild   (void) const
{
    return leftChild;
}

inline KnotInterval *KnotInterval::getRightChild  (void) const
{
    return rightChild;
}

inline KnotInterval *KnotInterval::getParent      (void) const
{
    return parent;
}

inline PatchNode    *KnotInterval::getLeftExtNode (void) const
{
    return leftExtNode;
}

inline PatchNode    *KnotInterval::getRightExtNode(void) const
{
    return rightExtNode;
}

inline void KnotInterval::setInterval    (KnotPrecision newInterval)
{
    interval = newInterval;
}

inline void KnotInterval::setLeftExtNode (PatchNode *newLeftExtNode)
{
    leftExtNode = newLeftExtNode;
}

inline void KnotInterval::setRightExtNode(PatchNode *newRightExtNode)
{
    rightExtNode = newRightExtNode;
}

/** Stream << operator for KnotInterval objects. */
std::ostream &operator<<(std::ostream &os, const KnotInterval &knot);

}

#endif
