/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cassert>
#include "knotinterval.h"

namespace snurbs {

using namespace std;

KnotInterval::KnotInterval(KnotPrecision value) : leftChild(NULL),
                                                  rightChild(NULL),
                                                  parent(NULL),
                                                  interval(value),
                                                  leftExtNode(NULL),
                                                  rightExtNode(NULL),
                                                  requestSum(0),
                                                  numRequests(0)
{
}


KnotInterval::~KnotInterval(void)
{
    delete leftChild;
    delete rightChild;
}

void KnotInterval::insertionRequest(KnotPrecision position)
{
    assert(0 <= position);
    assert(position <= interval);
    requestSum += position;
    ++numRequests;
}

void KnotInterval::clearRequests(void)
{
    requestSum = 0;
    numRequests = 0;
}

void KnotInterval::subdivide(void)
{
    assert(leftChild == NULL && rightChild == NULL);

    if (toBeSubdivided())
    {
        KnotPrecision newKnot = resolveRequests();
        leftChild  = new KnotInterval(           newKnot);
        rightChild = new KnotInterval(interval - newKnot);
        leftChild->parent  = this;
        rightChild->parent = this;
    }
}

bool KnotInterval::toBeSubdivided(void) const
{
    return numRequests > 0 && interval > 0;
}

KnotPrecision KnotInterval::resolveRequests(void) const
{
    if (numRequests == 0)
    {
        return interval / 2;
    }
    else
    {
        return requestSum / numRequests;
    }
}

ostream &operator<<(ostream &os, const KnotInterval &knot)
{
    os << &knot << " : value " << knot.getInterval() << endl;

    return os;
}

}
