﻿// $Id: TdcbGridAggregator.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib.Parameters;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Aggregates cost using the dual-cross-bilateral grid with temporal component.
	/// </summary>
	class TdcbGridAggregator : AggregatorStereoNode, IDisposable
	{
		#region Parameters

		[Parameter(Minimum = 1, Maximum = 10, Default = 5, FriendlyName = "Number of grids")]
		public int NumGrids { get; set; }

		[Parameter(Minimum = 5, Maximum = 20, Default = 10, FriendlyName = "Lightness sigma")]
		public int SigmaC { get; set; }

		[Parameter(Minimum = 5, Maximum = 20, Default = 10, FriendlyName = "Spatial sigma")]
		public int SigmaS { get; set; }


		[Parameter(Minimum = 1, Maximum = 4, Default = 2, FriendlyName = "Weighting (see code)")]
		public int Weighting { get; set; }

		[Parameter(Minimum = 0, Maximum = 10, Default = 2, FriendlyName = "A (see code)")]
		public float WeightA { get; set; }

		[Parameter(Minimum = -10, Maximum = 0, Default = -10, FriendlyName = "B (see code)")]
		public float WeightB { get; set; }

		#endregion

		// bilateral grids
		private List<InputImage> gridTextures = new List<InputImage>();


		public TdcbGridAggregator() : base() { }
		public TdcbGridAggregator(ExecutionStream stream) : base(stream) { }

		/// <summary>
		/// Aggregates (locally optimizes) the costs.
		/// </summary>
		/// <param name="leftImage">The left input image.</param>
		/// <param name="rightImage">The right input image.</param>
		/// <param name="costGrid">The cost grid.</param>
		public override void AggregateCosts(InputImage leftImage, InputImage rightImage, CostSpace costGrid)
		{
			// remove grids if necessary
			while (gridTextures.Count > NumGrids)
			{
				gridTextures.Last().Dispose();
				gridTextures.Remove(gridTextures.Last());
			}

			// add more grids if necessary
			if (gridTextures.Count < NumGrids)
			{
				gridTextures.Add(new InputImage());
			}

			// rotate previous grids along by one, i.e. i => (i + 1) % n
			InputImage lastGrid = gridTextures.Last();
			gridTextures.Remove(lastGrid);
			gridTextures.Insert(0, lastGrid);

			ManagedAggregators.RunTdcbGridKernel(costGrid, leftImage, rightImage, gridTextures.ToArray(), SigmaS, SigmaC, Weighting, WeightA, WeightB, stream);
		}

		public void Dispose()
		{
			gridTextures.ForEach(x => x.Dispose());
		}
	}

	class TdcbGridAggregatorFactory : StereoNodeFactory<AggregatorStereoNode>
	{
		public override AggregatorStereoNode Create(ExecutionStream stream)
		{
			return new TdcbGridAggregator(stream);
		}

		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (gridType == CostSpaceType.Single);
		}

		public override string Name
		{
			get { return "Temporal DCB grid"; }
		}

		public override string SortKey { get { return "_5_" + Name; } } // put this fifth
	}
}
