﻿// $Id: FloatParameterControl.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using VideoLib.Parameters;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A control to modify a floating-point parameter.
    /// </summary>
    public partial class FloatParameterControl : UserControl, IParameterControl
    {
        private ParameterInfo paramInfo;
        private object parameter;
        private List<object> followers;

        private float paramRange = 1.0f;
        private float paramMin = 0.0f;

        /// <summary>
        /// Initializes a new instance of the <see cref="ContinuousParameterControl"/> class.
        /// </summary>
        /// <param name="paramInfo">The param info.</param>
        /// <param name="parameter">The parameter.</param>
        public FloatParameterControl(ParameterInfo paramInfo, object parameter)
            : this(paramInfo, parameter, null)
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="ContinuousParameterControl"/> class.
        /// </summary>
        /// <param name="paramInfo">The parameter info.</param>
        /// <param name="parameter">The parameter.</param>
        /// <param name="followers">Any followers.</param>
        public FloatParameterControl(ParameterInfo paramInfo, object parameter, List<object> followers)
        {
            InitializeComponent();

            // copy arguments
            this.paramInfo = paramInfo;
            this.parameter = parameter;
            this.followers = followers ?? new List<object>(); // replace by empty list if null
            
            // set minimum and range from parameter info
            if (paramInfo.Maximum != null && paramInfo.Minimum != null)
            {
                paramMin = Convert.ToSingle(paramInfo.Minimum);
                paramRange = Convert.ToSingle(paramInfo.Maximum) - paramMin;
            }

            // set parameter name
            nameLabel.Text = paramInfo.Name;

            // get the value of the parameter
            float value = paramInfo.GetValue<float>(parameter);

            // set up the track bar
            slider.Minimum = 0;
            slider.Maximum = 250; // magic number for 'continuous' slider
            int tickpos = (paramRange <= 0.0f ? 0 : (int)(0.5f + slider.Maximum * (value - paramMin) / paramRange)); // round to nearest tick
            slider.Value = Math.Min(slider.Maximum, Math.Max(slider.Minimum, tickpos));
            slider.ValueChanged += new EventHandler(slider_ValueChanged);
            
            // set the trackbar to the actual value of the parameter (the code above might round it)
            SetValueLabel(value);
        }

        /// <summary>
        /// Handles the ValueChanged event of the slider control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void slider_ValueChanged(object sender, EventArgs e)
        {
            if (parameter != null)
            {
                float value = (float)slider.Value / (float)slider.Maximum * paramRange + paramMin;
                paramInfo.SetValue<float>(parameter, value);
                SetValueLabel(value);

                foreach (object follower in followers)
                    paramInfo.SetValue<float>(follower, value);
            }
        }

        /// <summary>
        /// Sets the value label.
        /// </summary>
        /// <param name="val">The value.</param>
        private void SetValueLabel(float val)
        {
            // try to show 3 significant digits
            if (val >= 10.0f) valueLabel.Text = val.ToString("00.0");
            else valueLabel.Text = val.ToString("0.00");
        }

        /// <summary>
        /// Handles the Click event of the resetButton control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void resetButton_Click(object sender, EventArgs e)
        {
            // position the slider at the tick nearest to the default value
            slider.Value = (paramRange <= 0.0f ? 0 : (int)(0.5f + slider.Maximum * (Convert.ToSingle(paramInfo.Default) - paramMin) / paramRange));

            // display default value
            SetValueLabel(Convert.ToSingle(paramInfo.Default));
        }

        #region IParameterControl Members

        public void RefreshValue()
        {
            float value = paramInfo.GetValue<float>(parameter);
            SetValueLabel(value);
            slider.Value = (paramRange <= 0.0f ? 0 : (int)(0.5f + slider.Maximum * (value - paramMin) / paramRange)); // round to nearest tick
        }

        #endregion
    }
}
