#pragma once

#include <builtin_types.h>

enum RgbImageType
{
	RGB_IMAGE_TYPE_ANY,
	RGB_IMAGE_TYPE_XRGB32
};

// A very simple, very open, cuda-held image class
class RgbImage
{
private:
	unsigned int* gpuImage;
	size_t gpuImagePitch;
	int w, h;

	RgbImageType imageType;

public:
	RgbImage() : gpuImage(0), gpuImagePitch(0), w(0), h(0) { }
	~RgbImage();

	void Create(RgbImageType type, int width, int height);
	void Destroy();

	// This method allows nodes to operate without requiring in-place modification, swapping the resultant data
	// and responsibility to the output grid
	void SwapData(RgbImage & other);

	unsigned int* GetImage() const { return gpuImage; }
	RgbImageType GetType() const { return imageType; }

	int GetPitch() const { return gpuImagePitch; }
	int GetWidth() const { return w; }
	int GetHeight() const { return h; }
};

enum CostSpaceGridType
{
	COST_SPACE_GRID_TYPE_ANY,
	COST_SPACE_GRID_TYPE_SINGLE,
	COST_SPACE_GRID_TYPE_HALFS_D
};

// A class encapsulating a couple of different formats of cost space grid
class CostSpaceGrid
{
private:
	cudaPitchedPtr gpuGrid;
	int w, h, d;

	CostSpaceGridType gridType;

public:
	CostSpaceGrid() : w(0), h(0), d(0) { gpuGrid.ptr = 0; }
	~CostSpaceGrid();

	void Create(CostSpaceGridType type, int width, int height, int depth);
	void Destroy();

	// This method allows nodes to operate without requiring in-place modification, swapping the resultant data
	// and responsibility to the output grid
	void SwapData(CostSpaceGrid & other);

	void SizeToMatch(const CostSpaceGrid & other);

	cudaPitchedPtr GetGrid() const { return gpuGrid; }
	CostSpaceGridType GetType() const { return gridType; }

	int GetWidth() const { return w; }
	int GetHeight() const { return h; }
	int GetDepth() const { return d; }
};

enum DepthMapType
{
	DEPTH_MAP_TYPE_ANY,
	DEPTH_MAP_TYPE_FLOAT
};

// A class encapsulating a depth-map, very similar to RgbImage
class DepthMap
{
private:
	float* gpuImage;
	size_t gpuImagePitch;
	int w, h;

	DepthMapType mapType;

public:
	DepthMap() : gpuImage(0), gpuImagePitch(0), w(0), h(0) { }
	~DepthMap();

	void Create(DepthMapType type, int width, int height);
	void Destroy();

	// This method allows nodes to operate without requiring in-place modification, swapping the resultant data
	// and responsibility to the output grid
	void SwapData(DepthMap & other);

	float* GetDepthMap() const { return gpuImage; }
	DepthMapType GetType() const { return mapType; }

	int GetPitch() const { return gpuImagePitch; }
	int GetWidth() const { return w; }
	int GetHeight() const { return h; }
};