﻿// $Id: StereoNodeFactory.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// A base class for stereo node factories, providing abstractions and helper methods for checking that a factory is
	/// compatible with input images, cost space formats, and depth map types
	/// <para/>This should not be derived from directly, rather use StereoNodeFactory[T]
	/// </summary>
	abstract class StereoNodeFactory
	{    
		protected bool isValid;

		/// <summary>
		/// Initializes a new instance of the <see cref="StereoNodeFactory"/> class.
		/// </summary>
		public StereoNodeFactory() { isValid = true; }

		/// <summary>
		/// Gets a value indicating whether this instance is valid for the last checked datatypes.
		/// </summary>
		public bool IsValid { get { return isValid; } }

		/// <summary>
		/// Checks if the <see cref="StereoNode"/> is compatible with the given datatypes.
		/// </summary>
		/// <param name="imageType">Image datatype.</param>
		/// <param name="gridType">Cost space datatype.</param>
		/// <param name="mapType">Depth map datatype.</param>
		public abstract void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType);

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public abstract string Name { get; }

		/// <summary>
		/// Gets the sort key. Defaults to <see cref="Name"/>.
		/// </summary>
		public virtual string SortKey { get { return Name; } }
	}

	/// <summary>
	/// A <see cref="StereoNodeFactory"/> that produces instances of a particular direct sub-type of <see cref="StereoNode"/>.
	/// <para/>The code in DisparityEstimator.cs (for <see cref="DisparityEstimator"/>) automatically enumerates sub-types of this
	/// that are defined in VideoLib.Stereo.GpGpu, so there is no need to reference a custom StereoNode or StereoNodeFactory directly.
	/// </summary>
	/// <typeparam name="T">
	/// The type of node produced (should be one of the direct sub-types of <see cref="StereoNode"/>).
	/// </typeparam>
	abstract class StereoNodeFactory<T> : StereoNodeFactory
	{
		/// <summary>
		/// Creates a <see cref="StereoNode"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>An instance of <see cref="StereoNode"/>.</returns>
		public abstract T Create(ExecutionStream stream);
	}
}
